//
// File:        IORSource.java
// Package:     gov.llnl.babel.backend.ior
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: IORSource.java,v 1.36 2003/08/05 15:31:11 epperly Exp $
// Description: generate IOR implementation source to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.ior;

import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.BabelConfiguration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Class <code>IORSource</code> generates an IOR implementation source file
 * to a language writer output stream.  The constructor takes a language
 * writer stream and method <code>generateCode</code> generates intermediate
 * object representation for the specified symbol to the output stream.  The
 * language writer output stream is not closed by this object.
 */
public class IORSource {
  private static int s_longestBuiltin;
  private static final String s_deleteBuiltin = IOR.getBuiltinName(IOR.DELETE);
  private static final String s_castBuiltin = IOR.getBuiltinName(IOR.CAST);

  /**
   * Store the SymbolID for SIDL.BaseClass, if the extendable being
   * printed is not abstract (i.e., is a concrete class).
   */
  SymbolID d_baseClass = null;

  /**
   * Store the SymbolID for SIDL.ClassInfo, if the extendable being
   * printed is not abstract (i.e., is a concrete class).
   */
  SymbolID d_classInfo = null;

  /**
   * Store the SymbolID for SIDL.ClassInfoI, if the extendable being
   * printed is not abstract (i.e., is a concrete class).
   */
  SymbolID d_classInfoI = null;

  static {
    s_longestBuiltin = 0;
    for(int j = 0; j < IOR.CLASS_BUILTIN_METHODS; ++j) {
      String mname = IOR.getBuiltinName(j);
      if (mname.length() > s_longestBuiltin) {
        s_longestBuiltin = mname.length();
      }
    }
  }

  private LanguageWriterForC d_writer;

  /**
   * This is a convenience utility function that writes the symbol
   * source information into the provided language writer output
   * stream.  The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   */
  public static void generateCode(Symbol symbol, LanguageWriterForC writer)
    throws CodeGenerationException {
    IORSource source = new IORSource(writer);
    source.generateCode(symbol);
  }

  /**
   * Create a <code>IORSource</code> object that will write symbol information
   * to the provided output writer stream.
   */
  public IORSource(LanguageWriterForC writer) {
    d_writer = writer;
  }

  /**
   * Write IOR source information for the provided symbol to the language
   * writer output stream provided in the constructor.  This method does
   * not close the language writer output stream and may be called for more
   * than one symbol (although the generated source may not be valid input
   * for the C compiler).  A code generation exception is generated if an
   * error is detected.  No code is generated for enumerated and package
   * symbols.
   */
  public void generateCode(Symbol symbol) throws CodeGenerationException {
    if (symbol != null) {
      switch (symbol.getSymbolType()) {
      case Symbol.PACKAGE:
        break;
      case Symbol.CLASS:
      case Symbol.INTERFACE:
        generateSource((Extendable) symbol);
        break;
      }
    }
  }

  /**
   * Lookup the SymbolIDs for SIDL.BaseClass, SIDL.ClassInfo and
   * SIDL.ClassInfoI.
   */
  private void lookupSymbolIDs()
  {
    SymbolTable table = SymbolTable.getInstance();
    d_baseClass = table.
      lookupSymbol(BabelConfiguration.getBaseClass()).getSymbolID();
    d_classInfo = table.
      lookupSymbol(BabelConfiguration.getClassInfo()).getSymbolID();
    d_classInfoI = table.
      lookupSymbol(BabelConfiguration.getClassInfoI()).getSymbolID();
  }

  /**
   * Generate the IOR source for a SIDL class or interface.  The source
   * file begins with a banner and include files followed by a declaration
   * of static methods and (for a class) external methods expected in the
   * skeleton file.  For classes, the source file then defines a number
   * of functions (cast, delete, initialize EPVs, new, init, and fini).
   * The source file concludes with remote method support.
   */
  private void generateSource(Extendable ext)
    throws CodeGenerationException {
    /*
     * Generate the file banner and include files.
     */
    SymbolID id = ext.getSymbolID();
    SymbolID baseClass = null;
    SymbolID classInfo = null;
    SymbolID classInfoI = null;
    String source = IOR.getSourceFile(id);
    String header = IOR.getHeaderFile(id);

    d_writer.writeBanner(ext, source, CodeConstants.C_IS_NOT_IMPL,
                         CodeConstants.C_DESC_IOR_PREFIX + id.getFullName());

    d_writer.printlnUnformatted("#include <stdlib.h>");
    d_writer.printlnUnformatted("#include <stddef.h>");
    d_writer.printlnUnformatted("#include <string.h>");
    d_writer.generateInclude(header, false);
    if (!ext.isAbstract()) {
      lookupSymbolIDs();
      d_writer.generateInclude(C.getImplHeaderFile(d_baseClass), true);
      d_writer.generateInclude(C.getHeaderFile(d_baseClass), true);
      d_writer.generateInclude(C.getHeaderFile(d_classInfo), true);
      d_writer.generateInclude(C.getHeaderFile(d_classInfoI), true);
    }
    d_writer.println();

    d_writer.printlnUnformatted("#ifndef NULL");
    d_writer.printlnUnformatted("#define NULL 0");
    d_writer.printlnUnformatted("#endif");
    d_writer.println();

    /*
     * Extract a pointer to the class (if OK) for class-specific methods.
     */
    Class cls = null;
    if (!ext.isInterface()) {
      cls = (Class) ext;
    }

    /*
     * Generate internal static variables and external references to be
     * supplied by the skeleton file.
     */
    generateStaticVariables(ext);
    if (cls != null) {
      generateExternalReferences(cls);
    }

    /*
     * Generate a number of local functions (cast, delete, initializion
     * of EPVs, new, init, and fini).  These functions are only needed
     * for classes.
     */
    if (cls != null) {
      generateCastFunction(cls);
      generateDeleteFunction(cls);
      generateInitEPV(cls);
      generateInitSEPV(cls);
      generateStaticFunction(cls);
      generateInitClassInfo(cls);
      generateInitMetadata(cls);
      generateNewFunction(cls);
      generateInitFunction(cls);
      generateFiniFunction(cls);
      generateVersionFunction(cls);
      generateExternalFunc(ext);
    }


    /*
     * Generate remote method support for interfaces and classes.
     */
    generateRemoteCastFunction(ext);
    generateRemoteDeleteFunction(ext);
    generateRemoteMethodBodies(ext);
    generateRemoteInitEPV(ext);
    generateRemoteConstructor(ext);
  }

  /**
   * Generate a single line comment.  This is called out as a separate
   * method to make the code formatting below a little prettier.
   */
  private void comment(String s) {
    d_writer.writeComment(s, false);
  }

  /**
   * Generate the static variables used to store the EPVs and also the
   * initialization flags for the EPVs.  For interfaces, we only need
   * to generate a remote EPV structure.  Classes require EPVs for their
   * static methods (if present), standard methods, remote methods, and
   * old, new, and remote versions for all parent classes and interfaces.
   */
  private void generateStaticVariables(Extendable ext) {
    comment("Static variables to hold version of IOR");
    d_writer.println("static const int32_t s_IOR_MAJOR_VERSION = " +
                     IOR.MAJOR_VERSION + ";");
    d_writer.println("static const int32_t s_IOR_MINOR_VERSION = " +
                     IOR.MINOR_VERSION + ";");
    if (!ext.isAbstract()) {
      comment("Static variable to hold shared ClassInfo interface.");
      d_writer.println("static " + C.getObjectName(d_classInfo) +
                       " s_classInfo = NULL;");
      d_writer.println("static int s_classInfo_init = 1;");
      d_writer.println();
    }
    comment("Static variables for managing EPV initialization.");

    /*
     * Output the initialization flags for the EPV structures.
     */
    boolean has_static = ext.hasStaticMethod(true);

    if (!ext.isInterface()) {
      d_writer.println("static int s_method_initialized = 0;");
    }
    d_writer.println("static int s_remote_initialized = 0;");
    if (has_static) {
      d_writer.println("static int s_static_initialized = 0;");
    }
    d_writer.println();

    /*
     * Output the EPV, remote EPV, and static EPV for this object.
     */
    SymbolID id    = ext.getSymbolID();
    String   name  = IOR.getSymbolName(id);
    String   lower = name.toLowerCase();
    String   epv   = IOR.getEPVName(id);
    String   sepv  = IOR.getSEPVName(id);

    if (!ext.isInterface()) {
      d_writer.print("static " + epv + " " );
      if (has_static) {
        d_writer.print(" ");
      }
      d_writer.println("s_new__" + lower + ";");
    }

    d_writer.print("static " + epv + " " );
    if (has_static) {
      d_writer.print(" ");
    }
    d_writer.println("s_rem__" + lower + ";");

    if (has_static) {
      d_writer.print("static " + sepv + " " );
      d_writer.println("s_stc__" + lower + ";");
    }
    d_writer.println();

    /*
     * If the object is a class, then collect all the parents in a set
     * and output EPV structures for the parents.
     */
    if (!ext.isInterface()) {
      Class cls            = (Class) ext;
      Set   parents        = Utilities.getAllParents(cls);
      Set   new_interfaces = Utilities.getUniqueInterfaceIDs(cls);

      if (!parents.isEmpty()) {
        List sorted = Utilities.sort(parents);
        for (Iterator i = sorted.iterator(); i.hasNext(); ) {
          SymbolID p_id    = (SymbolID) i.next();
          String   p_name  = IOR.getSymbolName(p_id);
          String   p_lower = p_name.toLowerCase();
          String   p_epv   = "static " + IOR.getEPVName(p_id);
          boolean  is_old  = !new_interfaces.contains(p_id);

          d_writer.print(p_epv + " ");
          if (is_old) {
            d_writer.print(" ");
          }
          d_writer.println("s_new__" + p_lower + ";");

          if (is_old) {
            d_writer.print(p_epv + "* ");
            d_writer.println("s_old__" + p_lower + ";");
          }

          d_writer.print(p_epv + " " );
          if (is_old) {
            d_writer.print(" ");
          }
          d_writer.println("s_rem__" + p_lower + ";");
          d_writer.println();
        }
      }
    }
  }

  /**
   * Generate external references for skeleton routines that define the
   * functions in the EPVs.  A class will have a method EPV.  If there are
   * static functions, then there must also be a static EPV.
   */
  private void generateExternalReferences(Class cls) {
    comment("Declare EPV routines defined in the skeleton file.");

    SymbolID id = cls.getSymbolID();
    d_writer.println("extern void " + IOR.getSetEPVName(id) + "(");
    d_writer.increaseTabLevel();
    d_writer.println(IOR.getEPVName(id) + "* epv);");
    d_writer.decreaseTabLevel();

    if (cls.hasStaticMethod(true)) {
      d_writer.println("extern void " + IOR.getSetSEPVName(id) + "(");
      d_writer.increaseTabLevel();
      d_writer.println(IOR.getSEPVName(id) + "* sepv);");
      d_writer.decreaseTabLevel();
    }

    d_writer.println();
  }

  /**
   * Generate the cast function for a class.  This will return null if
   * the cast is invalid and a pointer to the object otherwise.  The logic
   * generates tests for the current class and then recursively queries the
   * parent classes.
   */
  private void generateCastFunction(Class cls) {
    comment("CAST: dynamic type casting support.");

    /*
     * Define the method signature and begin the method implementation.
     */
    SymbolID id = cls.getSymbolID();

    d_writer.println("static void* ior_" + IOR.getSymbolName(id) + '_' +
                     s_castBuiltin + '(');
    d_writer.increaseTabLevel();
    d_writer.println(IOR.getObjectName(id) + "* self,");
    d_writer.println("const char* name)");
    d_writer.decreaseTabLevel();
    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Output parent pointers to simplify access to parent classes.
     * Recursively generate the casting checks for this class and
     * all parent classes.
     */
    d_writer.println("void* cast = NULL;");
    d_writer.println();

    generateParentSelf(cls, 0, 0);
    d_writer.println();

    generateCastLogic(cls, 0);
    d_writer.println();

    /*
     * Return the cast pointer and close the function body.
     */
    d_writer.println("return cast;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Recursively generate the cast logic for a class and its interfaces.
   * This routine first generates the check against the class type and
   * then against all newly defined interfaces in the class.  Finally,
   * it recurses on the class parent (if it exists).
   */
  private void generateCastLogic(Class cls, int level) {
    String self = "s" + String.valueOf(level);

    /*
     * Check the cast against the class type.
     */
    if (level > 0) {
      d_writer.print("} else ");
    }
    d_writer.println("if (!strcmp(name, \"" + cls.getFullName() + "\")) {");
    d_writer.increaseTabLevel();
    d_writer.println("cast = (void*) " + self + ";");
    d_writer.decreaseTabLevel();

    /*
     * Check the cast against all newly defined interfaces in this class.
     */
    List interfaces = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
    for (Iterator i = interfaces.iterator(); i.hasNext(); ) {
      SymbolID id    = (SymbolID) i.next();
      String   fn    = id.getFullName();
      String   lower = IOR.getSymbolName(id).toLowerCase();;
      d_writer.println("} else if (!strcmp(name, \"" + fn + "\")) {");
      d_writer.increaseTabLevel();
      d_writer.println("cast = (void*) &" + self + "->d_" + lower + ";");
      d_writer.decreaseTabLevel();
    }

    /*
     * If there is a parent class, then recursively call the cast function
     * on the parent class.  Otherwise, close the if test block.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      generateCastLogic(parent, level+1);
    } else {
      d_writer.println("}");
    }
  }

  /**
   * Call the destructor for the object and then deallocate the associated
   * storage using <code>free</code>.
   */
  private void generateDeleteFunction(Class cls) {
    comment("DELETE: call destructor and free object memory.");

    SymbolID id   = cls.getSymbolID();
    String   name = IOR.getSymbolName(id);

    d_writer.println("static void ior_" + name + '_' +
                     s_deleteBuiltin + '(');
    d_writer.increaseTabLevel();
    d_writer.println(IOR.getObjectName(id) + "* self)");
    d_writer.decreaseTabLevel();

    d_writer.println("{");
    d_writer.increaseTabLevel();

    d_writer.println(IOR.getFiniName(id) + "(self);");
    d_writer.println("memset((void*)self, 0, sizeof(" +
                     IOR.getObjectName(id) + "));");
    d_writer.println("free((void*) self);");

    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the function that initializes the method entry point vector
   * for this class.  This class performs three functions.  First, it saves
   * the EPVs from the parent classes and interfaces (assuming there is a
   * parent).  Second, it fills the EPV for this class using information from
   * the parent (if there is a parent, and NULL otherwise).  It then calls
   * the skeleton method initialization function for the EPV.  Third, this
   * function resets interface EPV pointers and parent EPV pointers to use
   * the correct function pointers from the class EPV structure.
   */
  private void generateInitEPV(Class cls) throws CodeGenerationException {
    comment("EPV: create method entry point vector (EPV) structure.");

    /*
     * Decleare the method signature and open the implementation body.
     */
    SymbolID id     = cls.getSymbolID();
    String   name   = IOR.getSymbolName(id);
    String   object = IOR.getObjectName(id);

    d_writer.println("static void " + name + "__init_epv(");
    d_writer.increaseTabLevel();
    d_writer.println(object + "* self)");
    d_writer.decreaseTabLevel();

    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Output pointers to the parent classes to simplify access to
     * their data structures.
     */
    if (cls.getParentClass() != null) {
      generateParentSelf(cls, 0, 0);
      d_writer.println();
    }

    /*
     * Output entry point vectors aliases for each parent class and
     * interface as well as a special one for the current object.
     */
    List parents = Utilities.sort(Utilities.getAllParents(cls));
    aliasEPVs(cls, parents, false);

    /*
     * Save all parent interface and class EPVs in static pointers.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      saveEPVs(parent, 1);
    }

    /*
     * Generate a list of the nonstatic methods in the class and get
     * the width of every method name.
     */
    List methods = (List) cls.getNonstaticMethods(true);
    int  mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin) +
      IOR.getVectorEntry("").length();

    /*
     * Output builtin methods: cast, delete, ctor, and dtor.
     */
    for(int j = 0; j < IOR.CLASS_BUILTIN_METHODS; j++) {
      String mname = IOR.getBuiltinName(j);
      d_writer.print("epv->");
      d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
      if ((IOR.CAST == j) || (IOR.DELETE == j)){
        d_writer.println(" = ior_" + name + '_' + mname + ';');
      }
      else {
        d_writer.println(" = NULL;");
      }
    }

    /*
     * Iterate through all of the nonstatic methods.  Assign them to
     * NULL if the parent class does not have the method and to the parent
     * function pointer if the parent has the method.
     */
    for (Iterator i = methods.iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      String mname  = method.getLongMethodName();
      d_writer.print("epv->");
      d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
      if ((parent != null) && (parent.hasMethodByLongName(mname, true))) {
        d_writer.print(" = ");
        d_writer.print(IOR.getCast(method, object + "*"));
        d_writer.println(" s1->d_epv->" + IOR.getVectorEntry(mname) +
                         ";");
      } else {
        d_writer.println(" = NULL;");
      }
    }
    d_writer.println();

    /*
     * Call the user initialization function to set up the EPV structure.
     */
    d_writer.println(IOR.getSetEPVName(id) + "(epv);");
    d_writer.println();

    /*
     * Iterate through all parent EPVs and set each of the function pointers
     * to use the corresponding function in the parent EPV structure.
     */
    copyEPVs(parents);
    d_writer.println("s_method_initialized = 1;");

    /*
     * Close the function definition.
     */
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the function that initializes the static entry point
   * vector (assuming the class contains static methods).  If the
   * parent class contains static methods, then copy the pointers
   * from the parent class, since static methods are "inherited" from
   * the parent (in a funny kinda way).  Call the skeleton-supplied
   * function to initialize the remaining static methods.
   */
  private void generateInitSEPV(Class cls) {
    if (cls.hasStaticMethod(true)) {
      comment("SEPV: create the static entry point vector (SEPV).");

      /*
       * Define the method signature and begin the method body.
       */
      SymbolID id    = cls.getSymbolID();
      String   name  = IOR.getSymbolName(id);
      String   lower = name.toLowerCase();
      int      width = name.length();

      d_writer.println("static void " + name + "__init_sepv(void)");
      d_writer.println("{");
      d_writer.increaseTabLevel();

      /*
       * Get information about the parent (if it exists and has static
       * methods).
       */
      Class    parent = cls.getParentClass();
      SymbolID pid    = null;
      boolean  includeParent = (parent != null) &&
        parent.hasStaticMethod(true);

      if (includeParent) {
        pid = parent.getSymbolID();
        int w = IOR.getSymbolName(pid).length();
        if (w > width) {
          width = w;
        }
      }
      width += "struct __sepv*".length();

      /*
       * Create pointers to the local static EPV structure and one
       * to the parent SEPV structure (if the parent exists and has
       * static members).
       */
      d_writer.printAligned(IOR.getSEPVName(id) + "*", width);
      d_writer.println(" s = &s_stc__" + lower + ";");
      if (includeParent) {
        d_writer.printAligned(IOR.getSEPVName(pid) + "*", width);
        d_writer.println(" p = " + IOR.getStaticsName(pid) + "();");
      }
      d_writer.println();

      /*
       * Iterate through all of the static methods.  Assign them to NULL
       * if the parent class does not have the method and to the parent
       * function pointer if the parent has the method.
       */
      List methods = (List) cls.getStaticMethods(true);
      int  mwidth  = Utilities.getWidth(methods) +
        IOR.getVectorEntry("").length();


      for (Iterator i = methods.iterator(); i.hasNext(); ) {
        Method method = (Method) i.next();
        String mname  = method.getLongMethodName();
        d_writer.print("s->");
        d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
        if ((parent != null) && (parent.hasMethodByLongName(mname, true))) {
          d_writer.println(" = p->" + IOR.getVectorEntry(mname) + ";");
        } else {
          d_writer.println(" = NULL;");
        }
      }
      d_writer.println();

      /*
       * Initialize the remainder of the static functions by calling
       * the skeleton initialization routine.
       */
      d_writer.println(IOR.getSetSEPVName(id) + "(s);");
      d_writer.println();
      d_writer.println("s_static_initialized = 1;");

      /*
       * Close the initialization guard and end the function body.
       */
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }
  }

  /**
   * Generate a function that will return a pointer to the static entry
   * point vector.  This function is called by clients to get access to
   * static methods in the class.  If this class has no static methods,
   * then do not generate an implementation.
   */
  private void generateStaticFunction(Class cls) {
    if (cls.hasStaticMethod(true)) {
      comment("STATIC: return static EPV structure for static methods.");

      SymbolID id    = cls.getSymbolID();
      String   name  = IOR.getSymbolName(id);
      String   lower = name.toLowerCase();

      d_writer.println(IOR.getSEPVName(id) + "*");
      d_writer.println(IOR.getStaticsName(id) + "(void)");
      d_writer.println("{");
      d_writer.increaseTabLevel();

      d_writer.println("if (!s_static_initialized) {");
      d_writer.increaseTabLevel();
      d_writer.println(name + "__init_sepv();");
      d_writer.decreaseTabLevel();
      d_writer.println("}");

      d_writer.println("return &s_stc__" + lower + ";");

      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }
  }

  private void generateInitClassInfo(Class cls) {
    if (!cls.isAbstract()) {
      if (d_classInfoI.equals(cls.getSymbolID())){
        d_writer.println("static void initMetadata(" +
                         IOR.getObjectName(cls.getSymbolID()) +
                         "*);");
        d_writer.println();
      }
      comment("initClassInfo: create a ClassInfo interface if necessary.");
      d_writer.println("static void");
      d_writer.println("initClassInfo(" + C.getObjectName(d_classInfo) +
                       " *info)");
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("if (s_classInfo_init) {");
      d_writer.increaseTabLevel();
      d_writer.println(C.getObjectName(d_classInfoI) + " impl;");
      d_writer.println("s_classInfo_init = 0;");
      d_writer.println("impl = " +
                       C.getFullMethodName(d_classInfoI, "_create") + "();");
      d_writer.println("s_classInfo = " + 
                       C.getFullMethodName(d_classInfo, "_cast") +
                       "(impl);");
      d_writer.println("if (impl) {");
      d_writer.increaseTabLevel();
      d_writer.println(C.getFullMethodName(d_classInfoI, "setName") +
                       "(impl, \"" + cls.getFullName() + "\");");
      d_writer.println(C.getFullMethodName(d_classInfoI, "setIORVersion") +
                       "(impl, s_IOR_MAJOR_VERSION, s_IOR_MINOR_VERSION);");
      if (d_classInfoI.equals(cls.getSymbolID())) {
        d_writer.println("initMetadata(impl);");
      }
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println("if (s_classInfo) {");
      d_writer.increaseTabLevel();
      d_writer.println("if (*info) {");
      d_writer.increaseTabLevel();
      d_writer.println(C.getFullMethodName(d_classInfo, "deleteRef") + 
                       "(*info);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println("*info = s_classInfo;");
      d_writer.println(C.getFullMethodName(d_classInfo, "addRef") + 
                       "(*info);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }
  }

  private void generateInitMetadata(Class cls) {
    if (!cls.isAbstract()) {
      SymbolID id     = cls.getSymbolID();
      String   object = IOR.getObjectName(id);
      comment("initMetadata: store IOR version & class in SIDL.BaseClass's data");
      d_writer.println("static void");
      d_writer.println("initMetadata(" + object + "* self)");
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("if (self) {");
      d_writer.increaseTabLevel();
      d_writer.println(C.getDataName(d_baseClass) + " *data = " +
                       C.getDataGetName(d_baseClass) + "(" +
                       C.getFullMethodName(d_baseClass, "_cast") +
                       "(self));");
      d_writer.println("if (data) {");
      d_writer.increaseTabLevel();
      d_writer.println("data->d_IOR_major_version = s_IOR_MAJOR_VERSION;");
      d_writer.println("data->d_IOR_minor_version = s_IOR_MINOR_VERSION;");
      d_writer.println("initClassInfo(&(data->d_classinfo));");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }
  }

  /**
   * Allocate the memory for a new instance of the class and then call
   * the constructor (initializer) on the object.  This method is not
   * output for abstract classes.
   */
  private void generateNewFunction(Class cls) {
    if (!cls.isAbstract()) {
      comment("NEW: allocate object and initialize it.");

      SymbolID id     = cls.getSymbolID();
      String   object = IOR.getObjectName(id);

      d_writer.println(object + "*");
      d_writer.println(IOR.getNewName(id) + "(void)");
      d_writer.println("{");
      d_writer.increaseTabLevel();

      d_writer.println(object + "* self =");
      d_writer.increaseTabLevel();
      d_writer.println("(" + object + "*) malloc(");
      d_writer.increaseTabLevel();
      d_writer.println("sizeof(" + object + "));");
      d_writer.decreaseTabLevel();
      d_writer.decreaseTabLevel();

      d_writer.println(IOR.getInitName(id) + "(self);");
      d_writer.println("initMetadata(self);");
      d_writer.println("return self;");

      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }
  }

  /**
   * Generate the initialization function that acts as the constructor
   * for a class object.  The logic is as follows.  First, if there is
   * a parent for this class, call its constructor.  Then, make sure that
   * the EPVs have been initialized.  If there are parents, make sure that
   * all parent EPVs are updated to point to the corrected EPVs.  Then
   * initialize the new interfaces in this object.  Finally, call the
   * user-defined constructor.
   */
  private void generateInitFunction(Class cls) {
    comment("INIT: initialize a new instance of the class object.");

    /*
     * Declare the method signature and open the implementation body.
     */
    SymbolID id = cls.getSymbolID();
    d_writer.println("void " + IOR.getInitName(id) + "(");
    d_writer.increaseTabLevel();
    d_writer.println(IOR.getObjectName(id) + "* self)");
    d_writer.decreaseTabLevel();

    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Output parent pointers to simplify access to parent classes.
     */
    generateParentSelf(cls, 0, 0);
    d_writer.println();

    /*
     * If there is a parent class, then call its constructor.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      String p = IOR.getSymbolName(parent.getSymbolID());
      d_writer.println(IOR.getInitName(parent.getSymbolID()) + "(s1);");
      d_writer.println();
    }

    /*
     * Ensure that the EPV has been initialized.
     */
    d_writer.println("if (!s_method_initialized) {");
    d_writer.increaseTabLevel();
    d_writer.println(IOR.getSymbolName(id) + "__init_epv(s0);");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    /*
     * Modify the EPVs in parent classes and their interfaces.
     */
    fixEPVs(cls, 0, true);

    /*
     * Iterate through the interfaces defined in this class and allocate
     * the symbol identifier and set the self pointer.
     */
    List interfaces = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
    for (Iterator i = interfaces.iterator(); i.hasNext(); ) {
      SymbolID iid = (SymbolID) i.next();
      String n = IOR.getSymbolName(iid).toLowerCase();
      d_writer.println("s0->d_" + n + ".d_object = self;");
      d_writer.println();
    }

    /*
     * Finally, set the data pointer for this class and allocate the
     * symbol identifier.
     */
    d_writer.println("s0->d_data = NULL;");
    d_writer.println();

    /*
     * Call the user constructor now that the object has been constructed.
     */
    d_writer.println(methodCall(cls, "self",
                                IOR.getBuiltinName(IOR.CONSTRUCTOR),
                                ""));

    /*
     * Close the function definition.
     */
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the finalization function that acts as the destructor
   * for a class object.  The logic is as follows.  First, call the
   * user-defined destructor for the object.  Deallocate all identifiers
   * for this object.  If there are parents, reset the EPVs to their
   * previous values and call the parent destructor.
   */
  private void generateFiniFunction(Class cls) {
    comment("FINI: deallocate a class instance (destructor).");

    /*
     * Declare the method signature and open the implementation block.
     */
    SymbolID id = cls.getSymbolID();
    d_writer.println("void " + IOR.getFiniName(id) + "(");
    d_writer.increaseTabLevel();
    d_writer.println(IOR.getObjectName(id) + "* self)");
    d_writer.decreaseTabLevel();

    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Output parent pointers to simplify access to parent classes.
     */
    generateParentSelf(cls, 0, 0);
    d_writer.println();

    /*
     * Call the user-defined destructor for this class.
     */
    d_writer.println(
                     methodCall(cls, "s0", IOR.getBuiltinName(IOR.DESTRUCTOR), ""));

    /*
     * If there is a parent class, then reset all parent pointers and
     * call the parent destructor.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      d_writer.println();
      fixEPVs(parent, 1, false);
      String p = IOR.getSymbolName(parent.getSymbolID());
      d_writer.println(IOR.getFiniName(parent.getSymbolID()) + "(s1);");
    }

    /*
     * Close the function definition.
     */
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  private void generateVersionFunction(Class cls) {
    comment("VERSION: Return the version of the IOR used to generate this IOR.");

    /*
     * Declare the method signature and open the implementation block.
     */
    SymbolID id = cls.getSymbolID();
    d_writer.println("void");
    d_writer.println(IOR.getVersionName(id) +
                     "(int32_t *major, int32_t *minor)");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("*major = s_IOR_MAJOR_VERSION;");
    d_writer.println("*minor = s_IOR_MINOR_VERSION;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }
  /**
   * Generate the remote cast method body (not yet implemented).
   */
  private void generateRemoteCastFunction(Extendable ext) {
    comment("REMOTE CAST: dynamic type casting for remote objects.");

    SymbolID id   = ext.getSymbolID();
    String   name = IOR.getSymbolName(id);

    d_writer.println("static void* remote_" + name + '_' +
                     s_castBuiltin + '(');
    d_writer.increaseTabLevel();
    if (ext.isInterface()) {
      d_writer.println("void* self,");
    } else {
      d_writer.println(IOR.getObjectName(id) + "* self,");
    }
    d_writer.println("const char* name)");
    d_writer.decreaseTabLevel();

    d_writer.println("{");
    d_writer.increaseTabLevel();

    d_writer.println("return NULL;");

    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the remote delete method body (not yet implemented).
   */
  private void generateRemoteDeleteFunction(Extendable ext) {
    comment("REMOTE DELETE: call the remote destructor for the object.");

    SymbolID id   = ext.getSymbolID();
    String   name = IOR.getSymbolName(id);

    d_writer.println("static void remote_" + name + '_' +
                     s_deleteBuiltin + '(');
    d_writer.increaseTabLevel();
    if (ext.isInterface()) {
      d_writer.println("void* self)");
    } else {
      d_writer.println(IOR.getObjectName(id) + "* self)");
    }
    d_writer.decreaseTabLevel();
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("free((void*) self);");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate remote method bodies (not yet implemented).
   */
  private void generateRemoteMethodBodies(Extendable ext)
    throws CodeGenerationException {
    SymbolID id     = ext.getSymbolID();
    String   name   = IOR.getSymbolName(id);
    String   object = IOR.getObjectName(id);

    /*
     * Generate method bodies for all non-static methods.
     */
    Iterator m = ext.getNonstaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      String method_name = method.getLongMethodName();
      Type type = method.getReturnType();
      comment("REMOTE METHOD STUB:" + method_name);

      /*
       * Write the method signature.
       */
      d_writer.println("static " + IOR.getReturnString(type));
      d_writer.println("remote_" + name + "_" + method_name + "(");
      d_writer.increaseTabLevel();

      /*
       * Write the method arguments.
       */
      boolean has_throws = !method.getThrows().isEmpty();
      List args = method.getArgumentList();

      if (ext.isInterface()) {
        d_writer.print("void* self");
      } else {
        d_writer.print(object + "* self");
      }
      if ((args.size() > 0) || has_throws) {
        d_writer.println(",");
      }

      for (Iterator a = args.iterator(); a.hasNext(); ) {
        Argument arg = (Argument) a.next();
        d_writer.print(IOR.getArgumentWithFormal(arg));
        if (a.hasNext() || has_throws) {
          d_writer.println(",");
        }
      }

      if (has_throws) {
        d_writer.print(IOR.getExceptionType());
        d_writer.print("*_ex");
      }
      d_writer.println(")");
      d_writer.decreaseTabLevel();

      /*
       * Write the method body.
       */
      d_writer.println("{");
      d_writer.increaseTabLevel();

      switch(type.getType()) {
      case Type.VOID:
        break;
      case Type.FCOMPLEX:
      case Type.DCOMPLEX:
        d_writer.print(IOR.getReturnString(type));
        d_writer.println(" _result;");
        d_writer.println("_result.real = _result.imaginary = 0;");
        d_writer.println("return _result;");
        break;
      case Type.SYMBOL:
        d_writer.println("return (" + IOR.getReturnString(type) + ") 0;");
        break;
      case Type.OPAQUE:
      case Type.ARRAY:
        d_writer.println("return NULL;");
        break;
      default:
        d_writer.println("return 0;");
        break;
      }

      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }
  }

  /**
   * Generate the function that initializes the method entry point vector
   * for a remote instance.  This method performs two basic functions.
   * First, it initializes the remote EPV with pointers to the remote stubs
   * in the IOR file.  Second, it sets interface and parent EPV function
   * pointers using the pointer values generated in the first step.
   */
  private void generateRemoteInitEPV(Extendable ext)
    throws CodeGenerationException {
    comment("REMOTE EPV: create remote entry point vectors (EPVs).");

    /*
     * Decleare the method signature and open the implementation body.
     */
    SymbolID id    = ext.getSymbolID();
    String   name  = IOR.getSymbolName(id);
    String   lower = name.toLowerCase();

    d_writer.println("static void " + name + "__init_remote_epv(void)");
    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Output entry point vectors aliases for each parent class and
     * interface as well as a special one for the current object.
     */
    List parents = null;
    if (ext.isInterface()) {
      d_writer.print(IOR.getEPVName(id) + "*");
      d_writer.println(" epv = &s_rem__" + lower + ";");
      d_writer.println();
    } else {
      parents = Utilities.sort(Utilities.getAllParents((Class) ext));
      aliasEPVs((Class) ext, parents, true);
    }

    /*
     * Generate a list of the nonstatic methods in the class and get
     * the width of every method name.
     */
    List methods = (List) ext.getNonstaticMethods(true);
    int  mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin)
      + IOR.getVectorEntry("").length();

    final int numBuiltins =
      (ext.isInterface() ? IOR.INTERFACE_BUILTIN_METHODS
       : IOR.CLASS_BUILTIN_METHODS);

    /*
     * Output standard methods: cast, delete, ctor, and dtor.
     */
    for(int j = 0; j < numBuiltins; j++) {
      String mname = IOR.getBuiltinName(j);
      d_writer.print("epv->");
      d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
      if ((IOR.CAST == j) || (IOR.DELETE == j)){
        d_writer.println(" = remote_" + name + '_' + mname + ';');
      }
      else {
        d_writer.println(" = NULL;");
      }
    }

    /*
     * Iterate through all of the nonstatic methods.  Assign them to
     * the remote stub function.
     */
    for (Iterator i = methods.iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      String mname  = method.getLongMethodName();
      d_writer.print("epv->");
      d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
      d_writer.println(" = remote_" + name + "_" + mname + ";");
    }

    /*
     * If this is a class, then iterate through all parent EPVs and set
     * each of the function pointers to use the corresponding function in
     * the class EPV structure.
     */
    if (parents != null) {
      d_writer.println();
      copyEPVs(parents);
    }
    d_writer.println("s_remote_initialized = 1;");

    /*
     * Close the function definition.
     */
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate remote method support (not yet implemented).
   */
  private void generateRemoteConstructor(Extendable ext) {
    comment("REMOTE: generate remote instance given URL string.");

    SymbolID id     = ext.getSymbolID();
    String   name   = IOR.getSymbolName(id);
    String   lower  = name.toLowerCase();
    String   object = IOR.getObjectName(id);

    /*
     * Declare the function prototype.
     */
    d_writer.println(object + "*");
    d_writer.println(IOR.getRemoteName(id) + "(const char *url)");
    d_writer.println("{");
    d_writer.increaseTabLevel();

    /*
     * Allocate the object data.
     */
    d_writer.println(object + "* self =");
    d_writer.increaseTabLevel();
    d_writer.println("(" + object + "*) malloc(");
    d_writer.increaseTabLevel();
    d_writer.println("sizeof(" + object + "));");
    d_writer.decreaseTabLevel();
    d_writer.decreaseTabLevel();
    d_writer.println();

    /*
     * Output parent pointers to simplify access to parent classes.
     */
    if (!ext.isInterface()) {
      generateParentSelf((Class) ext, 0, 0);
      d_writer.println();
    }

    /*
     * Ensure that the EPV has been initialized.
     */
    d_writer.println("if (!s_remote_initialized) {");
    d_writer.increaseTabLevel();
    d_writer.println(name + "__init_remote_epv();");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    /*
     * Recursively modify the parent classes and set up interface pointers.
     */
    if (ext.isInterface()) {
      d_writer.println("self->d_epv    = &s_rem__" + lower + ";");
      d_writer.println("self->d_object = NULL; /* FIXME */");
      d_writer.println();
    } else {
      remoteEPVs((Class) ext, 0);
    }

    /*
     * Return a self pointer and close the function body.
     */
    d_writer.println("return self;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  /**
   * Recursively output self pointers to the SIDL objects for this class
   * and its parents.  The self pointers are of the form sN, where N is
   * an integer represented by the level argument.  If the width is zero,
   * then the width of all parents is generated automatically.
   */
  private void generateParentSelf(Class cls, int level, int width) {
    if (cls != null) {

      /*
       * Calculate the width of this class and all parents for pretty
       * output.  Ooh, very pretty.
       */
      if (width == 0) {
        Class parent = cls;
        while (parent != null) {
          int w = IOR.getObjectName(parent.getSymbolID()).length();
          if (w > width) {
            width = w;
          }
          parent = parent.getParentClass();
        }
      }

      /*
       * Now use the width information to print out symbols.
       */
      SymbolID id = cls.getSymbolID();
      d_writer.printAligned(IOR.getObjectName(id) + "*", width+1);
      d_writer.print(" s" + String.valueOf(level) + " = ");
      if (level == 0) {
        d_writer.println("self;");
      } else {
        d_writer.print("&s" + String.valueOf(level-1) + "->d_");
        d_writer.println(IOR.getSymbolName(id).toLowerCase() + ";");
      }
      generateParentSelf(cls.getParentClass(), level+1, width);
    }
  }

  /*
   * Recursively modify the EPVs in parent classes and set up interface
   * pointers.  Nothing is done if the class argument is null.  The flag
   * is_new determines whether the EPVs are being set to a newly defined
   * EPV or to a previously saved EPV.
   */
  private void fixEPVs(Class cls, int level, boolean is_new) {
    if (cls != null) {
      fixEPVs(cls.getParentClass(), level+1, is_new);

      /*
       * Update the EPVs for all of the new interfaces in this
       * particular class.
       */
      String self  = "s" + String.valueOf(level);
      String news  = is_new ? "new" : "old";
      List   ifce  = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      int    width = Utilities.getWidth(ifce) + "d_.d_epv".length();

      for (Iterator i = ifce.iterator(); i.hasNext(); ) {
        SymbolID id = (SymbolID) i.next();
        String n = IOR.getSymbolName(id).toLowerCase();
        d_writer.print(self + "->");
        d_writer.printAligned("d_" + n + ".d_epv", width);
        d_writer.print(" = ");
        if (is_new) {
          d_writer.print("&");
        }
        d_writer.println("s_" + news + "__" + n + ";");
      }

      /*
       * Modify the class entry point vector.
       */
      String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
      d_writer.print(self + "->");
      d_writer.printAligned("d_epv", width);
      d_writer.print(" = ");
      if (is_new) {
        d_writer.print("&");
      }
      d_writer.println("s_" + news + "__" + lower + ";");
      d_writer.println();
    }
  }

  /*
   * Recursively save the class and interface EPVs in this class and all
   * parent classes.  Nothing is done if the class argument is null.
   */
  private void saveEPVs(Class cls, int level) {
    if (cls != null) {
      saveEPVs(cls.getParentClass(), level+1);

      /*
       * Save the EPVs for all of the new interfaces in this class.
       */
      String self  = "s" + String.valueOf(level);
      List   ifce  = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      int    width = Utilities.getWidth(ifce);

      for (Iterator i = ifce.iterator(); i.hasNext(); ) {
        SymbolID id = (SymbolID) i.next();
        String n = IOR.getSymbolName(id).toLowerCase();
        d_writer.print("s_old__" + n);
        d_writer.printSpaces(width - n.length());
        d_writer.println(" = " + self + "->d_" + n + ".d_epv;");
      }

      /*
       * Save the class entry point vector.
       */
      String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
      d_writer.print("s_old__" + lower);
      d_writer.printSpaces(width - lower.length());
      d_writer.println(" = " + self + "->d_epv;");
      d_writer.println();
    }
  }

  /*
   * Recursively modify the remote EPVs in parent classes and set up
   * the interfaces.
   */
  private void remoteEPVs(Class cls, int level) {
    if (cls != null) {
      remoteEPVs(cls.getParentClass(), level+1);

      /*
       * Update the EPVs for all of the new interfaces in this
       * particular class.
       */
      String self = "s" + String.valueOf(level);
      List   ifce = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      for (Iterator i = ifce.iterator(); i.hasNext(); ) {
        SymbolID id = (SymbolID) i.next();
        String n = IOR.getSymbolName(id).toLowerCase();
        d_writer.println(self + "->d_" + n +
                         ".d_epv    = &s_rem__" + n + ";");

        d_writer.println(self + "->d_" + n +
                         ".d_object = NULL; /* FIXME */");
        d_writer.println();
      }

      /*
       * Modify the class entry point vector.
       */
      String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
      d_writer.println(self + "->d_data = NULL; /* FIXME */");
      d_writer.println(self + "->d_epv  = &s_rem__" + lower + ";");
      d_writer.println();
    }
  }

  /**
   * Generate the entry point vector alias for each parent class and each
   * interface as well as a special one for the current object.
   */
  private void aliasEPVs(Class cls, Collection parents, boolean remote) {
    /*
     * Get the width of the symbols for pretty printing.
     */
    SymbolID id     = cls.getSymbolID();
    String   name   = IOR.getSymbolName(id);
    String   epv    = IOR.getEPVName(id);
    String   prefix = remote ? "&s_rem__" : "&s_new__";

    int width = epv.length() + 1;
    int w     = Utilities.getWidth(parents) + "struct __epv*".length();
    if (w > width) {
      width = w;
    }

    /*
     * Output the EPV pointer for this class and its class and interface
     * parents.
     */
    d_writer.printAligned(epv + "*", width);
    d_writer.println(" epv = " + prefix + name.toLowerCase() + ";");

    int e = 0;
    for (Iterator i = parents.iterator(); i.hasNext(); ) {
      SymbolID sid   = (SymbolID) i.next();
      String   lower = IOR.getSymbolName(sid).toLowerCase();
      d_writer.printAligned(IOR.getEPVName(sid) + "*", width);
      d_writer.printAligned(" e" + String.valueOf(e++), 4);
      d_writer.println(" = " + prefix + lower + ";");
    }
    d_writer.println();
  }

  /*
   * Copy EPV function pointers from the most derived EPV data structure
   * into all parent class and interface EPVs.
   */
  private void copyEPVs(Collection parents) throws CodeGenerationException {
    int e = 0;
    for (Iterator i = parents.iterator(); i.hasNext(); ) {
      /*
       * Extract information about the parent extendable object.
       * Generate a list of the nonstatic methods and calculate
       * the width of every method name.
       */
      SymbolID   id   = (SymbolID) i.next();
      Extendable ext  = (Extendable) Utilities.lookupSymbol(id);
      String     name = IOR.getSymbolName(id);

      List methods = (List) ext.getNonstaticMethods(true);
      int  mwidth  =
        Math.max(Utilities.getWidth(methods), s_longestBuiltin) +
        IOR.getVectorEntry("").length();

      /*
       * Calculate the "self" pointer for the cast function.
       */
      String self = null;
      if (ext.isInterface()) {
        self = "void*";
      } else {
        self = IOR.getObjectName(id) + "*";
      }

      /*
       * Generate the assignments for the cast and delete functions.
       */
      String estring = "e" + String.valueOf(e) + "->";
      String vecEntry = IOR.getVectorEntry(s_castBuiltin);

      d_writer.print(estring);
      d_writer.printAligned(vecEntry, mwidth);
      d_writer.print(" = (void* (*)(" + self);
      d_writer.print(",const char*)) epv->");
      d_writer.print(vecEntry);
      d_writer.println(";");

      vecEntry = IOR.getVectorEntry(s_deleteBuiltin);
      d_writer.print(estring);
      d_writer.printAligned(vecEntry, mwidth);
      d_writer.println(" = (void (*)(" + self + ")) epv->" +
                       vecEntry + ";");

      /*
       * Iterate over all methods in the EPV and set the method pointer.
       */
      for (Iterator j = methods.iterator(); j.hasNext(); ) {
        Method method = (Method) j.next();
        String mname  = method.getLongMethodName();
        d_writer.print(estring);
        d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
        d_writer.print(" = " + IOR.getCast(method, self));
        d_writer.println(" epv->" + IOR.getVectorEntry(mname) + ";");
      }

      d_writer.println();
      e++;
    }
  }

  private static String methodCall(Extendable ext,
                                   String var,
                                   String method,
                                   String args)
  {
    String result = "(*(" + var + "->d_epv->" + IOR.getVectorEntry(method) +
      "))(" + var;
    if (ext.isInterface()) {
      result += "->d_object";
    }
    result += args;
    result += ");";
    return result;
  }

  private void generateReplaceValue(Symbol symbol) {
    d_writer.println("if (result) {");
    d_writer.increaseTabLevel();
    if (symbol instanceof Extendable) {
      d_writer.println("if (value) {");
      d_writer.increaseTabLevel();
      d_writer.println(methodCall((Extendable)symbol, "value",
                                  "addRef", ""));
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println("if (*result) {");
      d_writer.increaseTabLevel();
      d_writer.println(methodCall((Extendable)symbol, "(*result)",
                                  "deleteRef", ""));
      d_writer.decreaseTabLevel();
      d_writer.println("}");
    }
    d_writer.println("*result = value;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  public static void generateExternalSignature(LanguageWriter lw,
                                               Symbol sym,
                                               String terminator)
  {
    final SymbolID id = sym.getSymbolID();
    lw.beginBlockComment(false);
    lw.println("This function returns a pointer to a static structure of");
    lw.println("pointers to function entry points.  Its purpose is to provide");
    lw.println("one-stop shopping for loading DLLs.");
    lw.println("loading DLLs");
    lw.endBlockComment(false);
    lw.println("const " + IOR.getExternalName(id) + "*");
    lw.println(IOR.getExternalFunc(id) + "(void)" + terminator);
  }

  private void generateExternalFunc(Symbol sym)
  {
    SymbolID id = sym.getSymbolID();
    d_writer.println("static const " + IOR.getExternalName(id));
    d_writer.println("s_externalEntryPoints = {");
    d_writer.increaseTabLevel();
    if (sym instanceof Class) {
      Class cls = (Class)sym;
      if (!cls.isAbstract()){
        d_writer.println(IOR.getNewName(id) + ",");
      }
      d_writer.println(IOR.getRemoteName(id) + ",");
      if (cls.hasStaticMethod(true)) {
        d_writer.println(IOR.getStaticsName(id) + ",");
      }
    }
    d_writer.decreaseTabLevel();
    d_writer.println("};");
    d_writer.println();
    generateExternalSignature(d_writer, sym, "");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("return &s_externalEntryPoints;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }
}
