//
// File:        Java.java
// Package:     gov.llnl.babel.backend.jdk
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Java.java,v 1.9 2003/09/29 21:18:17 epperly Exp $
// Description: common Java binding routines shared by Java code generators
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriterForJava;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Class <code>Java</code> contains common Java language binding routines
 * shared by the Java backend code generators.  This class simply collects
 * many common Java binding routines into one place.
 */
public class Java { 

  private static final HashMap s_java_val  = new HashMap();
  private static final HashMap s_java_ref  = new HashMap();
  private static final HashMap s_java_arr  = new HashMap();
  private static final HashMap s_java_sig  = new HashMap();
  private static final HashMap s_jni_arg   = new HashMap();
  private static final HashMap s_init_ior  = new HashMap();
  private static final HashMap s_init_jni  = new HashMap();
  private static final HashMap s_init_java = new HashMap();
  private static final HashMap s_in        = new HashMap();
  private static final HashMap s_inout     = new HashMap();
  private static final HashMap s_post      = new HashMap();
  private static final HashMap s_return    = new HashMap();

  private static final Integer s_void      = new Integer(Type.VOID);
  private static final Integer s_boolean   = new Integer(Type.BOOLEAN);
  private static final Integer s_char      = new Integer(Type.CHAR);
  private static final Integer s_dcomplex  = new Integer(Type.DCOMPLEX);
  private static final Integer s_double    = new Integer(Type.DOUBLE);
  private static final Integer s_fcomplex  = new Integer(Type.FCOMPLEX);
  private static final Integer s_float     = new Integer(Type.FLOAT);
  private static final Integer s_int       = new Integer(Type.INT);
  private static final Integer s_long      = new Integer(Type.LONG);
  private static final Integer s_opaque    = new Integer(Type.OPAQUE);
  private static final Integer s_string    = new Integer(Type.STRING);
  private static final Integer s_enum      = new Integer(Type.ENUM);
  private static final Integer s_class     = new Integer(Type.CLASS);
  private static final Integer s_interface = new Integer(Type.INTERFACE);
  private static final Integer s_array     = new Integer(Type.ARRAY);

  static {
    s_java_val.put(s_void,     "void");
    s_java_val.put(s_boolean,  "boolean");
    s_java_val.put(s_char,     "char");
    s_java_val.put(s_dcomplex, "SIDL.DoubleComplex");
    s_java_val.put(s_double,   "double");
    s_java_val.put(s_fcomplex, "SIDL.FloatComplex");
    s_java_val.put(s_float,    "float");
    s_java_val.put(s_int,      "int");
    s_java_val.put(s_long,     "long");
    s_java_val.put(s_opaque,   "long");
    s_java_val.put(s_string,   "java.lang.String");
    s_java_val.put(s_enum,     "int");

    s_java_ref.put(s_boolean,  "SIDL.Boolean$Holder");
    s_java_ref.put(s_char,     "SIDL.Character$Holder");
    s_java_ref.put(s_dcomplex, "SIDL.DoubleComplex$Holder");
    s_java_ref.put(s_double,   "SIDL.Double$Holder");
    s_java_ref.put(s_fcomplex, "SIDL.FloatComplex$Holder");
    s_java_ref.put(s_float,    "SIDL.Float$Holder");
    s_java_ref.put(s_int,      "SIDL.Integer$Holder");
    s_java_ref.put(s_long,     "SIDL.Long$Holder");
    s_java_ref.put(s_opaque,   "SIDL.Opaque$Holder");
    s_java_ref.put(s_string,   "SIDL.String$Holder");
    s_java_ref.put(s_enum,     "SIDL.Integer$Holder");

    s_java_arr.put(s_boolean,  "SIDL.Boolean$Array#");
    s_java_arr.put(s_char,     "SIDL.Character$Array#");
    s_java_arr.put(s_dcomplex, "SIDL.DoubleComplex$Array#");
    s_java_arr.put(s_double,   "SIDL.Double$Array#");
    s_java_arr.put(s_fcomplex, "SIDL.FloatComplex$Array#");
    s_java_arr.put(s_float,    "SIDL.Float$Array#");
    s_java_arr.put(s_int,      "SIDL.Integer$Array#");
    s_java_arr.put(s_long,     "SIDL.Long$Array#");
    s_java_arr.put(s_opaque,   "SIDL.Opaque$Array#");
    s_java_arr.put(s_string,   "SIDL.String$Array#");

    s_java_sig.put("boolean", "Z");
    s_java_sig.put("char",    "C");
    s_java_sig.put("int",     "I");
    s_java_sig.put("long",    "J");
    s_java_sig.put("float",   "F");
    s_java_sig.put("double",  "D");
    s_java_sig.put("void",    "V");

    s_jni_arg.put("void",             "void");
    s_jni_arg.put("boolean",          "jboolean");
    s_jni_arg.put("char",             "jchar");
    s_jni_arg.put("double",           "jdouble");
    s_jni_arg.put("float",            "jfloat");
    s_jni_arg.put("int",              "jint");
    s_jni_arg.put("long",             "jlong");
    s_jni_arg.put("java.lang.String", "jstring");

    s_init_ior.put(s_boolean,   "#1 #2 = FALSE;");
    s_init_ior.put(s_char,      "#1 #2 = (#1) 0;");
    s_init_ior.put(s_dcomplex,  "#1 #2 = { 0.0, 0.0 };");
    s_init_ior.put(s_double,    "#1 #2 = 0.0;");
    s_init_ior.put(s_fcomplex,  "#1 #2 = { 0.0, 0.0 };");
    s_init_ior.put(s_float,     "#1 #2 = 0.0;");
    s_init_ior.put(s_int,       "#1 #2 = 0;");
    s_init_ior.put(s_long,      "#1 #2 = 0;");
    s_init_ior.put(s_opaque,    "#1 #2 = (#1) NULL;");
    s_init_ior.put(s_string,    "#1 #2 = (#1) NULL;");
    s_init_ior.put(s_enum,      "#1 #2 = (#1) 0;");
    s_init_ior.put(s_class,     "#1 #2 = (#1) NULL;");
    s_init_ior.put(s_interface, "#1 #2 = (#1) NULL;");
    s_init_ior.put(s_array,     "#1 #2 = (#1) NULL;");

    s_init_jni.put("jboolean", "#1 #2 = JNI_FALSE;");
    s_init_jni.put("jchar",    "#1 #2 = 0;");
    s_init_jni.put("jdouble",  "#1 #2 = 0.0;");
    s_init_jni.put("jfloat",   "#1 #2 = 0.0;");
    s_init_jni.put("jint",     "#1 #2 = 0;");
    s_init_jni.put("jlong",    "#1 #2 = 0;");
    s_init_jni.put("jobject",  "#1 #2 = (#1) NULL;");
    s_init_jni.put("jstring",  "#1 #2 = (#1) NULL;");

    s_init_java.put("boolean", "#1 #2 = false");
    s_init_java.put("char",    "#1 #2 = 0");
    s_init_java.put("double",  "#1 #2 = 0.0;");
    s_init_java.put("float",   "#1 #2 = 0.0;");
    s_init_java.put("int",     "#1 #2 = 0;");
    s_init_java.put("long",    "#1 #2 = 0;");
    s_init_java.put("object",  "#1 #2 = (#1) NULL;");
    s_init_java.put("string",  "#1 #2 = (#1) NULL;");

    s_in.put(s_boolean,   "#1 = (#3) #2;");
    s_in.put(s_char,      "#1 = (#3) #2;");
    s_in.put(s_dcomplex,  "#1 = SIDL_Java_J2I_dcomplex(env, #2);");
    s_in.put(s_double,    "#1 = (#3) #2;");
    s_in.put(s_fcomplex,  "#1 = SIDL_Java_J2I_fcomplex(env, #2);");
    s_in.put(s_float,     "#1 = (#3) #2;");
    s_in.put(s_int,       "#1 = (#3) #2;");
    s_in.put(s_long,      "#1 = (#3) #2;");
    s_in.put(s_opaque,    "#1 = (#3) JLONG_TO_POINTER(#2);");
    s_in.put(s_string,    "#1 = SIDL_Java_J2I_string(env, #2);");
    s_in.put(s_enum,      "#1 = (#3) #2;");
    s_in.put(s_class,     "#1 = (#3) SIDL_Java_J2I_cls(env, #2);");
    s_in.put(s_interface, "#1 = (#3) SIDL_Java_J2I_ifc(env, #2, \"#4\");");
    s_in.put(s_array,     "#1 = (#3) SIDL_Java_J2I_borrow_array(env, #2);");

    s_inout.put(s_boolean,   "#1 = SIDL_Java_J2I_boolean_holder(env, #2);");
    s_inout.put(s_char,      "#1 = SIDL_Java_J2I_character_holder(env, #2);");
    s_inout.put(s_dcomplex,  "#1 = SIDL_Java_J2I_dcomplex_holder(env, #2);");
    s_inout.put(s_double,    "#1 = SIDL_Java_J2I_double_holder(env, #2);");
    s_inout.put(s_fcomplex,  "#1 = SIDL_Java_J2I_fcomplex_holder(env, #2);");
    s_inout.put(s_float,     "#1 = SIDL_Java_J2I_float_holder(env, #2);");
    s_inout.put(s_int,       "#1 = SIDL_Java_J2I_int_holder(env, #2);");
    s_inout.put(s_long,      "#1 = SIDL_Java_J2I_long_holder(env, #2);");
    s_inout.put(s_opaque,    "#1 = SIDL_Java_J2I_opaque_holder(env, #2);");
    s_inout.put(s_string,    "#1 = SIDL_Java_J2I_string_holder(env, #2);");
    s_inout.put(s_enum,      "#1 = (#3) SIDL_Java_J2I_int_holder(env, #2);");
    s_inout.put(s_class,
      "#1 = (#3) SIDL_Java_J2I_cls_holder(env, #2, \"#4\");");
    s_inout.put(s_interface,
      "#1 = (#3) SIDL_Java_J2I_ifc_holder(env, #2, \"#4\");");
    s_inout.put(s_array,     "#1 = (#3) SIDL_Java_J2I_take_array(env, #2);");

    s_post.put(s_boolean,   "SIDL_Java_I2J_boolean_holder(env, #2, #1);");
    s_post.put(s_char,      "SIDL_Java_I2J_character_holder(env, #2, #1);");
    s_post.put(s_dcomplex,  "SIDL_Java_I2J_dcomplex_holder(env, #2, &#1);");
    s_post.put(s_double,    "SIDL_Java_I2J_double_holder(env, #2, #1);");
    s_post.put(s_fcomplex,  "SIDL_Java_I2J_fcomplex_holder(env, #2, &#1);");
    s_post.put(s_float,     "SIDL_Java_I2J_float_holder(env, #2, #1);");
    s_post.put(s_int,       "SIDL_Java_I2J_int_holder(env, #2, #1);");
    s_post.put(s_long,      "SIDL_Java_I2J_long_holder(env, #2, #1);");
    s_post.put(s_opaque,    "SIDL_Java_I2J_opaque_holder(env, #2, #1);");
    s_post.put(s_string,    "SIDL_Java_I2J_string_holder(env, #2, #1);");
    s_post.put(s_enum,      "SIDL_Java_I2J_int_holder(env, #2, (int) #1);");
    s_post.put(s_class,     "SIDL_Java_I2J_cls_holder(env, #2, #1, \"#4\");");
    s_post.put(s_interface, "SIDL_Java_I2J_ifc_holder(env, #2, #1, \"#4\");");
    s_post.put(s_array,     "SIDL_Java_I2J_set_array(env, #2, #1);");

    s_return.put(s_boolean,   "#1 = (#3) #2;");
    s_return.put(s_char,      "#1 = (#3) #2;");
    s_return.put(s_dcomplex,  "#1 = SIDL_Java_I2J_dcomplex(env, &#2);");
    s_return.put(s_double,    "#1 = (#3) #2;");
    s_return.put(s_fcomplex,  "#1 = SIDL_Java_I2J_fcomplex(env, &#2);");
    s_return.put(s_float,     "#1 = (#3) #2;");
    s_return.put(s_int,       "#1 = (#3) #2;");
    s_return.put(s_long,      "#1 = (#3) #2;");
    s_return.put(s_opaque,    "#1 = (#3) POINTER_TO_JLONG(#2);");
    s_return.put(s_string,    "#1 = SIDL_Java_I2J_string(env, #2);");
    s_return.put(s_enum,      "#1 = (#3) #2;");
    s_return.put(s_class,     "#1 = SIDL_Java_I2J_cls(env, #2, \"#4\");");
    s_return.put(s_interface, "#1 = SIDL_Java_I2J_ifc(env, #2, \"#4\");");
    s_return.put(s_array,     "#1 = SIDL_Java_I2J_new_array(env, #2, \"#4\");");
  }
   
   

  /**
   * Generate the Java filename for the client using the specified symbol
   * identifier.  Simply append the suffix ".java" to the symbol name.
   */
  public static String getClientJavaFile(SymbolID id) {
    return id.getShortName() + ".java";
  }

  /**
   * Generate the JNI source filename for the client using the specified
   * symbol identifier. Simply append the suffix "_jniStub.c" to the symbol
   * name.
   */
  public static String getClientJNIFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + "_jniStub.c";
  }

  /**
   * Generate the Java filename for the server using the specified symbol
   * identifier.  Simply append the suffix ".java" to the symbol name.
   */
  public static String getJavaImplSourceFile(SymbolID id) {
      return id.getShortName() +"_Impl.java";
  }

  /**
   * Generate the JNI source filename for the server using the specified
   * symbol identifier. Simply append the suffix "_jniSkel.c" to the symbol
   * name.
   */
  public static String getServerJNIFile(SymbolID id) {
    return id.getFullName().replace('.', '_') + "_jniSkel.c";
  }

 /**
   * Return the shortened Java name of a SIDL symbol type.  The Java name is
   * the same as the SIDL name without the namespace information.  This name is
   * the one used when declaring the Java type.  The fully qualified name is
   * returned by <code>getFullJavaSymbolName</code>.
   */
  public static String getJavaSymbolName(SymbolID id) {
    return id.getShortName();
  }

  /**
   * Return the fully qualified Java name that corresponds to a SIDL symbol
   * type.  This name is the same as the SIDL name, including all package
   * information.
   */
  public static String getFullJavaSymbolName(SymbolID id) {
    return id.getFullName();
  }

  /**
   * Return the name of the JNI registration function.  The registration
   * function is the one invoked whenever a new Java class or interface is
   * loaded.
   */
  public static String getRegisterFunction(SymbolID id) {
    return id.getFullName().replace('.', '_') + "__register";
  }

  /**
   * Return the string name of the Java base class that all automatically
   * generated SIDL classes must extend.
   */
  public static String getJavaBaseClass() {
    return "gov.llnl.sidl.BaseClass";
  }

  /**
   * Return the string name of the Java base interface that all automatically
   * generated SIDL interfaces must extend.
   */
  public static String getJavaBaseInterface() {
    return "gov.llnl.sidl.BaseInterface";
  }

  /**
   * Return the name of the inner wrapper class for interfaces.
   */
  public static String getInterfaceWrapper() {
    return "Wrapper";
  }

  /**
   * Return the name of the inner holder class used for inout and out
   * method arguments.
   */
  public static String getHolderName() {
    return "Holder";
  }

    

  /**
   * Return the Java type string corresponding to a SIDL array.  Array types
   * for primitives are represented in a hash table.  All other array types
   * are the name of the symbol with the "Array#" inner class, where the "#"
   * is the array dimension.
   */
  private static String getJavaArrayType(int dim, Type type) {
    String val = (String) s_java_arr.get(new Integer(type.getDetailedType()));
    if (val == null) {
      //val = getFullJavaSymbolName(type.getSymbolID()) + "$Array#";
      val = "java.lang.Object";
    }
    return val.replace('#', Character.forDigit(dim, 10));
  }

  

  /**
   * Return a string for the Java return type corresponding to the specified
   * SIDL type.  This method retains the "$" in the type string for inner
   * classes.  The mapping between SIDL types and Java types is fairly simple
   * for most primitive types and objects.
   */
  private static String getJavaInternalReturnType(Type type) {
    int t = type.getDetailedType();
    String val = (String) s_java_val.get(new Integer(t));
    if (val == null) {
      if ((t == Type.CLASS) || (t == Type.INTERFACE)) {
        val = getFullJavaSymbolName(type.getSymbolID());
      } else if (t == Type.ARRAY) {
        val = getJavaArrayType(type.getArrayDimension(), type.getArrayType());
      }
    }
    return val;
  }

  /**
   * Return a string for the Java return type corresponding to the specified
   * SIDL type.  The return string from this routine is a valid Java type.
   * The mapping between SIDL type and Java type is fairly simple for most
   * primitive types and object types.
   */
  public static String getJavaReturnType(Type type) {
    return getJavaInternalReturnType(type).replace('$', '.');
  }

  /**
   * Convert a type string to a JNI descriptor.  Convert the basic types
   * according to the hash table map.  If the type is not one of the basic
   * types, then convert the symbol type according to JNI conventions.
   */
  private static String getDescriptor(String type) {
    String jni = (String) s_java_sig.get(type);
    if (jni == null) {
      jni = "L" + type.replace('.', '/') + ";";
    }
    return jni;
  }

  /**
   * Return a string for the Java argument corresponding to the specified
   * SIDL argument.  The mapping between SIDL argument and Java argument is
   * fairly simple for most primitive types and object types.  For arguments
   * that are IN, the type is the same as the return type.  For INOUT and OUT
   * arguments, the type is the special holder class.  This routine retains
   * the "$" for inner classes.
   */
  private static String getJavaInternalArgument(Argument arg) {
    String  val  = null;
    Type    type = arg.getType();
    int     t    = type.getDetailedType();
    int     m    = arg.getMode();
    Integer T    = new Integer(t);

    val = (String)((m == Argument.IN) ? s_java_val.get(T) : s_java_ref.get(T));
    if (val == null) {
      if ((t == Type.CLASS) || (t == Type.INTERFACE)) {
        val = getFullJavaSymbolName(arg.getType().getSymbolID());
        if (m != Argument.IN) {
          val = val + "$" + Java.getHolderName();
        }
      } else if (t == Type.ARRAY) {
        val = getJavaArrayType(type.getArrayDimension(), type.getArrayType());
      }
    }

    return val;
  }

  /**
   * Return a string for the Java argument corresponding to the specified
   * SIDL argument.  The mapping between SIDL argument and Java argument is
   * fairly simple for most primitive types and object types.  For arguments
   * that are IN, the type is the same as the return type.  For INOUT and OUT
   * arguments, the type is the special holder class.  This routine removes
   * the "$" for inner classes and replaces it with a ".".
   */
  public static String getJavaArgument(Argument arg) {
    return getJavaInternalArgument(arg).replace('$', '.');
  }

  /**
   * Return a string for the Java argument corresponding to the specified
   * SIDL argument with a formal name.
   */
  public static String getJavaFormalArgument(Argument arg) {
    return getJavaArgument(arg) + " " + arg.getFormalName();
  }

  /**
   * Convert the method argument list and return type into a Java signature
   * string according to JNI conventions.  See any JNI reference for the type
   * mapping.
   */
  public static String getJavaSignature(Method method) {
    StringBuffer buffer = new StringBuffer();
    buffer.append("(");
    for (Iterator a = method.getArgumentList().iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      buffer.append(getDescriptor(getJavaInternalArgument(arg)));
    }
    buffer.append(")");
    Type type = method.getReturnType();
    buffer.append(getDescriptor(getJavaInternalReturnType(type)));
    return buffer.toString();
  }
    
 /**
  * Generate a string that will print a default return value (Java) for 
  * given method
  * @param method the method that needs a default return string
  */
    public static String getDefaultReturnValue ( Method method )
    {
	// Default return for "objects"
	if (method.getReturnType().getType() >= Type.OPAQUE ||
	    method.getReturnType().getType() == Type.FCOMPLEX ||
	    method.getReturnType().getType() == Type.DCOMPLEX) {
	    return "null";
	}
	// Default return for numeric types
	else if (method.getReturnType().getType() > Type.CHAR && 
		 method.getReturnType().getType() < Type.OPAQUE) {
	    return "0";
	}
	// Default return for char
	else if (method.getReturnType().getType() == Type.CHAR ){
	    return "'\\0'";
	}
	// Default return for boolean
	else if (method.getReturnType().getType() == Type.BOOLEAN) {
	    return "false";
	}
	// For void or undefined
	else {
	    return "";
	}
    }
    /**
     * Generate a string that will print a default return value (JNI) for 
     * given method
     * @param method the method that needs a default return string
     */
     public static String getDefaultJNIReturnValue ( Method method )
    {
	// Default return for "objects"
	if (method.getReturnType().getType() >= Type.OPAQUE) {
	    return "NULL";
	}
	// Default return value for fcomplex and dcomplex
	// Structs should be handled differently ??
	else if (method.getReturnType().getType() == Type.FCOMPLEX ||
		 method.getReturnType().getType() == Type.DCOMPLEX) {
	    return "";
	}    
	// Default return for numeric types
	else if (method.getReturnType().getType() > Type.CHAR && 
		 method.getReturnType().getType() < Type.OPAQUE) {
	    return "0";
	}
	// Default return for char
	else if (method.getReturnType().getType() == Type.CHAR ){
	    return "'\\0'";
	}
	// Default return for boolean
	else if (method.getReturnType().getType() == Type.BOOLEAN) {
	    return "0";
	}
	// For void or undefined
	else {
	    return "";
	}
    }
    
    
  /**
   * Return the name of the JNI function corresponding to the specified SIDL
   * method.  These names are the static local names used in the JNI stub file.
   */
  public static String getJNIFunction(Method method) {
    return "jni_" + method.getLongMethodName();
  }

  /**
   * Return a string for the JNI native type corresponding to the specified
   * Java type.  Everything that is not a primitive type like integer or
   * boolean is converted into an JNI object.
   */
  public static String getJNINativeType(String type) {
    String jni = (String) s_jni_arg.get(type);
    if (jni == null) {
      jni = "jobject";
    }
    return jni;
  }

  /**
   * Return a string for the JNI native type corresponding to the specified
   * Java return type.  This method calls <code>getJavaReturnType</code> and
   * converts the resulting strings into its JNI representation.
   */
  public static String getJNIReturnType(Type type) {
    return getJNINativeType(getJavaReturnType(type));
  }

  /**
   * Declare and initialize a variable with an IOR type.  This method converts
   * the SIDL type into an IOR type string and substitutes the type string in
   * the initialization phrase from the initialization hash table.
   */
  public static void declareIORVariable(
      LanguageWriterForC writer, Type type, String variable)
      throws CodeGenerationException {
    writer.println(
      substitute(
        (String) s_init_ior.get(new Integer(type.getDetailedType())),
        IOR.getReturnString(type),
        variable));
  }

  /**
   * Declare and initialize a variable with an IOR type.  This method converts
   * the SIDL type into an IOR type string and substitutes the type string in
   * the initialization phrase from the initialization hash table.
   */
  public static void declareIORVariable(
      LanguageWriterForC writer, Argument arg, String prefix)
      throws CodeGenerationException {
    Type type = arg.getType();
    writer.println(
      substitute(
        (String) s_init_ior.get(new Integer(type.getDetailedType())),
        IOR.getReturnString(type),
        prefix + arg.getFormalName()));
  }

  /**
   * Declare and initialize a variable with a Java type.  This method converts
   * the SIDL type into a Java type string and substitutes the type string in
   * the initialization phrase from the initialization hash table.
   */
  public static void declareJavaVariable(
      LanguageWriterForC writer, Type type, String variable) {
    String rtype = Java.getJNIReturnType(type);
    writer.println(
      substitute(
        (String) s_init_jni.get(rtype),
        rtype,
        variable));
  }

  /**
   * Declare and initialize a variable with a Java type.
   */
  public static void declareJavaVariable(
      LanguageWriterForJava writer, Type type, String variable) {
    String rtype = type.getTypeString();
    writer.println(
      substitute(
        (String) s_init_java.get(rtype),
        rtype,
        variable));
  }

  /**
   * Return a string for the JNI argument corresponding to the specified
   * SIDL argument.  This method converts the SIDL argument into a Java
   * argument and then converts the Java argument into its JNI native
   * representation.  The formal argument name is prepended with an
   * "_arg_" to prevent name collisions with other argument types.
   */
  public static String getJNIFormalArgument(Argument arg) {
    return getJNINativeType(getJavaArgument(arg))
         + " _arg_"
         + arg.getFormalName();
  }

  /**
   * Method <code>preprocessJNIArgument</code> converts between Java arguments
   * and IOR arguments.  Conversion routines are output to the language writer.
   * This routine retrieves the conversion string from the appropriate hash
   * table and then substitutes the appropriate variable names in the conversion
   * string.
   */
  public static void preprocessJNIArgument(
      LanguageWriterForC writer, Argument arg, String prefix)
      throws CodeGenerationException {
    String convert = null;
    Type type = arg.getType();
    Integer detail = new Integer(type.getDetailedType());

    if (arg.getMode() == Argument.IN) {
      convert = (String) s_in.get(detail);
    } else if (arg.getMode() == Argument.INOUT) {
      convert = (String) s_inout.get(detail);
    }

    if (convert != null) {
      writer.println(
        substitute(
          convert,
          prefix + arg.getFormalName(),
          "_arg_" + arg.getFormalName(),
          IOR.getReturnString(type),
          type.isSymbol() ? type.getSymbolID().getFullName() : null));
    }
  }

  /**
   * Method <code>postprocessJNIArgument</code> converts between IOR arguments
   * and Java arguments.  Conversion routines are output to the language writer.
   * This routine retrieves the conversion string from the hash table and then
   * substitutes the appropriate variable names in the conversion string.
   */
  public static void postprocessJNIArgument(
      LanguageWriterForC writer, Argument arg, String prefix)
      throws CodeGenerationException {
    if (arg.getMode() != Argument.IN) {
      Type type = arg.getType();
      String convert = (String) s_post.get(new Integer(type.getDetailedType()));
      if (convert != null) {
        writer.println(
          substitute(
            convert,
            prefix + arg.getFormalName(),
            "_arg_" + arg.getFormalName(),
            IOR.getReturnString(type),
            type.isSymbol() ? type.getSymbolID().getFullName() : null));
      }
    }
    if (arg.getType().getType() == Type.STRING) {
      writer.println("SIDL_String_free(" + prefix + arg.getFormalName() + ");");
    }
  }

  /**
   * Method <code>postprocessJNIReturn</code> converts between IOR return
   * arguments and Java return arguments.  Conversion routines are output
   * to the language writer.  This routine retrieves the conversion string
   * from the hash table and then substitutes the appropriate variable names
   * in the conversion string.
   */
  public static void postprocessJNIReturn(
      LanguageWriterForC writer, Type type,
      String ior_result, String java_result) {
    int t = type.getDetailedType();
    writer.println(
      substitute(
        (String) s_return.get(new Integer(t)),
        java_result,
        ior_result,
        getJNIReturnType(type),
        getJavaInternalReturnType(type)));
    if (t == Type.STRING) {
      writer.println("SIDL_String_free(" + ior_result + ");");
    }
  }

  /**
   * Substitute string tokens of the form #n for two arguments.
   */
  private static String substitute(String s, String s0, String s1) {
    return substitute(s, new String[] { s0, s1 });
  }

  /**
   * Substitute string tokens of the form #n for four arguments.
   */
  private static String substitute(
      String s, String s0, String s1, String s2, String s3) {
    return substitute(s, new String[] { s0, s1, s2, s3 });
  }
    
  /**
   * Substitute certain string tokens in the conversion string with the
   * specified method arguments.  String tokens of the form #n, where n is
   * a nonzero single digit number, are substituted with the corresponding
   * element in the argument array.
   */
  private static String substitute(String s, String[] args) {
    if (s == null) {
	return null;
    }
    StringBuffer sb = new StringBuffer(s);

    int i = 0;
    while (i < sb.length()-1) {
      if (sb.charAt(i) == '#') {
        try {
          int n = Integer.parseInt(String.valueOf(sb.charAt(i+1)))-1;
          if ((n >= 0) && (n < args.length)) {
            sb.replace(i, i+2, args[n]);
          }
        } catch (NumberFormatException ex) {
          // ignore exception - no substitution
        }
      }
      i++;
    }

    return sb.toString();
  }
}
