/* PLEASE READ:
 *****************************************************************************
 * NOTES/ IMPLEMENTATION ISSUES: (Sarah Knoop, 8/14/03)
 * 1.) Currently, the #include statements generated have sufficed for successful 
 *     compilation in all test passed. This may not remain the case. The generation 
 *     code may have to add more #includes for future tests, or derive some way of
 *     searching for include dependencies.
 * 2.) Currently, the printing and throwing of exceptions is done using JNI. Full 
 *     design and implementation of exception handling on this side has not been 
 *     done nor tested. There is just enough here to help with debugging, etc.
 * 3.) A relatively major concern that has seen little attention thus far is the 
 *     management of local JNI references. In the generation code, often many such 
 *     references are created and depend upon the completion of the method to be freed.
 *     I strongly believe that more proper management is needed and more local refernces
 *     should be explicitly deleted with calls to DeleteLocalRef when they have fulfilled 
 *     their usefulness.
 * 4.) Current argument types broken or unabled are: Objects, Enums, Arrays, Classes
 *     Interfaces and Opaques. Code is sufficiently documented and modular so insertion
 *     of such detatils should be relatively easy (hopefully).
 * 5.) Work was started here on Object arguments, but ran into problem with the Stub
 *     implementation. Specifically, there is a nasty little java likning error because
 *     regression/objarg/libJava/SIDL/BaseClass.java extends runtime/java/gov/llnl/sidl/BaseClass.java
 *     which contains native methods that don't seem to have a native implementation anywhere. The
 *     error resides in the static initializer for the later class. The System.loadLibrary(<name>) call 
 *     looks(via the LD_LIBRARY_PATH on solaris and something else on Linux)
 *     to load the library "lib<name>.so" in which the compiled native code lies (which, 
 *     again, I cannot seem to find). Thus, testing could not proceed, though I believe that once
 *     this hurdle is overcome, objarg tests should be close to completion. A little more work
 *     is needed in the area of returning objects, whereas converting object arguments
 *     seems on track. (See JNI, Liang p. 15 -17).
 *
 *****************************************************************************/
//
// File:        ServerJNI.java
// Package:     gov.llnl.babel.backend.jdk
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: ServerJNI.java,v 1.9 2003/08/15 17:48:35 knoop Exp $
// Description: write Java server (skel) JNI code that links Java with the IOR
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.jdk.Java;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;

/**
 * Class <code>ServerJNI</code> writes the JNI C code that links the Java
 * server to the IOR.  The constructor takes a C language writer stream and
 * method <code>generateCode</code> writes the C JNI code for the specified
 * symbol to the output stream.  The language writer output stream is not
 * closed by this object.
 */
public class ServerJNI {
    
    private LanguageWriterForC d_writer;
    private Extendable d_ext;
    private static final String JPREFIX = "_j_";
    private static final String JRETURN = "_j_retval";
    private static final String CRETURN = "_retval";
    private static final String JPTR = "_jptr_";
    private static final String JCLS = "_jCls_";
    private static final String JCTORMID = "_jCtorMID_";
  
    /**
     * Create a <code>ServerJNI</code> object that will write symbol information 
     * to the provided output language writer stream.
     * @param ext   an interface or class symbol that needs source
     *              file for a Java extension class.
     * @param writer the output writer stream
     */
    public ServerJNI(Extendable ext, LanguageWriterForC writer) throws CodeGenerationException{
	if (ext == null) {
	    throw new CodeGenerationException("Unexpected null extendable object");
	}
	d_ext = ext;
	d_writer = writer;
	
	SymbolID id = d_ext.getSymbolID();
	int type = d_ext.getSymbolType();
	String filename = Java.getServerJNIFile(id);
	System.out.println("JNISkel file: " + filename);
	// Do not allow line breaks on the open parenthesis, just space and comma
	d_writer.setLineBreakString(", ");
	d_writer.writeBanner(d_ext, filename, CodeConstants.C_IS_IMPL,
			     CodeConstants.C_DESC_CJNI_PREFIX + id.getFullName()); 
	// Banner does not indicate "server" code
    }

    /**
     * Create a <code>ServerJNI</code> object that will write symbol information 
     * to a created language writer stream.
     * @param ext   an interface or class symbol that needs source
     *              file for a Java extension class.
     */
    public ServerJNI(Extendable ext) throws CodeGenerationException {
	 if (ext == null) {
	     throw new CodeGenerationException("Unexpected null extendable object");
	 }
	 
	 d_ext = ext;

	 SymbolID id = d_ext.getSymbolID();
	 int type = d_ext.getSymbolType();
	 String filename = Java.getServerJNIFile(id);
	 System.out.println("JNISkel file: " + filename);
	 d_writer = new LanguageWriterForC((FileManager.getInstance()).createFile(id,type,"SKELSRCS",filename));
	 // Do not allow line breaks on the open parenthesis, just space and comma
	 d_writer.setLineBreakString(", ");
	 d_writer.writeBanner(d_ext, filename, CodeConstants.C_IS_IMPL,
			      CodeConstants.C_DESC_CJNI_PREFIX + id.getFullName());
	 // Banner does not indicate "server" code
    }


    /**
     * This is a convenience utility function that writes the JNI server
     * information into the provided language writer output stream.  The
     * output stream is not closed on exit.  A code generation exception
     * is thrown if an error is detected, such as I/O trouble or a violation
     * @param symbol  an interface or class symbol that needs source
     *                file for a Java extension class.
     * @param writer the output writer stream* of data type invariants. 
     */
    public static void generateCode(Extendable symbol, LanguageWriterForC writer)
	throws CodeGenerationException 
    {
	ServerJNI jni = new ServerJNI(symbol, writer);
	jni.generateCode();
    }
    
    
    /**
     * This is a convenience utility function that writes the JNI server
     * information into the provided language writer output stream.  The
     * output stream is not closed on exit.  A code generation exception
     * is thrown if an error is detected, such as I/O trouble or a violation
     * of data type invariants.
     * @param sybmol   an interface or class symbol that needs source
     *                  file for a Java extension class.
     */
    public static void generateCode(Extendable symbol)
	throws CodeGenerationException 
    {
	ServerJNI jni = new ServerJNI(symbol);
	jni.generateCode();
    }
    
    /** 
     * Write Java JNI information for the provided symbol to the language 
     * writer output stream provided in the class constructor.
     */
    public synchronized void generateCode() throws CodeGenerationException {
     	System.out.println("******************* Generate code - JNI Skel **************************");
	
	generateIncludes();
	generatePointerJLongConv();
	generateJNISkelData();
	generateInitData();
	
	// generate skels to ctor, dtor and SIDL defined methods here
	generateCtor();
	generateDtor();

        /*
	 * Output the glue code for all methods in the class or interface.  For
	 * an interface, we must define all methods.  For a class, we only define
	 * new methods. (-Still applicable?? - SK)
	 */
	List methods = (List) d_ext.getMethods(d_ext.isInterface());
	for (Iterator m = methods.iterator(); m.hasNext(); ) {
	    Method method = (Method) m.next();
	    if (d_ext.isInterface() || !method.isAbstract()) {
		generateMethod(method);
		d_writer.println();
	    }
	}

	generateSetEPV();
	d_writer.close();
  }


    /***************************************************************************************
     * Private support methods: generateCode() immediate submethods
     ***************************************************************************************/
    
    /**
     * Prints the necessary includes statements: IOR, SIDL_Java, etc.
     * - Not sure what all should be generated here? -SK
     */
    private void generateIncludes() 
    {
	SymbolID id = d_ext.getSymbolID();
	d_writer.generateInclude("SIDL_Java.h", false);
	d_writer.generateInclude("SIDL_Loader.h", false);
	d_writer.generateInclude("SIDL_String.h", false);
	d_writer.generateInclude(IOR.getHeaderFile(id), false);
	d_writer.generateInclude("babel_config.h", false);
	d_writer.println();
    }

    /**
     * Prints out needed pointer <-> jlong conversions (-Still needed??? -SK)
     */
    private void generatePointerJLongConv () 
    {
	d_writer.writeComment("Convert between jlong and void* pointers.", false);
        d_writer.disableLineBreak();
	d_writer.println("#if (SIZEOF_VOID_P == 8)");
	d_writer.println("#define JLONG_TO_POINTER(x) ((void*)(x))");
	d_writer.println("#define POINTER_TO_JLONG(x) ((jlong)(x))");
	d_writer.println("#else");
	d_writer.println("#define JLONG_TO_POINTER(x) ((void*)(int32_t)(x))");
	d_writer.println("#define POINTER_TO_JLONG(x) ((jlong)(int32_t)(x))");
	d_writer.println("#endif");
	d_writer.println();
	d_writer.println("#ifndef NULL");
	d_writer.println("#define NULL 0");
	d_writer.println("#endif");
	d_writer.println();
        d_writer.enableLineBreak();
    }

    /**
     * Generates the static data struct to cache the JNIEnv, the class and method ID's
     * for the Skel. 
     */
    private void generateJNISkelData ()
    {
	d_writer.writeComment("JNISkel data struct to cache JNIEnv, class, and needed MID's", true);
	d_writer.print("static struct ");
	d_writer.println(d_ext.getSymbolID().getFullName().replace('.', '_') + "_jniSkel__data");
	d_writer.println("{");
	d_writer.increaseTabLevel();
	d_writer.println("JNIEnv *env;");
	d_writer.println("jclass thisCls;");
	d_writer.println("jmethodID ctorMID;");
	d_writer.println("jmethodID dtorMID;");

	// generate SIDL defined MIDs
	List methods = (List) d_ext.getMethods(d_ext.isInterface());
	for (Iterator m = methods.iterator(); m.hasNext(); ) {
	    Method method = (Method) m.next();
	    d_writer.println("jmethodID " + method.getShortMethodName() + "MID;");
	}

	d_writer.decreaseTabLevel();
	d_writer.println("} data;");
	d_writer.println();
    }
    /**
     * Generates the init method for the static data struct.
     * Method init's all struct memebers
     */
    private void generateInitData() 
    {
	d_writer.writeComment("Method to initialize struct members", true);
	d_writer.println("static void");
	d_writer.println("init_data(void)");
	d_writer.println("{");
	d_writer.increaseTabLevel();

	// Generate all inits for struct data
	generateInitEnv();
	generateInitCls();
	generateJNIMethodID("ctor", null, false);
	generateJNIMethodID("dtor", "()V", false);

	// SIDL defined methods
	List methods = (List) d_ext.getMethods(d_ext.isInterface());
	for (Iterator m = methods.iterator(); m.hasNext(); ) {
	    Method method = (Method) m.next();
	    String name = method.getLongMethodName();
	    String descriptor = Java.getJavaSignature(method);
	    generateJNIMethodID(name, descriptor, method.isStatic());
	}	
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	d_writer.println();
    } 

    /**
     * Writes JNI function that calls back to the Java constructor of this class
     */
    private void generateCtor() 
    {
	SymbolID id = d_ext.getSymbolID();
	d_writer.writeComment("Constructor", true);
	//Steps:
	// 1. generate header, named approptiately as in set_epv
	d_writer.println("static void");
	d_writer.print(Java.getJNIFunction(IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id)) + "(");
	d_writer.println("struct " + id.getFullName().replace('.', '_')+"__object *self)");
	d_writer.println("{");
	d_writer.increaseTabLevel();

	// 2. Call New Object jni function ("this")
	d_writer.print("jobject this = ");
	String [] args1 = {"data.env", "data.thisCls", "data.ctorMID"};
	printCallToJNIMethod("NewObject", args1);
	d_writer.println(";");
	
	// 3. Exception code
	generateJNIException("");

	// 4. set self_data to NewGlobalReference(this)
	d_writer.print("self->d_data = ");
	String [] args2 = {"data.env", "this"};
	printCallToJNIMethod("NewGlobalRef", args2);
	d_writer.println(";");

	// 5. Delete local ref to this
	printCallToJNIMethod("DeleteLocalRef", args2);
	d_writer.println(";\n");

	// 6. Exception code
	generateJNIException("");

	d_writer.decreaseTabLevel();	       
	d_writer.println("}\n");
    }
    
    /**
     * Writes JNI finction that calls back to the "destructor" method of this class
     */
    private void generateDtor()
    {	
	SymbolID id = d_ext.getSymbolID();
	d_writer.writeComment("Deconstructing method", true);
	//Steps: 
	//1.generate header 
	d_writer.println("static void");
	d_writer.print(Java.getJNIFunction(IOR.getBuiltinMethod(IOR.DESTRUCTOR, id)) + "(");
	d_writer.println("struct " + id.getFullName().replace('.', '_')+"__object *self)");
	d_writer.println("{");
	d_writer.increaseTabLevel();

	//2. Call back to the "dtor" method
	String [] args1 = {"data.env", "(jobject)self->d_data, data.dtorMID"};
	printCallToJNIMethod("CallVoidMethod", args1);
	d_writer.println(";");

	//3. Exception code
	generateJNIException("");

	//4. Delete the Global refernce stored in self_data
	String [] args2 = {"data.env", "(jobject)self->d_data"};
	printCallToJNIMethod("DeleteGlobalRef", args2);
	d_writer.println(";");

	d_writer.decreaseTabLevel();	
	d_writer.println("}\n");
    }


    /**
     * Generate the method glue code that hooks the C IOR to Java Server.  
     * 
     * Output the signature, the glue code to morph IOR arguments to Java arguments,
     * make the call, and then glue code to morph out and return arguments from
     * Java to IOR.
     */
    private void generateMethod( Method method) throws CodeGenerationException {
	System.out.println("Generating method: jni_" + method.getShortMethodName());
	generateMethodHeader(method);
	generateMethodBody(method);
    }

   
    /**
     * Writes the set_epv method needed to set values in epv fundtion table
     */
    private void generateSetEPV() 
    {
	SymbolID id = d_ext.getSymbolID();
	d_writer.println("void");
	d_writer.print(IOR.getSetEPVName(id));
	d_writer.print("(");
	d_writer.print(IOR.getEPVName(id));
	d_writer.println(" *epv) {");
	d_writer.increaseTabLevel();

	d_writer.writeCommentLine("initialize skel data struct");
	d_writer.println("init_data();");
	d_writer.println();
	
	d_writer.writeCommentLine("initialize builtin methods");
	initializeEPVMethodPointer(IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id), id); 
	initializeEPVMethodPointer(IOR.getBuiltinMethod(IOR.DESTRUCTOR, id), id); 
	d_writer.println();
	
	d_writer.writeCommentLine("initialize local methods");
	Iterator i = d_ext.getMethods(false).iterator();
	while (i.hasNext()) {
	    Method m = (Method)i.next();
	    if ( !m.isAbstract() ) { 
		initializeEPVMethodPointer(m, id);
	    }
	}
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	d_writer.println();
    }


    /***************************************************************************************
     * Private support methods: generateInitData() immediate submethods
     ***************************************************************************************/
    private void generateInitEnv () 
    {
	d_writer.println("data.env = SIDL_Java_getEnv();");
	d_writer.println("if (data.env == NULL) {");
	d_writer.increaseTabLevel();
	d_writer.println("return;");
	d_writer.decreaseTabLevel();
	d_writer.println("}");
    }

    private void generateInitCls ()
    {
	d_writer.print("data.thisCls = ");
	String fullFilename = "\"" + d_ext.getSymbolID().getFullName().replace('.', '/') +"_Impl\"";
	String [] args = {"data.env", fullFilename};
	printCallToJNIMethod("FindClass", args);
	d_writer.println(";");
	generateJNIException("");					     
    }

    /**
     * Generates code to initialize a jmethodID for the given method. 
     * Handles special case for jmethodID for constructors when given 
     * "ctor" in the name parameter.
     */
    private void generateJNIMethodID (String name, String descriptor, boolean isStatic)
    {
	d_writer.print("data." + name + "MID = " );
	
	String [] args;
	if (name.equals("ctor")) {
	    String [] args1 = {"data.env", "data.thisCls", "\"<init>\"", "\"()V\""};
	    args = args1;
	    printCallToJNIMethod("GetMethodID", args); 
	}
	else {
	    String [] args1 = {"data.env", "data.thisCls", "\""+name+"\"", "\""+descriptor+"\""};
	    args = args1;
	    if(isStatic) {
		printCallToJNIMethod("GetStaticMethodID", args); 
	    }
	    else {
		printCallToJNIMethod("GetMethodID", args); 
	    }
	}
	d_writer.println(";");
	generateJNIException("");
    }


    /***************************************************************************************
     * Private support methods: generateMethod() all submethods
     ***************************************************************************************/

    /**
     * Writes the jni method header comments and signature for the given method.
     * All jni skel methods are static.
     */
    private void generateMethodHeader(Method m) throws CodeGenerationException
    {
	 d_writer.writeComment(m, false); 
	 d_writer.print("static " );
	 d_writer.println(IOR.getReturnString(m.getReturnType()));
	 d_writer.println(Java.getJNIFunction(m) + "(");
	 d_writer.increaseTabLevel();

	 // Reference to the self pointer
	 d_writer.print("struct " + d_ext.getSymbolID().getFullName().replace('.', '_')+"__object *self");

	 // print remaining arguements
	 List args = m.getArgumentList();
	 if(args.isEmpty()) {
	     d_writer.println(")");
	 }
	 else {
	     d_writer.println(",");
	     for (Iterator a = args.iterator(); a.hasNext(); ) {
		 Argument arg = (Argument) a.next();
		 d_writer.print(IOR.getArgumentWithFormal(arg));
		 d_writer.println(a.hasNext() ? "," : ")");
	     }
	 }
	 d_writer.decreaseTabLevel();
    }

    /**
     * Generates the implementation for the method. Preprocesses any parameters,
     * makes call back and postprocess the return 
     */
    private void generateMethodBody (Method m) throws CodeGenerationException
    {
	d_writer.println("{");
	d_writer.increaseTabLevel();
	
	// implementation
	String [] methodArgs = generateSetUpForCallback(m);
	generateCallback(m, methodArgs);
	// Delete local ref's to parameters???
	generateJNIException(Java.getDefaultJNIReturnValue(m));
	processOutInout(m,methodArgs);
	generateReturn(m);

	d_writer.decreaseTabLevel();
	d_writer.println("}");
   
    }

    /**
     * Sets up for the jni callback. Declares local ref to the object if method is instance. 
     * Converts all params to appropriate j<type> counter parts. Declares local vars for 
     * processing the return value. Return variables from the callbacks are identified as
     * JRETURN and actual jni return values as CRETURN.Returns a string array of the list 
     * of identifiers of the appropriate j<type> needed for the callback
     * 
     * NOTE: strings are trickier, need intermediate value to convert the constant return
     * value returned by the jni call GetStringUTFChars to the variable return value.
     */
    private String[] generateSetUpForCallback (Method m) throws CodeGenerationException
    {
	List args = m.getArgumentList();
	String [] methodArgs = new String [args.size()];

	// Check for instance method, if so declare and init jobject
	if(!m.isStatic()) {
	    d_writer.writeCommentLine("Reference to the object");
	    d_writer.println("jobject this = (jobject)(self->d_data);");
	    d_writer.println();
	}
	
	// Process args
	declareJNIArgsList(args, methodArgs);
	// Declare needed vars for return
	declareJNIReturn(m.getReturnType());
	
	// Initialize all in/inout j<type> variables from argument list
	if(!args.isEmpty()) {
	    d_writer.writeCommentLine("Initialize JNI variables");
	    int i = 0;
	    for (Iterator a = args.iterator(); a.hasNext(); ) {
		Argument arg = (Argument) a.next();
		printInitJVariable(arg,methodArgs[i]);  
		i++;
	    }
	    d_writer.println();
	}
      
	return methodArgs;
    }


    /**
     * Declares corresponding jni types for the argument list needed for the 
     * jni callback. Inserts these names into the provided array.
     * Also, declares any other needed jni types that will be used to 
     * initialize these local correspondants to the argument list.
     */
    private void declareJNIArgsList(List args, String [] methodArgs) 
    {
	if(!args.isEmpty()) {
	    d_writer.writeCommentLine("JNI variables for parameters");
	    int i = 0;
	    for (Iterator a = args.iterator(); a.hasNext(); ) {
		Argument arg = (Argument) a.next();

		// declare appropriate type for in/out/inout
		String jtype; 
		if(arg.getMode() == Argument.IN && 
		   arg.getType().getType() < Type.OPAQUE &&
		   arg.getType().getType() != Type.FCOMPLEX &&
		   arg.getType().getType() != Type.DCOMPLEX) {
		    jtype = Java.getJNIReturnType(arg.getType());		   
		}
		else {
		    jtype = "jobject";
		    //Class and CtorMID needed for reference types
		    d_writer.println("static jclass "+ JCLS + arg.getFormalName()+ " = NULL;");
		    d_writer.println("static jmethodID "+ JCTORMID + arg.getFormalName()+ 
					 " = NULL;");
		    // For objects, need to convert the incommin IOR ptr.
		    if(arg.getType().getType() == Type.SYMBOL) {
			d_writer.println("jlong " + JPTR + arg.getFormalName() + 
					 " = POINTER_TO_JLONG("+arg.getFormalName()+");");
		    }
		}

		String jID = JPREFIX + arg.getFormalName();
		methodArgs[i] = jID;
		i++;
		if (jtype.equals("jobject") || jtype.equals("jstring")) {
		    d_writer.println(jtype + " " + jID + " = NULL;");
		}
		else {
		    d_writer.println(jtype + " " + jID + ";");
		}
	    }
	    d_writer.println();
	}
    }


    /**
     * Declares needed jni and C types for the return of this method
     */
    private void declareJNIReturn(Type t) throws CodeGenerationException
    {
	if(t.getType() != Type.VOID) {
	    d_writer.writeCommentLine("Return variables");
	    String jrtype = Java.getJNIReturnType(t);
	    d_writer.println(jrtype + " "+ JRETURN+";");
	    // for strings
	    if (t.getType() == Type.STRING) {
		d_writer.println("const char* _temp_str;");
	    }
	    String rtype =  IOR.getReturnString(t);
	    d_writer.println(rtype + " " + CRETURN+ ";");
	    d_writer.println();
	}
    }

    /**
     * Prints initialization of local jni (in/inout) variable corresponding 
     * to given arguement.
     */
    private void printInitJVariable(Argument arg, String varname) 
	throws CodeGenerationException
    {
	switch(arg.getType().getType()) {
        case Type.VOID:
	    // do nothing
	    break;
        case Type.OPAQUE:
	case Type.ENUM:
	case Type.CLASS:
        case Type.INTERFACE:
	    break;
        case Type.ARRAY:
	    break;
	case Type.FCOMPLEX:
        case Type.DCOMPLEX:
	    if(arg.getMode() == Argument.IN) {
		printInitInComplex(arg, varname);
	    }
	    else {
		printInitInoutOut(arg, varname);
	    }
	    break;
        case Type.BOOLEAN:
        case Type.CHAR:
        case Type.INT:
        case Type.LONG:
        case Type.FLOAT:
        case Type.DOUBLE:
	    if(arg.getMode() == Argument.IN) {
		// just a cast?
		d_writer.print(varname + " = ");
		d_writer.print("("+ Java.getJNIReturnType(arg.getType())+ ")"); 
		d_writer.println(arg.getFormalName() +";");
	    }
	    else {
		printInitInoutOut(arg, varname);
	    }
	    break;
        case Type.STRING:
	     if(arg.getMode() == Argument.IN) {
		 String [] args = {"data.env", arg.getFormalName()};
		 d_writer.print(varname + " = ");
		 printCallToJNIMethod("NewStringUTF", args);
		 d_writer.println(";");
	     }
	     else {
		printInitInoutOut(arg, varname);
	     }
	    // EXCEPTION CODE???
	    break;
        case Type.SYMBOL:
	    printInitObject(arg,varname);
	    break;
        default:
	    // do nothing
	} 
    }

    /**
     * Prints the jni code needed to utilize the SIDL_Java built in conversion routines
     * for primitive type holders and other reference types for INOUT/OUT params
     */
    private void printInitInoutOut(Argument arg, String varname) 
    {
	// get the Holder class
	String holderType = getSIDLHolderTypeName(arg.getType());
	String callSymbol = getSIDLConversionName(arg.getType());

	// print check for NULL Cls
	d_writer.print("if ("+ JCLS + arg.getFormalName()+" == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	String fullFilename = "\"SIDL/"+holderType+"$Holder\"";
	String [] ctorargs = {"data.env", fullFilename};
	d_writer.print(JCLS + arg.getFormalName()+" = ");
	printCallToJNIMethod("FindClass", ctorargs);
	d_writer.println(";");
	// EXCEPTION CODE???
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	
	// print check for NULL methodID
	d_writer.print("if ("+ JCTORMID + arg.getFormalName()+" == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	d_writer.print(JCTORMID + arg.getFormalName()+" = ");
	String [] midargs = {"data.env", JCLS+arg.getFormalName(), "\"<init>\"", "\"()V\""};
	printCallToJNIMethod("GetMethodID", midargs);
	d_writer.println(";");
	// EXCEPTION CODE???
	d_writer.decreaseTabLevel();
	d_writer.println("}");
  
	// construct object, init
	d_writer.print(varname + " = ");
	String [] initargs = {"data.env", JCLS+arg.getFormalName(),JCTORMID+arg.getFormalName()};
	printCallToJNIMethod("NewObject",initargs);
	d_writer.println(";");
	   
	d_writer.print("SIDL_Java_I2J_" + callSymbol + "_holder");
	if(arg.getType().getType() == Type.FCOMPLEX || arg.getType().getType() == Type.DCOMPLEX) {
	    d_writer.println("(data.env,"+ varname +", "+ arg.getFormalName() +");");
	}
	else {
	    d_writer.println("(data.env,"+ varname +", *"+ arg.getFormalName() +");");
	}
	d_writer.println();
    }
    /**
     * To intitialize IN fcomplex and dcomplex types
     */
    private void printInitInComplex(Argument arg, String varname) 
    {	// get the Holder class
	String holderType = getSIDLHolderTypeName(arg.getType());
	String callSymbol = getSIDLConversionName(arg.getType());
	
	// print check for NULL Cls
	d_writer.print("if ("+ JCLS + arg.getFormalName()+" == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	String fullFilename = "\"SIDL/"+holderType+"\"";
	String [] ctorargs = {"data.env", fullFilename};
	d_writer.print(JCLS + arg.getFormalName()+" = ");
	printCallToJNIMethod("FindClass", ctorargs);
	d_writer.println(";");
	// EXCEPTION CODE???
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	
	// print check for NULL methodID
	d_writer.print("if ("+ JCTORMID + arg.getFormalName()+" == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	d_writer.print(JCTORMID + arg.getFormalName()+" = ");
	String [] midargs = {"data.env", JCLS+arg.getFormalName(), "\"<init>\"", "\"()V\""};
	printCallToJNIMethod("GetMethodID", midargs);
	d_writer.println(";");
	// EXCEPTION CODE???
	d_writer.decreaseTabLevel();
	d_writer.println("}");

	// construct object, init
	d_writer.print(varname + " = ");
	d_writer.print("SIDL_Java_I2J_" + callSymbol);
	d_writer.println("(data.env, &"+ arg.getFormalName() +");");	
    }

    /**
     * Prints code needed to initialize local jobject from the given argument
     */
    private void printInitObject(Argument arg, String varname)
    {
	// print check for NULL Cls
	d_writer.print("if ("+ JCLS + arg.getFormalName()+" == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	
	String fullFilename = ("\"" + arg.getType().getTypeString()).replace('.','/') +"\""; //+"_Impl\"";
	String [] ctorargs = {"data.env", fullFilename};
	d_writer.print(JCLS + arg.getFormalName()+" = ");
	printCallToJNIMethod("FindClass", ctorargs);
	d_writer.println(";");
	// EXCEPTION CODE???
	d_writer.decreaseTabLevel();
	d_writer.println("}");
	
	// print check for NULL methodID
	d_writer.print("if ("+ JCTORMID + arg.getFormalName()+" == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	d_writer.print(JCTORMID + arg.getFormalName()+" = ");
	String [] midargs = {"data.env", JCLS+arg.getFormalName(), "\"<init>\"", "\"(J)V\""};
	printCallToJNIMethod("GetMethodID", midargs);
	d_writer.println(";");
	// EXCEPTION CODE???
	d_writer.decreaseTabLevel();
	d_writer.println("}");

	// construct object, init
	d_writer.print(varname + " = ");
	String [] objargs = {"data.env", JCLS+arg.getFormalName(), 
			     JCTORMID+ arg.getFormalName(),JPTR + arg.getFormalName()};
	printCallToJNIMethod("NewObject", objargs);
	d_writer.println(";");

    }
    

    /**
     * Gets the kind of SIDLHolder needed for this type
     */
    private String getSIDLHolderTypeName(Type t){
	String holderType = null;
	switch(t.getType()) {
	case Type.VOID:
        case Type.OPAQUE:
	case Type.ENUM:
	case Type.CLASS:
        case Type.INTERFACE:
	case Type.SYMBOL:
	    break;
	case Type.FCOMPLEX:
	    holderType = "FloatComplex";	   
	    break;
        case Type.DCOMPLEX:
	    holderType = "DoubleComplex";
	    break;
	case Type.INT:
	    holderType = "Integer";
	    break;
        case Type.CHAR:
	    holderType = "Character";
	    break;
	case Type.BOOLEAN:
	    holderType = "Boolean";
	    break;
	case Type.LONG:
        case Type.FLOAT:
        case Type.DOUBLE:
	case Type.STRING:
	case Type.ARRAY: 
	    StringBuffer sb = new StringBuffer(t.getTypeString());
	    // Capitalize first letter
	    sb.setCharAt(0,(char)(sb.charAt(0) - 32));
	    holderType = sb.toString();
	}
	return holderType;
    }

    /**
     * Gets the conversion type string needed to typify the conversion method to 
     * use between the SIDL Holder/SIDL type and the C data type, and visa versa
     */
    private String getSIDLConversionName(Type t) 
    {
	String callSymbol = null;
	switch(t.getType()) {
	case Type.VOID:
        case Type.OPAQUE:
	case Type.ENUM:
	case Type.CLASS:
        case Type.INTERFACE:
        case Type.ARRAY: 
	    callSymbol = "array";
	    break;
	case Type.CHAR:
	    callSymbol = "character";
	    break;
	case Type.BOOLEAN:
	    callSymbol = "boolean";
	    break;
	case Type.FCOMPLEX:
        case Type.DCOMPLEX:
	case Type.INT:
	case Type.LONG:
        case Type.FLOAT:
        case Type.DOUBLE:
	case Type.STRING:
	case Type.SYMBOL:
	    callSymbol = t.getTypeString();
	}
	return callSymbol;	
    } 
   
    
    /**
     * Writes the callback to the Java source
     */
    private void generateCallback(Method m, String [] methodArgs)
	throws CodeGenerationException
    {
	d_writer.writeCommentLine("Callback to java method");
	// Configure jni callback return type
	Type t = m.getReturnType();
	System.out.println("Return type number is: " + t.getType());
	String jniCallType = getJNICallbackType(t);
	
	if(t.getType() != Type.VOID) {
	    d_writer.print(JRETURN + " = ");
	}
	// configure args for jni callback
	String [] args = new String[(3+methodArgs.length)];
	args[0] = "data.env";
	args[2] = "data." + m.getShortMethodName() + "MID";
	for(int i = 0; i < methodArgs.length; i++) {
	    args[i+3] = methodArgs[i];
	}
	// print callback
	if(m.isStatic()) {
	    args[1] = "data.thisCls";	   
	    printCallToJNIMethod("CallStatic"+jniCallType+"Method",args);
	}
	else {
	    args[1] = "this";
	    printCallToJNIMethod("Call"+jniCallType+"Method",args); 
	}
	d_writer.println(";");
    }


    /** 
     * Fills in the <Type> for the Call<Type>Method jni callback
     */
    private String getJNICallbackType (Type t) 
    {
	String jniCallType = null;
	switch(t.getType()) {
        case Type.VOID:
	    jniCallType = "Void";
	    break;
        case Type.OPAQUE:
	case Type.ENUM:
        case Type.FCOMPLEX:
        case Type.DCOMPLEX:
	case Type.CLASS:
        case Type.INTERFACE:
        case Type.ARRAY:
	case Type.STRING:
	case Type.SYMBOL:
	    jniCallType = "Object";
	    break;
        case Type.BOOLEAN:
	    jniCallType = "Boolean";
	    break;
        case Type.CHAR:
        case Type.INT:
        case Type.LONG:
        case Type.FLOAT:
        case Type.DOUBLE:
	    StringBuffer sb = new StringBuffer(t.getTypeString());
	    // Capitalize first letter
	    sb.setCharAt(0,(char)(sb.charAt(0) - 32));
	    jniCallType = sb.toString();
	    break;
        default:
	}
	return jniCallType;
    }


    /**
     * Updates the values of the out/inout agrs to the method with those 
     * corresponding j<type>s used in the callback.
     */
    private void processOutInout(Method m, String [] methodArgs)
    {
	List args = m.getArgumentList();
	if(!args.isEmpty()) {
	    d_writer.writeCommentLine("Postprocess inout/out args");
	    int i = 0;
	    for (Iterator a = args.iterator(); a.hasNext(); ) {
		Argument arg = (Argument) a.next();
		if((arg.getMode() == Argument.OUT || arg.getMode() == Argument.INOUT) && 
		   arg.getType().getType() != Type.SYMBOL ) {
		    if (arg.getType().getType() != Type.FCOMPLEX || arg.getType().getType() != Type.DCOMPLEX) {
			d_writer.print("*");
		    }
		    d_writer.print(arg.getFormalName() + " = ");
		    
		    d_writer.print("SIDL_Java_J2I_" + getSIDLConversionName(arg.getType())+"_holder");
		    d_writer.println("(data.env,"+ methodArgs[i]+");");
		}
		i++;
	    }
	    d_writer.println();
	}
    }

    /**
     * Writes code to post-process the jni return from the callback. 
     * Delete any local refs and return.
     */
    private void generateReturn(Method m) throws CodeGenerationException
    {
	//STEPS:
	// 1.Convert java return to the jni required return type (Strings are SPECIAL!!)
	switch(m.getReturnType().getType()) {
        case Type.VOID:
	    // do nothing
	    break;
        case Type.OPAQUE:
	case Type.ENUM:     
	case Type.CLASS:
        case Type.INTERFACE:
        case Type.ARRAY:
	    // don't know ??
	    d_writer.println(CRETURN+" = "+JRETURN+";");
	    break;
	case Type.SYMBOL:
	    // do something!?!
	    break;
	case Type.FCOMPLEX:
        case Type.DCOMPLEX:
	     d_writer.print(CRETURN + " = ");
	     d_writer.print("SIDL_Java_J2I_" + 
			    getSIDLConversionName(m.getReturnType()));
	     d_writer.println("(data.env,"+ JRETURN +");");
	    break;
        case Type.BOOLEAN:
        case Type.CHAR:
        case Type.INT:
        case Type.LONG:
        case Type.FLOAT:
        case Type.DOUBLE:
	    // just set equal, with cast??
	    d_writer.println(CRETURN+" = (" + IOR.getReturnString(m.getReturnType())+ ")" + JRETURN + ";");
	    break;
        case Type.STRING:
	    generateNullCheckImmediateReturn(JRETURN,Java.getDefaultJNIReturnValue(m)); 
	    String [] args = {"data.env", JRETURN, "NULL"};
	    d_writer.print("_temp_str = ");
	    printCallToJNIMethod("GetStringUTFChars", args);
	    d_writer.println(";");
	    d_writer.println(CRETURN +" = SIDL_String_strdup(_temp_str);");
	    break;
        default:
	    // do nothing
	} 
	// If reference type, delete local ref to return
	if(m.getReturnType().getType() >= Type.OPAQUE || 
	   m.getReturnType().getType() == Type.FCOMPLEX ||
	   m.getReturnType().getType() == Type.DCOMPLEX) {
	    // 2.Delete local ref to return
	    String [] args2 = {"data.env", JRETURN};
	    printCallToJNIMethod("DeleteLocalRef", args2);
	    d_writer.println(";");
	    generateJNIException(Java.getDefaultJNIReturnValue(m));
	}
	if(m.getReturnType().getType() != Type.VOID)
	    // 3.print return 
	    d_writer.println("return " + CRETURN +";");
    }
    

    /***************************************************************************************
     * Private support methods: general Utility functions, other support methods 
     * ???? should move some to Java.java ???
     ***************************************************************************************/
    /**
     * Generates code that checks if given variable is null and if so, 
     * immediatly return from the method with a default return value
     */
    private void generateNullCheckImmediateReturn(String var, String defaultReturn) 
    {
	d_writer.print("if (" + var  + " == NULL");
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	d_writer.println("return " + defaultReturn + ";");
	d_writer.decreaseTabLevel();
	d_writer.println("}");
    }

    /**
     * Genereates code that throws exception, prints error message
     * and returns from the calling method
     */
    private void generateJNIException (String defaultReturn) 
    {
	d_writer.print("if (");
	String [] args = {"data.env"};
	printCallToJNIMethod("ExceptionOccurred", args);
	d_writer.println(") {");
	d_writer.increaseTabLevel();
	printCallToJNIMethod("ExceptionDescribe", args);
	d_writer.println(";");
	d_writer.println("return " + defaultReturn + ";");
	d_writer.decreaseTabLevel();
	d_writer.println("}");
    }

    /**
     * Prints a call to the given JNI method with the given list of
     * arguements through the JNIEnv pointer. Does not terminate call with 
     * a semicolon, in case call is nested.
     */
    private void printCallToJNIMethod (String name, String [] args)
    {
	d_writer.print("(*data.env)->");
	d_writer.print(name + "(");
	for(int i = 0; i < args.length - 1; i++) {
	    d_writer.print(args[i] + ", ");
	}
	d_writer.print(args[args.length-1] + ")");
    }

    /**
     * For non-<code>static</code> methods, write an assignment statement to
     * initialize a particular membor of the entry point
     * vector. Initialization of <code>static</code> methods appears
     * elsewhere. 
     * 
     * @param m          a description of the method to initialize
     * @param id         the name of class that owns the method
     */
    private void initializeEPVMethodPointer(Method m, SymbolID id)
    {
	final String methodName = m.getLongMethodName();
	switch (m.getDefinitionModifier()) {
	case Method.FINAL:
	case Method.NORMAL:
	    d_writer.print("epv->");
	    d_writer.print(IOR.getVectorEntry(methodName));
	    d_writer.print(" = ");
	    d_writer.print(Java.getJNIFunction(m));
	    d_writer.println(";");
	    break;
	case Method.ABSTRACT:
	    d_writer.print("epv->");
	    d_writer.print(IOR.getVectorEntry(methodName));
	    d_writer.println(" = NULL;");
	    break;
	default:
	    /* do nothing */
	    break;
	}
    }

    /**
     * Return the number of methods that the server skel declares.
     */
    private int countMethods()
    {
	final boolean isInterface = d_ext.isInterface();
	Collection methods = d_ext.getMethods(isInterface);
	int count;
	if (isInterface) {
	    count = methods.size();
	}
	else {
	    count = 0;
	    Iterator i = methods.iterator();
	    while (i.hasNext()) {
        Method m = (Method)i.next();
        if (!m.isAbstract()) {
	    ++count;
        }
	    }
	}
	return count;
    }


    /****************************************************************************************************************** 
     ******************************************************************************************************************
     *                                   UNUSED BELOW HERE - SK
     ******************************************************************************************************************
     ******************************************************************************************************************/
    
    /* OLD IMPLEMENTATION (pre-7/9/03), HAS BEEN MODIFIED (PARTS CUT OUT) MAY NOT FUNCTION AS IS





     private void generateMethod( Method method) throws CodeGenerationException {
 
      boolean throws_exception = !method.getThrows().isEmpty();

      final ArrayList vArgs = method.getArgumentList();
      final int nargs = vArgs.size();
      final boolean throwsExceptions = (method.getThrows().size() > 0);
      
      StringBuffer func_decl = new StringBuffer(nargs*32); // everything through first {
      StringBuffer pre_impl = new StringBuffer(nargs*128); // everything before impl call
      StringBuffer impl_call = new StringBuffer(nargs*32); // the call to the impl method
      StringBuffer impl_suffix = new StringBuffer(16); // might be "._get_ior()" iff return type is object
      StringBuffer post_impl = new StringBuffer(16); // might be "._get_ior()" iff return type is object
      */
    /*
     * Output the method signature and the agrument list.  Static methods do
     * not take an object reference but rather a class reference.  All user
     * arguments are prefixed with "_arg_" and the temporary variables are
     * prefixed with "_tmp_" to avoid name collisions with other arguments.
     */

      /* 
       * Method header
       */
      
      
      //if (method.isStatic()) {
      //func_decl.append("static ");
      //}
	
      //Type return_type = method.getReturnType();
      //func_decl.append(IOR.getReturnString(return_type));
      //func_decl.append("\n");
      
      
      /* 
       * Method body
       */
      /*
	 
	if (return_type.getType() != Type.VOID) {
	pre_impl.append(IOR.getReturnString(return_type));
	pre_impl.append("\n");
	
	if (return_type.getDetailedType() == Type.ENUM) { 
        pre_impl.append(" _result;\n");
	} else if (return_type.getDetailedType() == Type.FCOMPLEX) {
        pre_impl.append(" _result = {0,0};\n");
	} else {
        pre_impl.append(" _result = 0;\n");
	}
      
	switch( return_type.getDetailedType() ) {
        case Type.VOID:
	break;
        case Type.OPAQUE:
	impl_call.append("_result = ");
	break;
        case Type.BOOLEAN:
          pre_impl.append( return_type.getTypeString() + " _local_result;\n" );
          impl_call.append( "_local_result = " );
          post_impl.append( "_result = ( _local_result ? TRUE : FALSE );\n" );
          break;
        case Type.CHAR:
        case Type.INT:
        case Type.LONG:
        case Type.FLOAT:
        case Type.DOUBLE:
          impl_call.append("_result = ");
          break;
        case Type.ENUM:
          //pre_impl.append( Cxx.getEnumName( return_type.getSymbolID()) + " _local_result;\n");
          impl_call.append("_local_result = ");
          post_impl.append("_result = (" + IOR.getEnumName( return_type.getSymbolID()) + ")_local_result;\n");
          break;
        case Type.FCOMPLEX:
          pre_impl.append( return_type.getTypeString() +" _local_result;\n" );
          impl_call.append( "_local_result = " );
          post_impl.append( "_result.real = _local_result.real();\n");
          post_impl.append( "_result.imaginary = _local_result.imag();\n" );
          break;
        case Type.DCOMPLEX:
          pre_impl.append( return_type.getTypeString() + " _local_result;\n" );
          impl_call.append( "_local_result = " );
          post_impl.append( "_result.real = _local_result.real();\n");
          post_impl.append( "_result.imaginary = _local_result.imag();\n" );
          break;
        case Type.STRING:
          pre_impl.append( return_type.getTypeString() + " _local_result;\n" );
          impl_call.append( "_local_result = " );
          post_impl.append( "_result = SIDL_String_strdup( _local_result.c_str() );\n" );
          break;
        case Type.SYMBOL:
          throw new CodeGenerationException( "Type.SYMBOL should have been resolved to\n" + 
                                             "Type.ENUM, Type.CLASS, or Type.INTERFACE");
          //break;
        case Type.CLASS:
        case Type.INTERFACE:
          pre_impl.append( return_type.getTypeString() + " _local_result;\n");
          impl_call.append( "_local_result = " );
          post_impl.append( "if ( _local_result._not_nil() ) {\n");
          post_impl.append( "  _local_result.addRef();\n");
          post_impl.append( "}\n" );
          post_impl.append( "_result = _local_result._get_ior();\n");
          break;
        case Type.ARRAY:
          pre_impl.append( return_type.getTypeString() + " _local_result;\n");
          impl_call.append("_local_result = " );
          post_impl.append("if ( _local_result._not_nil() ) {\n");
          post_impl.append("  _local_result.ensure(" + 
                           return_type.getArrayDimension() + ", " +  ");\n");
                           //s_ensureOrderConstant[return_type.getArrayOrder()] + ");\n");
          post_impl.append("  _local_result.addRef();\n");
          post_impl.append("}\n" );
          post_impl.append("_result = _local_result._get_ior();\n");
          break;
        default:      
      } // end case( return_type.getType() )      
    }
            
    /* End new stuff -=-- BRN */
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	////////////////////////////////////////////////////////////////////////////////////////////////////////////////

  
    /*
     * Begin the method body with the declarations required for the return
     * type, object reference (if not a static method), and the temporary
     * variables used to translate Java constructs to C.
     */
    /*							  
    d_writer.writeComment("Declare return and temporary variables.", false);
    if (!method.isStatic()) {
      d_writer.println(IOR.getObjectName(d_ext.getSymbolID()) + "* _ior = NULL;");
    }
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      Java.declareIORVariable(d_writer, arg, "_tmp_");
    }
    if (return_type.getType() != Type.VOID) {
      Java.declareIORVariable(d_writer, return_type, "_ior_res");
      Java.declareJavaVariable(d_writer, return_type, "_res");
    }
    
    if (throws_exception) {
      d_writer.println("struct SIDL_BaseException__object* _ex = NULL;");
    }
    d_writer.println();
    
    ///
    // Grab the pointer to the IOR object and pre-process all of the method
    // arguments to convert data from Java to C.
    ///
    							  
    d_writer.writeComment("Preprocess Java types and convert into IOR.", false);
    if (!method.isStatic()) {
      d_writer.println("_ior = _get_ior(env, obj);");
    }
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      Java.preprocessJNIArgument(d_writer, arg, "_tmp_");
    }
    d_writer.println();
    
    ///
    // Call the IOR method.  If there is a return type, then store the return
    // value in <code>_ior_result</code>.  We need to lookup the appropriate
    // function pointer in the structure and then pass the appropriate this
    // pointer.  Pass each of the arguments to the output method.
    ///
    
    d_writer.writeComment("Call the IOR method through the EPV.", false);
    if (return_type.getType() != Type.VOID) {
      d_writer.print("_ior_res = ");
    }
    d_writer.print("(*(");
    d_writer.print(method.isStatic() ? "s_sepv" : "_ior->d_epv");
    d_writer.print("->f_" + method.getLongMethodName() + "))(");
    if (method.isStatic() && args.isEmpty() && !throws_exception) {
      d_writer.println(");");
    } else {
      d_writer.println();
      d_writer.increaseTabLevel();
      if (!method.isStatic()) {
        d_writer.print(method.isAbstract() ? "_ior->d_object" : "_ior");
        d_writer.println(args.isEmpty() && !throws_exception ? ");" : ",");
      }
      for (Iterator a = args.iterator(); a.hasNext(); ) {
        Argument arg = (Argument) a.next();
        if (arg.getMode() != Argument.IN) {
          d_writer.print("&");
        }
        d_writer.print("_tmp_" + arg.getFormalName());
        d_writer.println((a.hasNext() || throws_exception) ? "," : ");");
      }
      if (throws_exception) {
        d_writer.println("&_ex);");
      }
      d_writer.decreaseTabLevel();
    }
    
    d_writer.println();
    
    ///
    // Post-process the method arguments if necessary.  This will involve
    // processing the INOUT and OUT arguments and the return argument.
    ///

    							  
    d_writer.writeComment(
      "Postprocess OUT, INOUT, returns, and exceptions.", false);
    for (Iterator a = args.iterator(); a.hasNext(); ) {
      Argument arg = (Argument) a.next();
      Java.postprocessJNIArgument(d_writer, arg, "_tmp_");
    }
    if (return_type.getType() != Type.VOID) {
      Java.postprocessJNIReturn(d_writer, return_type, "_ior_res", "_res");
    }
    if (throws_exception) {
      d_writer.println("SIDL_Java_CheckException(");
      d_writer.increaseTabLevel();
      d_writer.println("env,");
      d_writer.println("_ex,");
      for (Iterator t = method.getThrows().iterator(); t.hasNext(); ) {
        d_writer.println("\"" + ((SymbolID) t.next()).getFullName() + "\",");
      }
      d_writer.println("NULL);");
      d_writer.decreaseTabLevel();
    }
    if (return_type.getType() != Type.VOID) {
      d_writer.println();
      d_writer.println("return _res;");
      }
    */

    /*
     * Done with the method body.  Decrease tab level and close the curly brace.
     */
  
      //}



    /********************************************************************************************
     *
     *  Not sure what the folloing does, should probably just delete. Is not used. -SK
     * 
     ********************************************************************************************/

  /**
   * Write Java JNI information for the provided symbol to the language
   * writer output stream provided in the class constructor.  This method
   * does not close the writer output stream.  Code is currently generated
   * only for SIDL interfaces and classes, since enumerations and packages
   * do not require JNI support.
   */
  private void generateCodeOld(Extendable ext) throws CodeGenerationException {
    if (d_ext == null) {
      throw new CodeGenerationException("Unexpected null extendable object");
    }

    /*
     * Output the file banner.  Include the IOR and SIDL Java include files.
     */
    SymbolID id = d_ext.getSymbolID();
    String file = Java.getServerJNIFile(id);
    d_writer.writeBanner(d_ext, file, CodeConstants.C_IS_NOT_IMPL,
                         CodeConstants.C_DESC_CJNI_PREFIX + id.getFullName());

    d_writer.generateInclude("SIDL_Java.h", false);
    d_writer.generateInclude("SIDL_Loader.h", false);
    d_writer.generateInclude("SIDL_String.h", false);
    d_writer.generateInclude(IOR.getHeaderFile(id), false);
    d_writer.generateInclude("babel_config.h", false);
    d_writer.println();

    /*
     * Output defines that convert between pointers and jlongs.
     */
    d_writer.writeComment("Convert between jlong and void* pointers.", false);
    d_writer.disableLineBreak();
    d_writer.println("#if (SIZEOF_VOID_P == 8)");
    d_writer.println("#define JLONG_TO_POINTER(x) ((void*)(x))");
    d_writer.println("#define POINTER_TO_JLONG(x) ((jlong)(x))");
    d_writer.println("#else");
    d_writer.println("#define JLONG_TO_POINTER(x) ((void*)(int32_t)(x))");
    d_writer.println("#define POINTER_TO_JLONG(x) ((jlong)(int32_t)(x))");
    d_writer.println("#endif");
    d_writer.println();
    d_writer.println("#ifndef NULL");
    d_writer.println("#define NULL 0");
    d_writer.println("#endif");
    d_writer.println();
    d_writer.enableLineBreak();

    /*
     * Output the method to extract the IOR reference from the object.
     * We only need the _get_ior function if the object is an interface
     * or if the object is a class with a non-static, non-abstract method.
     */
    boolean need_ior_function = d_ext.isInterface();

    if (!need_ior_function) {
      Iterator m = d_ext.getMethods(false).iterator();
      while (m.hasNext()) {
        Method method = (Method) m.next();
        if (!method.isAbstract() && !method.isStatic()) {
          need_ior_function = true;
          break;
        }
      }
    }

    if (need_ior_function) {
      d_writer.writeComment(
        "Function to extract IOR reference from the Java object.", false);
      d_writer.println("static " + IOR.getObjectName(id) + "* _get_ior(");
      d_writer.increaseTabLevel();
      d_writer.println("JNIEnv* env,");
      d_writer.println("jobject obj)");
      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("void* ptr = NULL;");
      d_writer.println("static jmethodID mid = (jmethodID) NULL;");
      d_writer.println();
      d_writer.println("if (mid == (jmethodID) NULL) {");
      d_writer.increaseTabLevel();
      d_writer.println("jclass cls = (*env)->GetObjectClass(env, obj);");
      d_writer.println(
        "mid = (*env)->GetMethodID(env, cls, \"_get_ior\", \"()J\");");
      d_writer.println("(*env)->DeleteLocalRef(env, cls);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
      d_writer.println(
        "ptr = JLONG_TO_POINTER((*env)->CallLongMethod(env, obj, mid));");
      d_writer.println("return (" + IOR.getObjectName(id) + "*) ptr;");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * If this is a non-abstract class, then output a static pointer to
     * the IOR external data structure (for the constructor pointer).
     */
    if (!d_ext.isAbstract()) {
      d_writer.writeComment("External reference to IOR methods.", false);
      d_writer.println("static const "
                     + IOR.getExternalName(id)
                     + "* s_external = NULL;");
      d_writer.println();
    }

    /*
     * If this is a class with static methods, then output a pointer to the
     * static EPV structure that will be initialized at class loading.
     */
    if (d_ext.hasStaticMethod(false)) {
      d_writer.writeComment("External reference to static EPV.", false);
      d_writer.println("static const "
                     + IOR.getSEPVName(id)
                     + "* s_sepv = NULL;");
      d_writer.println();
    }

    /*
     * If this is a non-abstract class, then output the function that will
     * create an instance of the object and return the pointer as a jlong
     * reference.
     */
    if (!d_ext.isAbstract()) {
      d_writer.writeComment(
        "Create object instance and return reference.", false);
      d_writer.println("static jlong jni__create_ior(");
      d_writer.increaseTabLevel();
      d_writer.println("JNIEnv* env,");
      d_writer.println("jclass  cls)");
      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();
      d_writer.println("(void) env;");
      d_writer.println("(void) cls;");
      d_writer.println(
        "return POINTER_TO_JLONG((*s_external->createObject)());");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();
    }

    /*
     * Output the glue code for all methods in the class or interface.  For
     * an interface, we must define all methods.  For a class, we only define
     * new methods.
     */
    List methods = (List) d_ext.getMethods(d_ext.isInterface());
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      if (d_ext.isInterface() || !method.isAbstract()) {
        generateMethod(method);
        d_writer.println();
      }
    }

    /*
     * Output the registration method.  This is the only external symbol.
     * First output the method signature and the arguments.
     */
    
    final int nmethods = countMethods() + (d_ext.isAbstract() ? 0 : 1);
    d_writer.writeComment("Register JNI methods with the Java JVM.", false);
    d_writer.println("void " + Java.getRegisterFunction(id) + "(JNIEnv* env)");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    if (nmethods > 0) {
      d_writer.println("JNINativeMethod methods[" +
                       Integer.toString(nmethods) + "];");
      d_writer.println("jclass cls;");
      d_writer.println();

      /*
       * If this is a non-abstract class, then output the code that dynamically
       * loads the external reference.  If there is an error, then throw an
       * unsatisfied link error.
       */
      if (!d_ext.isAbstract()) {
        if (BabelConfiguration.isSIDLBaseClass(id)) {
          d_writer.println("s_external = " + IOR.getExternalFunc(id) + "();");
        } else {
          d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
          d_writer.println("s_external = " + IOR.getExternalFunc(id) + "();");
          d_writer.printlnUnformatted("#else");
          d_writer.println("void* address = SIDL_Loader_lookupSymbol(\""
                           + IOR.getExternalFunc(id)
                           + "\");");
          d_writer.println("if (address != NULL) {");
          d_writer.increaseTabLevel();
          d_writer.println("s_external = (*(("
                           + IOR.getExternalName(id)
                           + "* (*)(void)) address))();");
          d_writer.decreaseTabLevel();
          d_writer.println("} else {");
          d_writer.increaseTabLevel();
          d_writer.println("jclass e = (*env)->FindClass(env, "
                           + "\"java/lang/UnsatisfiedLinkError\");");
          d_writer.println("if (e != NULL) {");
          d_writer.increaseTabLevel();
          d_writer.disableLineBreak();
          d_writer.println("(*env)->ThrowNew(env, e, \""
                           + "Could not find implementation for SIDL class "
                           + id.getFullName()
                           + "\");");
          d_writer.enableLineBreak();
          d_writer.println("(*env)->DeleteLocalRef(env, e);");
          d_writer.decreaseTabLevel();
          d_writer.println("}");
          d_writer.decreaseTabLevel();
          d_writer.println("}");
          d_writer.printlnUnformatted("#endif");
          d_writer.println();
        }
      }
      
      /*
       * If this class has a static method, then initialize the static pointer.
       */
      if (d_ext.hasStaticMethod(false)) {
        d_writer.println("s_sepv = (*(s_external->getStaticEPV))();");
        d_writer.println();
      }

      /*
       * Initialize the method array of names, signatures, and pointers in
       * preparation for registering the methods.
       */
      int idx = 0;
      if (!d_ext.isAbstract()) {
        d_writer.println("methods[0].name      = \"_create_ior\";");
        d_writer.println("methods[0].signature = \"()J\";");
        d_writer.println("methods[0].fnPtr     = (void *)jni__create_ior;");
        idx++;
      }
      
      for (Iterator m = methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        if (d_ext.isInterface() || !method.isAbstract()) {
          String prefix = "methods[" + Integer.toString(idx) + "].";
          d_writer.println(prefix
                           + "name      = \""
                           + method.getShortMethodName()
                           + "\";");
          d_writer.disableLineBreak();
          d_writer.println(prefix
                           + "signature = \""
                           + Java.getJavaSignature(method)
                           + "\";");
          d_writer.enableLineBreak();
          d_writer.println(prefix
                           + "fnPtr     = (void *)"
                           + Java.getJNIFunction(method)
                           + ";");
          ++idx;
        }
      }
      d_writer.println();

      /*
       * Register the methods with the Java virtual machine.
       */
      String lookup_name = Java.getFullJavaSymbolName(id).replace('.', '/');
      if (d_ext.isInterface()) {
        lookup_name = lookup_name + "$Wrapper";
      }
      d_writer.println("cls = (*env)->FindClass(env, \"" + lookup_name + "\");");
      d_writer.println("if (cls) {");
      d_writer.increaseTabLevel();
      d_writer.println("(*env)->RegisterNatives(env, cls, methods, "
                       + String.valueOf(nmethods)
                       + ");");
      d_writer.println("(*env)->DeleteLocalRef(env, cls);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
    }
    else {
      d_writer.writeComment("Intentionally empty: no methods to register",
                            false);
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  } 

}
