//
// File:        SetupGenerator.java
// Package:     gov.llnl.babel.backend.python
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.10 $
// Date:        $Date: 2003/09/17 14:39:51 $
// Description: Generate setup.py for Python extension modules
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;
import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.BuildGenerator;
import gov.llnl.babel.backend.FileListener;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForPython;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;
import java.util.Set;
import java.util.HashMap;
import java.util.Iterator;
import java.util.TreeSet;


/**
 * This class writes a <code>setup.py</code> file to build all the 
 * Python extension modules and implementation code.
 */
public class SetupGenerator implements BuildGenerator, FileListener {

  private Map d_mod2csrc  = new HashMap();
  private Set d_pymodules  = new TreeSet();
  private Set d_headers = new TreeSet();
  private Set d_packages = new TreeSet();

  private void writeList(LanguageWriter lw,
                         String heading,
                         Iterator i)
  {
      lw.println(heading + " = [");
      lw.increaseTabLevel();
      while (i.hasNext()) {
        lw.print("'" + i.next().toString() + "'");
        lw.println(i.hasNext() ? "," : "");
      }
      lw.decreaseTabLevel();
      lw.println("],");
  }

  private void processArgs(LanguageWriter lw)
  {
    lw.println();
    lw.println("inc_re = compile('^--include-dirs=(.*)$')");
    lw.println("lib_re = compile('^--library-dirs=(.*)$')");
    lw.println("old_argv = sys.argv");
    lw.println("sys.argv = []");
    lw.println("inc_dirs = ['.']");
    lw.println("lib_dirs = []");
    lw.println();
    lw.println("for i in old_argv:");
    lw.increaseTabLevel();
    lw.println("m = inc_re.match(i)");
    lw.println("if (m):");
    lw.increaseTabLevel();
    lw.println("if (len(m.group(1))): inc_dirs.append(m.group(1))");
    lw.decreaseTabLevel();
    lw.println("else:");
    lw.increaseTabLevel();
    lw.println("m = lib_re.match(i)");
    lw.println("if (m):");
    lw.increaseTabLevel();
    lw.println("if (len(m.group(1))): lib_dirs.append(m.group(1))");
    lw.decreaseTabLevel();
    lw.println("else:");
    lw.increaseTabLevel();
    lw.println("sys.argv.append(i)");
    lw.decreaseTabLevel();
    lw.decreaseTabLevel();
    lw.decreaseTabLevel();
  }

  /**
   * Generate the setup.py to build the Python extension modules.
   *
   * @exception java.io.IOException this is a exception that contains
   * all the I/O exceptions that occurred during file generation.
   */
  public void createAll()
    throws IOException
  {
    File od = new File(BabelConfiguration.getInstance().getOutputDirectory());
    File setup = new File(od, "setup.py");
    FileWriter fw = null;
    Iterator i;
    od.mkdirs();
    try {
      fw = new FileWriter(setup);
      LanguageWriterForPython lw = new
        LanguageWriterForPython(new PrintWriter(fw));
      lw.printlnUnformatted("#! /usr/bin/env python");
      lw.println("# Build file for Python modules");
      lw.println("import sys");
      lw.println("from re import compile");
      lw.println("from distutils.core import setup, Extension");
      processArgs(lw);
      lw.println("setup(name='babel',");
      lw.increaseTabLevel();
      lw.println("include_dirs=inc_dirs,");
      // writeList(lw, "py_modules", d_pymodules.iterator());
      writeList(lw, "headers", d_headers.iterator());
      writeList(lw, "packages", d_packages.iterator());
      lw.println("ext_modules = [");
      lw.increaseTabLevel();
      for(i = d_mod2csrc.keySet().iterator(); i.hasNext(); ){
        String module = (String)i.next();
        Set files = (Set)d_mod2csrc.get(module);
        lw.println("Extension('" + module + "',");
        lw.increaseTabLevel();
        lw.print("[");
        for(Iterator j = files.iterator(); j.hasNext(); ){
          lw.print("\"" + j.next().toString() + "\"");
          lw.println(j.hasNext() ? "," : "");
        }
        lw.println("],");
        lw.println("library_dirs=lib_dirs,");
        lw.print("libraries=[\"sidl\"])");
        lw.println(i.hasNext() ? "," : "");
        lw.decreaseTabLevel();
      }
      lw.decreaseTabLevel();
      lw.println("])");
      lw.decreaseTabLevel();
    }
    finally {
      if (null != fw) {
        fw.close();
      }
    }
  }

  private Set getCSources(String module)
  {
    Set result = (Set)d_mod2csrc.get(module);
    if (null == result) {
      result = new TreeSet();
      d_mod2csrc.put(module, result);
    }
    return result;
  }

  private static String join(String dir, String file)
  {
    StringBuffer buf = new StringBuffer(dir.length() + file.length());
    final String outputDir = 
      BabelConfiguration.getInstance().getOutputDirectory();
    if (!".".equals(outputDir) && dir.startsWith(outputDir)) {
      // remove the leading output directory
      dir = dir.substring(outputDir.length());
      if (dir.startsWith(File.separator) ||
          dir.startsWith("/")) {
        dir = dir.substring(1);
      }
                         
    }
    if (File.separatorChar != '/') {
      dir = dir.replace(File.separatorChar, '/');
    }
    return buf.append(dir).append(file).toString();
  }

  /**
   * This method is called by the {@link gov.llnl.babel.backend.FileManager}
   * for each new file it creates. This object caches the information it
   * needs to setup.py creation later.
   * 
   *
   * @param id     the file is related to this symbol ID.
   * @param type   indicates the type of the symbol ID (one of the
   *               constants from {@link gov.llnl.babel.symbols.Type}.
   * @param role   this describes the role the file plays. For example,
   *               the file could be a <code>STUBSRCS</code> file or a
   *               <code>IMPLSRCS</code> file. The role strings used
   *               are determined by the backend.
   * @param dir    the path (relative or absolute) of the directory where
   *               the file will be created.
   * @param name   the name of the file not including any directory
   *               information. The complete name of the file should
   *               be <code>dir + name</code>.
   */
  public void newFile(SymbolID id,
                      int      type,
                      String   role,
                      String   dir,
                      String   name)
  {
    final String module = id.getFullName();
    if ("PYMOD_HDRS".equals(role)) {
      d_headers.add(join(dir, name));
    }
    else if ("PYMOD_SRCS".equals(role)) {
      Set files = getCSources(module);
      files.add(join(dir, name));
    }
    else if ("PYTHONSRC".equals(role)) {
      if (Type.ENUM != type) {
        d_pymodules.add(module + "_Impl");
      }
      else {
        d_pymodules.add(module);
      }
    }
    else if ("PYTHONADMIN".equals(role)) {
      d_packages.add(module);
    }
  }

  public Set getLanguages()
  {
    Set result = new TreeSet();
    result.add("python");
    return result;
  }
}
