/* 
 
Copyright 2006 Rene Grothmann, modified by Eric Hakenholz

This file is part of C.a.R. software.

    C.a.R. is a free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, version 3 of the License.

    C.a.R. is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 
 */
 
 
 package rene.zirkel.objects;

// file: CircleObject.java

import java.awt.Frame;
import java.util.Enumeration;

import rene.dialogs.Warning;
import rene.util.xml.XmlWriter;
import rene.zirkel.Zirkel;
import rene.zirkel.ZirkelCanvas;
import rene.zirkel.construction.Construction;
import rene.zirkel.construction.ConstructionException;
import rene.zirkel.expression.Expression;
import rene.zirkel.expression.InvalidException;

public class CircleObject extends PrimitiveCircleObject implements
MoveableObject {
	protected PointObject P2;
	boolean Fixed = false;
	Expression E;
	boolean ExpressionFailed;
	boolean Ellipse = false;

	public CircleObject(final Construction c, final PointObject p1,
			final PointObject p2) {
		super(c, p1);
		P2 = p2;
		validate();
		updateText();
	}

	@Override
	public String getTag() {
		return "Circle";
	}

	@Override
	public void updateText() {
		if (Fixed) {
			if (E == null) {
				setText(text3(Zirkel.name("text.circle.fixed"), M.getName(), P2
						.getName(), "" + round(R)));
			} else {
				setText(text3(Zirkel.name("text.circle.fixed"), M.getName(), P2
						.getName(), "\"" + E.toString() + "\""));
			}
		} else {
			setText(text2(Zirkel.name("text.circle"), M.getName(), P2.getName()));
		}
	}

	@Override
	public void validate() {
		super.validate();
		ExpressionFailed = false;
		if (!M.valid() || !P2.valid()) {
			Valid = false;
			return;
		} else {
			Valid = true;
			X = M.getX();
			Y = M.getY();
			final double X2 = P2.getX(), Y2 = P2.getY();
			// compute normalized vector in the direction of the line:
			final double DX = X2 - X, DY = Y2 - Y;
			R = Math.sqrt(DX * DX + DY * DY);
			if (Fixed) {
				if (!P2.moveableBy(this)) {
					Fixed = false;
				} else {
					try {
						double FixR = E.getValue();
						if (FixR < 0) {
							FixR = 0;
						}
						if (R < 1e-10) {
							P2.move(X + FixR, Y);
						} else {
							P2.move(X + DX * FixR / R, Y + DY * FixR / R);
						}
						R = FixR;
						P2.movedBy(this);
					} catch (final Exception e) {
						R = 0;
						P2.move(X, Y);
						ExpressionFailed = true;
						Valid = false;
					}
				}
			}
		}
	}

	@Override
	public void printArgs(final XmlWriter xml) {
		xml.printArg("through", P2.getName());
		if (Fixed && E != null) {
			xml.printArg("fixed", E.toString());
		}
		super.printArgs(xml);
	}

	@Override
	public boolean canFix() {
		return /* M.moveableBy(this) || */P2.moveableBy(this);
	}

	@Override
	public boolean fixed() {
		return Fixed;
	}

	@Override
	public void setFixed(final boolean flag, final String s)
	throws ConstructionException {
		if (!flag || s.equals("")) {
			Fixed = false;
			E = null;
		} else {
			E = new Expression(s, getConstruction(), this);
			if (!E.isValid()) {
				throw new ConstructionException(E.getErrorText());
			}
			Fixed = true;
		}
		updateText();
	}

	@Override
	public boolean nearto(final int c, final int r, final ZirkelCanvas zc) {
		if (ExpressionFailed && M.valid()) {
			return M.nearto(c, r, zc);
		}
		return super.nearto(c, r, zc);
	}

	@Override
	public void edit(final ZirkelCanvas zc) {
		super.edit(zc);
		if (E != null && !E.isValid()) {
			final Frame F = zc.getFrame();
			final Warning w = new Warning(F, E.getErrorText(), Zirkel
					.name("warning"), true);
			w.center(F);
			w.setVisible(true);
		}
	}

	public boolean isValidFix() {
		return E != null && E.isValid();
	}

	@Override
	public void updateCircleDep() {
		addDep(P2);
	}

	@Override
	public void translate() {
		super.translate();
		P2 = (PointObject) P2.getTranslation();
		try {
			setFixed(Fixed, E.toString());
			E.translate();
		} catch (final Exception e) {
			Fixed = false;
		}
	}

	@Override
	public String getStringLength() {
		if (E != null) {
			return E.toString();
		} else {
			return "" + roundDisplay(R);
		}
	}

	@Override
	public double getValue() throws ConstructionException {
		if (!Valid) {
			throw new InvalidException("exception.invalid");
		} else {
			return R;
		}
	}

	@Override
	public Enumeration depending() {
		super.depending();
		if (!Fixed || E == null) {
			return depset(P2);
		} else {
			depset(P2);
			final Enumeration e = E.getDepList().elements();
			while (e.hasMoreElements()) {
				DL.add((ConstructionObject) e.nextElement());
			}
			return DL.elements();
		}
	}

	public void setP1P2(final PointObject p1, final PointObject p2) {
		M = p1;
		P2 = p2;
	}

	public PointObject getP2() {
		return P2;
	}

	@Override
	public boolean contains(final PointObject p) {
		return p == P2;
	}

	public void dragTo(final double x, final double y) {

		M.move(x1 + (x - x3), y1 + (y - y3));
		P2.move(x2 + (x - x3), y2 + (y - y3));
	}

	@Override
	public void move(final double x, final double y) {
	}

	public boolean moveable() {
		if (!Fixed && M.moveable() && P2.moveable()) {
			return true;
		}
		return false;
	}

	double x1, y1, x2, y2, x3, y3;

	public void startDrag(final double x, final double y) {
		x1 = M.getX();
		y1 = M.getY();
		x2 = P2.getX();
		y2 = P2.getY();
		x3 = x;
		y3 = y;
	}

	public double getOldX() {
		return 0;
	}

	public double getOldY() {
		return 0;
	}

	@Override
	public void snap(final ZirkelCanvas zc) {
		if (moveable()) {
			M.snap(zc);
			P2.snap(zc);
		}
	}
}
