/*
 * $Header: /home/cvspublic/jakarta-commons/httpclient/src/java/org/apache/commons/httpclient/HttpMultiClient.java,v 1.15 2002/09/02 14:52:48 jsdever Exp $
 * $Revision: 1.15 $
 * $Date: 2002/09/02 14:52:48 $
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "HttpClient", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.net.ssl.SSLSocketFactory;

/**
 *
 * An Http user-agent that supports multiple connections
 * to Http servers.
 *
 * @author Marc A. Saegesser
 * @author Sean C. Sullivan
 *
 * @see HttpClient
 *
 */
public class HttpMultiClient {

    // -------------------------------------------------------- Class Variables

    /** Log object for this class. */
    private static final Log log = LogFactory.getLog(HttpMultiClient.class);

    // ----------------------------------------------------- Instance Variables
    /** cookies and other shared state */
    private HttpSharedState state = null;
    /** manager of http connections on a host:port basis */
    private HttpConnectionManager mgr = new HttpConnectionManager();
    /** specifies strict HTTP compliance */
    private boolean strictMode = false; //experimental features off
    /** how long to wait for a connection to become available */
    private int timeoutConnection = 0;
    /** how long to wait for a request to complete */
    private int timeoutRequest = 0;
    /** factory for ssl connections */
    private SSLSocketFactory sslSocketFactory = null;

    // ----------------------------------------------------------- Constructors

    /**
     * No-args constructor.
     *
     * HttpMultiClient objects will have strictMode enabled by default.
     *
     */
    public HttpMultiClient() {
    }
    
    /**
     * Constructor that uses proxy host and port. Strict mode is enabled by 
     * default.
     *
     * @param proxyHost host name of the proxy server to use
     * @param proxyPort port number of the proxy server to use
     */
    public HttpMultiClient(String proxyHost, int proxyPort) {
        this();
        mgr.setProxyHost(proxyHost);
        mgr.setProxyPort(proxyPort);
    }

    /**
     * Set the shared state.
     *
     * @param state the new shared state
     *
     * @see #getState()
     *
     */
    public void setState(HttpSharedState state) {
        this.state = state;
    }

    /**
     * Get the shared state
     *
     * @return the shared state
     *
     * @see #setState(HttpSharedState)
     *
     */
    public HttpSharedState getState() {
        if (state == null) {
            state = new HttpSharedState();
        }

        return state;
    }

    /**
     *
     * @param strictMode <code>true</code> if strict mode should be used
     *
     * @see #isStrictMode()
     *
     */
    public void setStrictMode(boolean strictMode) {
        this.strictMode = strictMode;
    }

    /**
     *
     * @return <code>true</code> if strict mode being used
     *
     * @see #setStrictMode(boolean)
     *
     */
    public boolean isStrictMode() {
        return strictMode;
    }

    /**
     * Sets the connection timeout. The connection timeout is how long (in
     * milliseconds) HttpMultiClient will block in executeMethod waiting for 
     * a connection to the requested server to become available.
     * <br />
     * Setting the connection timeout to 0 will cause executeMethod() to
     * block indefinitely (this is the default behaviour).
     *
     * @param timeout the time, in milliseconds, to block waiting for an 
     *      available connection
     * @see #getConnectionTimeout
     * @see #executeMethod
     */
    public void setConnectionTimeout(int timeout) {
        this.timeoutConnection = timeout;
    }

    /**
     * Returns the value of connection timeout.
     *
     * @return the connection timeout value
     * @see #setConnectionTimeout
     */
    public int getConnectionTimeout() {
        return timeoutConnection;
    }

    /**
     * Sets the request timeout. The executeMethod method calls 
     * HttpConnection.setSoTimeout() with this value before executing the 
     * request.
     *
     * @param timeout the SoTimeout value, in milliseconds
     * @see #getRequestTimeout
     * @see HttpConnection#setSoTimeout
     */
    public void setRequestTimeout(int timeout) {
        this.timeoutRequest = timeout;
    }

    /**
     * Returns the value of the request timeout.
     *
     * @return the request timeout
     * @see setRequestTimeout
     */
    public int getRequestTimeout() {
        return timeoutRequest;
    }

    /**
     * Sets the hostname for the HTTP proxy server to use for all
     * requests.
     *
     * @param host The hostname of the HTTP proxy server
     * @see HttpConnectionManager#setProxyHost
     */
    public void setProxyHost(String host) {
        mgr.setProxyHost(host);
    }

    /**
     * Returns the hostname for the HTTP proxy server in use for
     * all requests.
     *
     * @return the hostname of the HTTP proxy server
     * @see HttpConnectionManager#getProxyHost
     */
    public String getProxyHost() {
        return mgr.getProxyHost();
    }

    /**
     * Sets the port number for the HTTP proxy server to use
     * for all requests.
     *
     * @param port the proxy server port number
     * @see HttpConnectionManager#setProxyPort
     */
    public void setProxyPort(int port) {
        mgr.setProxyPort(port);
    }

    /**
     * Specifies an alternative factory for SSL sockets.
     * @see HttpConnection#setSSLSocketFactory(SSLSocketFactory) HttpConnection
     *      setSSLSocketFactory
     * @param sslSocketFactory a living instance of the alternative
     *      SSLSocketFactory
     */
    public void setSSLSocketFactory(SSLSocketFactory sslSocketFactory) {
        this.sslSocketFactory = sslSocketFactory;
    }

    /**
     * Returns the HTTP proxy server port number
     * 
     * @return the HTTP proxy server port number
     * @see HttpConnectionManager#getProxyPort
     */
    public int getProxyPort() {
        return mgr.getProxyPort();
    }

    /**
     *
     * Execute the given {@link HttpUrlMethod} using my current
     * {@link HttpConnection connection} and {@link HttpState}.  
     *
     * @param method the {@link HttpMethod} to execute. Must be non-null.
     * @return the method's response code
     *
     * @throws IOException if an I/O error occurs
     * @throws HttpException if a protocol exception occurs
     *
     */
    public int executeMethod(HttpUrlMethod method) 
    throws IOException, HttpException {
        log.trace("enter executeMethod(HttpUrlMethod)");

        if (null == method) {
            throw new NullPointerException("method parameter");
        }

        HttpConnection connection = mgr.getConnection(method.getUrl(), 
            timeoutConnection);
        connection.setSoTimeout(timeoutRequest);
        if (!connection.isOpen()) {
            connection.setSSLSocketFactory(sslSocketFactory);
        }

        int status = 0;

        method.setStrictMode(strictMode);

        try {
            status = method.execute(getState(), connection);
        } catch (IOException ex) {
            throw ex;
        } catch (HttpException ex) {
            throw ex;
        } finally {
            mgr.releaseConnection(connection);
        }

        // FIXME: Why is this different from the redirect processing in
        //        HttpMethodBase
        if (status == HttpStatus.SC_MOVED_PERMANENTLY 
            || status == HttpStatus.SC_MOVED_TEMPORARILY
            || status == HttpStatus.SC_SEE_OTHER
            || status == HttpStatus.SC_TEMPORARY_REDIRECT) {
            Header header = method.getResponseHeader("Location");
            String url = header.getValue();
            if (url == null) {
                log.error("HttpMultiClient.executeMethod:  Received redirect "
                    + "without Location header.");
                throw new HttpException("Received redirect without Location "
                    + "header.");
            }

            method.recycle();
            method.setUrl(url);
            return executeMethod(method);
        }

        return status;
    }

}
