#!/bin/bash
#
#  userful-video-command.sh
#
############################


############
# usage()
############
function usage()
{
	echo ""
	echo "Usage: userful video [OPTION]"
	echo ""
	echo -e "\tOPTIONS:"
	echo -e "\t\t-h"
	echo -e "\t\t-help\t- Print this help message"
	echo -e "\t\t-show\t- Show every video card"
	echo -e "\t\t-blacklist [opt] [arg]\t- Manipulate blacklist"
	echo -e "\t\t-whitelist [opt] [arg]\t- Manipulate whitelist"
	echo ""
	exit 1
}

###################
# usageBlacklist()
###################
function usageCommon()
{
	echo ""
	echo "Usage: userful video -$1 [OPT] [ARG]"
	echo ""
	echo -e "\tOPTIONS:"
	echo -e "\t\t-h"
	echo -e "\t\t-help\t\t- Print this help message"
	echo -e "\t\t-show\t\t- Show video $1"
	echo -e "\t\t-enable\t\t- Enable video $1"
	echo -e "\t\t-disable\t- Disable video $1"
	echo -e "\t\t--add [DEVICE_ID]\t- Add [DEVICE_ID] into $1"
	echo -e "\t\t--remove [DEVICE_ID]\t- Remove [DEVICE_ID] from $1"
	echo ""
	exit 1
}

###################
# showVideoList()
#
# 	Parameters:
# 		list_type
###################
function showVideoList()
{
	[[ -z $1 ]] && usage

	if [[ "$1" == "blacklist" ]]; then
		while read line; do
			#if [[ ! "$line" =~ "^[[:space:]]*#.*" ]]; then
			if ! echo $line | grep -q '^[[:space:]]*#.*'; then
				echo $line
			fi
		done < /etc/userful/video-blacklist
	elif [[ "$1" == "whitelist" ]]; then
		while read line; do
			#if [[ ! "$line" =~ "^[[:space:]]*#.*" ]]; then
			if ! echo $line | grep -q '^[[:space:]]*#.*'; then
				echo $line
			fi
		done < /etc/userful/video-whitelist
	else
		usage
	fi
}

###################################
# setVideoList()
#
# 	Parameters:
# 		$1: blacklist or whitelist
# 		$2: -disable or -enable
###################################
function setVideoList()
{
	[[ -z $1 || -z $2 ]] && usage

	if [[ "$1" == "blacklist" ]];then
		# set video-blacklist
		video_list_file=/etc/userful/video-blacklist
		list_string=VIDEO_BLACKLIST
	elif [[ "$1" == "whitelist" ]];then
		# set video-whitelist
		video_list_file=/etc/userful/video-whitelist
		list_string=VIDEO_WHITELIST
	else
		usage
	fi

	if [[ "$2" == "-disable" ]]; then
		setting_value=off
		echo -e "\nDisable $list_string.."
	elif [[ "$2" == "-enable" ]]; then
		setting_value=on
		echo -e "\nEnable $list_string.."
	else
		usageCommon $1
	fi

	#
	# update video-blacklist or video-whitelist
	if grep "^[[:space:]]*$list_string=" $video_list_file > /dev/null 2>&1; then
		sed -i -e "s/^[[:space:]]*$list_string=.*/$list_string=$setting_value/g" $video_list_file
	elif grep "^[[:space:]]*#[[:space:]]*$list_string=" $video_list_file > /dev/null 2>&1; then
		sed -i -e "s/^[[:space:]]*#[[:space:]]*$list_string=.*/$list_string=$setting_value/g" $video_list_file
	else
		echo "$list_string=$setting_value" >> $video_list_file
	fi

	echo -e "Done !\n"
}

########################################
# removeVideoList()
#   - Remove entry(s) from blacklist/whitelist
#
# Parameters:
# 	$1: blacklist or whitelist
# 	$2-$n : DEVICE_ID to remove
#########################################
function removeVideoList()
{
	list_type=$1
	if [[ "$1" == "blacklist" ]];then
		video_list_file=/etc/userful/video-blacklist
	elif [[ "$1" == "whitelist" ]];then
		video_list_file=/etc/userful/video-whitelist
	else
		usage
	fi

	[[ -z $2 ]] && usageCommon $1

	shift
	while [[ -n $1 ]]; do
		if ! grep "^[[:space:]]*\<$1\>" $video_list_file > /dev/null 2>&1; then
			echo -e "\nDEVICE_ID \"$1\" is not in $list_type."
		else
			echo -e "\nRemoved \"$1\" from $list_type..."
			echo -e "\nReboot system to reconfigure X config with $list_type."
			sed -i -e "/^[[:space:]]*\<$1\>/d" $video_list_file
			# Cause reconfiguration of X on next reboot
			[[ -f /etc/X11/userful.Mxorg.conf ]] && rm -f /etc/X11/userful.Mxorg.conf
		fi
		shift
	done
	echo
}

########################################
# addVideoList()
#   - Add entry(s) to blacklist/whitelist
#
# Parameters:
# 	$1: blacklist or whitelist
# 	$2-$n : DEVICE_ID to add
#########################################
function addVideoList()
{
	list_type=$1
	if [[ "$1" == "blacklist" ]];then
		video_list_file=/etc/userful/video-blacklist
	elif [[ "$1" == "whitelist" ]];then
		video_list_file=/etc/userful/video-whitelist
	else
		usage
	fi

	[[ -z $2 ]] && usageCommon $1

	tmp_cards=$(mktemp /tmp/videocards.XXXXXXXX)
	/opt/userful/bin/userful-command-tool.sh video -show >> $tmp_cards

	#cat $tmp_cards

	shift
	while [[ -n $1 ]]; do

		if grep "^[[:space:]]*\<$1\>" $video_list_file > /dev/null 2>&1; then
			# Found same DEVICE_ID in the current list file
			echo -e "\nDEVICE_ID \"$1\" is already in the $list_type."
			shift
			continue
		fi

		if ! grep "\<DEVICE_ID=$1\>" $tmp_cards > /dev/null 2>&1; then
			# No video card with same DEVICE_ID
			echo -e "\nThere is no video card with the DEVICE_ID \"$1\" in your system."
			INPUT=
			while [[ "$INPUT" != "y" && "$INPUT" != "n" ]]; do
				echo -n "Do you want to add \"$1\" into $list_type ? [y/n] "
				read -n 1 INPUT
				echo 
			done

			if [[ "$INPUT" == "y" ]]; then
				echo -e "\nAdded \"$1\" to $list_type."
				echo -e "\nReboot system to reconfigure X config with $list_type."
				echo "$1" >> $video_list_file
				# Cause reconfiguration of X on next reboot
				[[ -f /etc/X11/userful.Mxorg.conf ]] && rm -f /etc/X11/userful.Mxorg.conf
			fi

		else
			echo -e "\nAdded \"$1\" to $list_type."
			echo -e "\nReboot system to reconfigure X config with $list_type."
			echo "$1" >> $video_list_file
			# Cause reconfiguration of X on next reboot
			[[ -f /etc/X11/userful.Mxorg.conf ]] && rm -f /etc/X11/userful.Mxorg.conf
		fi
		shift
	done

	echo
	rm -f $tmp_cards
}

################
# videoList()
################
function videoList()
{
	list_type=$1
	shift

	[[ -z $1 ]] && usageCommon $list_type

	case "$1" in
		-h|-help)
			usageCommon $list_type
			;;

		-show)
			showVideoList $list_type
			;;

		-enable|-disable)
			setVideoList $list_type $1
			;;

		--add)
			shift
			addVideoList $list_type $*
			;;

		--remove)
			shift
			removeVideoList $list_type $*
			;;

		*)
			echo -e "\nError: Invalid option - $1"
			usageCommon $list_type
	esac
}


#-------------------
# Start from here
#-------------------

# Import functions
source /opt/userful/bin/userful-bash-lib.sh

#
# Check if user has root access..
IsRoot

[[ -z $1 ]] && usage

case "$1" in
	-h|-help)
		usage
		;;

	-show)
		showVideoCards
		;;

	-blacklist)
		shift
		videoList blacklist $*
		;;

	-whitelist)
		shift
		videoList whitelist $*
		;;

	*)
		echo -e "\nError: Invalid option - $1"
		usage
		;;
esac

