/*  parseArgs(): parse the command line
 *
 *  Because this program must pass a part of its command line
 *  unchanged to the new program, argc / argv are inappropriate. 
 *  Instead, it uses a copy of the original PSP command line.
 *
 *  The variable 'c' points to the current position on the command line.
 *
 *  Both '-' and '/' are recognized as switch characters.
 *  The first argument not starting with '-' or '/', will be
 *  considered to be the filename.
 */

int parseArgs(char far *psp_cmdline)
{
   char cmdline[128];
   char *fnameStart;
   char *c = cmdline;
   int i;
   
   /* Convert the PSP command line to a normal, zero-terminated C string */
   farmemcpy(cmdline, psp_cmdline + 1, psp_cmdline[0]);
   cmdline[ psp_cmdline[0] ] = 0;
   
   /* reset the switch-flags */
   for (i = 0; i < switchNo; i++)
        sw[i] = 0;

   /* get this program's switches */
   while (*c) {
       c += strspn(c, " \t");    /* skip spaces */

       if (*c != '/' && *c != '-') break;   /* the filename starts here */
       ++c;

       if (readSwitch(&c) != OK)          
           return err_invalid_cmdline;
   }

   if (sw[sw_help])    /* help switch found */
       return err_help;

   /* does a file name follow? */
   if (!c[0]) return err_invalid_cmdline;

   /* The next argument will be the file name */
   fnameStart = c;
   c += strcspn(c, filename_delimitors); 
   memcpy(filename, fnameStart, c - fnameStart);
   filename[c - fnameStart] = 0;

   if (!filename[0]) 
       return err_invalid_cmdline;
   
   /* Convert the filename to uppercase, for prettier output. The DOS
      function is used, since it supports non-ASCII characters */

   if (_osmajor >= 4) DosStrupr(filename);

   /* Pass the rest of the command line as arguments to the new program */
   strcpy(newcmdline + 1, c);

   /* The command line starts with a length count, and ends with a CR */
   newcmdline[0] = strlen(c);
   newcmdline[ newcmdline[0] + 1] = '\r';
      
   return OK;
}

/*  readSwitch():
 *
 *  This routine will read a switch, and any associated parameters.
 *  If an unknown switch is found, the command line is invalid.
 *  
 *  The current position pointer will be advanced.
 */

int readSwitch(char **pc)
{
   int i;
   char *c = *pc;
   
   while (*c != 0 && *c != ' ' && *c != '\t')
   {
     for (i = 0; i < switchNo; i++) {
        size_t swlen = strlen(swStrings[i]);
    
        if (!strnicmp(c, swStrings[i], swlen)) {    
            c += swlen;
            break;
        }
     }
     
     if (i == switchNo)  /* this was not a valid switch */
         return err_invalid_cmdline;
 
     if (sw[i])          /* this switch already encountered */
         return err_invalid_cmdline;

     sw[i] = 1;

     /* The switches that take arguments require special handling: */
     switch(i)
     {
         long larg;
         char *oldc;
         int r;
     case sw_mem:
         c++;
         larg = getnumber(&c);
 
         if (larg <= 0 || larg > 0xFFFF0)
             return err_invalid_mem;
         minMem = topara(larg);
 
         if (*c == ',') {
             c++;
             larg = getnumber(&c);
             
             if (larg <= 0 || larg > 0xFFFF0)
                 return err_invalid_mem;

             maxMem = topara(larg);
         }
         break;
      case sw_at:
         oldc = c;
         if (!(at_address = strtol(c, &c, 16))) {
             if (c == oldc) 
                 return err_invalid_at_address;
             else at_address = 1;
         }
         break;
     case sw_load:
        for (i = 0; i < umbRegions; i++) umbRegion[i].access = 0;

        r = 0;

        do {
             DWORD region_minSize = 0xffff;
             int region_number = strtol(c + 1, &c, 10);

             if (*c == ',') {
                 c++;
                 if ((larg = getnumber(&c)) != -1L)
                     region_minSize = topara(larg);
             }

             if (region_number >= umbRegions)
                 fprintf (stderr, "LOADHI: Illegal UMB region %d - ignored\n",
                          region_number);
             else {
                 regionOrder[r++] = region_number;
                 umbRegion[region_number].minSize = region_minSize;
                 umbRegion[region_number].access = 1;
             }
         } while (*c == ';');

         for (i = 0; i < umbRegions; i++) {
              if (!umbRegion[i].access)
                  regionOrder[r++] = i;
         }
         umbRegion[0].access = 1;
     }
   }
   
   *pc = c;
   return OK;
}

long getnumber(char **pc)
{
   char *c = *pc;
   long l = strtol(c, pc, 10);

   if (c == *pc) /* no argument found */
       return -1L;

   if ((**pc | 0x20) == 'k')
   {    
       l <<= 10;
       (*pc)++;
   }
   return l;
}
