/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.internal.core.feature;

import java.io.PrintWriter;
import java.util.Hashtable;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.pde.core.plugin.IPlugin;
import org.eclipse.pde.internal.core.PDECore;
import org.eclipse.pde.internal.core.ifeature.*;
import org.eclipse.pde.internal.core.ifeature.IFeatureImport;
import org.w3c.dom.Node;

public class FeatureImport
	extends VersionableObject
	implements IFeatureImport {
	private int match = NONE;
	private int idMatch = PERFECT;
	private IPlugin plugin;
	private IFeature feature;
	private int type = PLUGIN;
	private boolean patch = false;
	private String os;
	private String ws;
	private String arch;

	public FeatureImport() {
	}

	public IPlugin getPlugin() {
		return plugin;
	}

	public IFeature getFeature() {
		return feature;
	}

	public int getIdMatch() {
		return idMatch;
	}

	public void setPlugin(IPlugin plugin) {
		this.plugin = plugin;
	}

	public void setFeature(IFeature feature) {
		this.feature = feature;
	}

	public String getOS() {
		return os;
	}

	public String getWS() {
		return ws;
	}

	public String getArch() {
		return arch;
	}

	protected void reset() {
		super.reset();
		patch = false;
		type = PLUGIN;
		match = NONE;
		idMatch = PERFECT;
		arch = null;
		os = null;
		ws = null;
	}

	protected void parse(Node node, Hashtable lineTable) {
		super.parse(node, lineTable);
		bindSourceLocation(node, lineTable);
		this.id = getNodeAttribute(node, "plugin");
		if (id != null)
			type = PLUGIN;
		else {
			this.id = getNodeAttribute(node, "feature");
			if (id != null)
				type = FEATURE;
		}
		this.os = getNodeAttribute(node, "os");
		this.ws = getNodeAttribute(node, "ws");
		this.arch = getNodeAttribute(node, "arch");
		String mvalue = getNodeAttribute(node, "match");
		if (mvalue != null && mvalue.length() > 0) {
			String[] choices = RULE_NAME_TABLE;
			for (int i = 0; i < choices.length; i++) {
				if (mvalue.equalsIgnoreCase(choices[i])) {
					match = i;
					break;
				}
			}
		}
		mvalue = getNodeAttribute(node, "id-match");

		if (mvalue != null && mvalue.length() > 0) {
			if (mvalue.equalsIgnoreCase(RULE_PREFIX))
				idMatch = PREFIX;
		}
		patch = getBooleanAttribute(node, "patch");
		if (id != null) {
			if (type == PLUGIN)
				setPlugin(
					PDECore.getDefault().findPlugin(id, getVersion(), match));
			else
				setFeature(
					PDECore.getDefault().findFeature(id, getVersion(), match));
		}
	}

	public void loadFrom(IFeature feature) {
		reset();
		this.feature = feature;
		type = FEATURE;
		id = feature.getId();
		version = feature.getVersion();
	}

	public int getMatch() {
		return match;
	}

	public void setMatch(int match) throws CoreException {
		ensureModelEditable();
		Integer oldValue = new Integer(this.match);
		this.match = match;
		firePropertyChanged(P_MATCH, oldValue, new Integer(match));
	}

	public void setIdMatch(int idMatch) throws CoreException {
		ensureModelEditable();
		Integer oldValue = new Integer(this.idMatch);
		this.idMatch = idMatch;
		firePropertyChanged(P_ID_MATCH, oldValue, new Integer(idMatch));
	}

	public int getType() {
		return type;
	}

	public void setType(int type) throws CoreException {
		ensureModelEditable();
		Integer oldValue = new Integer(this.type);
		this.type = type;
		firePropertyChanged(P_TYPE, oldValue, new Integer(type));
	}

	public boolean isPatch() {
		return patch;
	}

	public void setPatch(boolean patch) throws CoreException {
		ensureModelEditable();
		Boolean oldValue = new Boolean(this.patch);
		this.patch = patch;
		firePropertyChanged(P_PATCH, oldValue, new Boolean(patch));
	}

	public void setOS(String os) throws CoreException {
		ensureModelEditable();
		String oldValue = this.os;
		this.os = os;
		firePropertyChanged(P_OS, oldValue, os);
	}

	public void setWS(String ws) throws CoreException {
		ensureModelEditable();
		String oldValue = this.ws;
		this.ws = ws;
		firePropertyChanged(P_WS, oldValue, ws);
	}

	public void setArch(String arch) throws CoreException {
		ensureModelEditable();
		String oldValue = this.arch;
		this.arch = arch;
		firePropertyChanged(P_ARCH, oldValue, arch);
	}

	public void restoreProperty(String name, Object oldValue, Object newValue)
		throws CoreException {
		if (name.equals(P_MATCH)) {
			setMatch(newValue != null ? ((Integer) newValue).intValue() : 0);
		} else if (name.equals(P_ID_MATCH)) {
			setIdMatch(newValue != null ? ((Integer) newValue).intValue() : 0);
		} else if (name.equals(P_TYPE)) {
			setType(
				newValue != null ? ((Integer) newValue).intValue() : PLUGIN);
		} else if (name.equals(P_PATCH)) {
			setPatch(
				newValue != null ? ((Boolean) newValue).booleanValue() : false);
		} else if (name.equals(P_OS)) {
			setOS((String) newValue);
		} else if (name.equals(P_WS)) {
			setWS((String) newValue);
		} else if (name.equals(P_ARCH)) {
			setArch((String) newValue);
		} else
			super.restoreProperty(name, oldValue, newValue);
	}

	public void write(String indent, PrintWriter writer) {
		String typeAtt = type == FEATURE ? "feature" : "plugin";
		writer.print(indent + "<import " + typeAtt + "=\"" + getId() + "\"");
		if (getVersion() != null) {
			writer.print(" version=\"" + getVersion() + "\"");
		}
		if (!patch && match != NONE) {
			writer.print(" match=\"" + RULE_NAME_TABLE[match] + "\"");
		}
		if (idMatch == PREFIX) {
			writer.print(" id-match=\"prefix\"");
		}
		if (os != null) {
			writer.print(" os=\"" + getOS() + "\"");
		}
		if (ws != null) {
			writer.print(" ws=\"" + getWS() + "\"");
		}
		if (arch != null) {
			writer.print(" arch=\"" + getArch() + "\"");
		}
		if (patch) {
			writer.print(" patch=\"true\"");
		}
		writer.println("/>");
	}
	public String toString() {
		if (plugin != null)
			return plugin.getTranslatedName();
		else if (feature != null)
			return feature.getLabel();
		return getId();
	}
}
