/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.team.internal.core.streams.PollingInputStream;
import org.eclipse.team.internal.core.streams.PollingOutputStream;
import org.eclipse.team.internal.core.streams.TimeoutOutputStream;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.Policy;

class ControlConnection {
	private InetAddress address;
	private int port;

	private Socket socket;
	protected InputStream socketIn;
	protected OutputStream socketOut;
	
	public ControlConnection(InetAddress address, int port) {
		this.address = address;
		this.port = port;
	}
	
	/**
	 * Opens and authenticates the control connection.
	 * @param monitor the progress monitor
	 */
	public void open(int timeout, IProgressMonitor monitor) throws FTPException {
		monitor.beginTask(null, 100);
		try {
			// open socket
			try {
				socket = new Socket(address, port);
				monitor.worked(90);
			} catch (IOException e) {
				throw new FTPCommunicationException(Policy.bind("FTPClient.CannotOpenControlConnection"), e); //$NON-NLS-1$
			}
			// set socket options
			try {
				socket.setKeepAlive(true);
				setTimeoutEnabled(true);
			} catch (SocketException e) {
				throw new FTPCommunicationException(Policy.bind("FTPClient.CannotSetSocketOptions"), e); //$NON-NLS-1$
			}
			// create input stream
			try {
				socketIn = new PollingInputStream(getSocket().getInputStream(),
					timeout, monitor);
			} catch (IOException e) {
				throw new FTPCommunicationException(Policy.bind("FTPClient.CannotObtainSocketInputStream"), e); //$NON-NLS-1$
			}
			// create output stream
			try { 
				socketOut = new PollingOutputStream(new TimeoutOutputStream(getSocket().getOutputStream(),
					512 /*bufferSize*/, 1000 /*writeTimeout*/, -1 /*closeTimeout*/),
					timeout, monitor);
			} catch (IOException e) {
				throw new FTPCommunicationException(Policy.bind("FTPClient.CannotObtainSocketOutputStream"), e); //$NON-NLS-1$
			}
			monitor.worked(10);
		} finally {
			monitor.done();
		}
	}

	/**
	 * Opens and authenticates the control connection.
	 * @param monitor the progress monitor
	 */
	public void close(IProgressMonitor monitor) throws FTPException {
		if (socket == null) return;
		monitor.beginTask(null, 100);
		// ensure the streams and the socket are closed properly even if multiple exceptions occur
		try {
			try {
				try {
					if (socketIn != null) socketIn.close();
				} finally {
					socketIn = null;
					if (socketOut != null) socketOut.close();
				}
			} finally {
				socketOut = null;
				monitor.worked(30);
				socket.close();
				monitor.worked(70);
			}
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorDisconnecting"), e); //$NON-NLS-1$
		} finally {
			socket = null;
			monitor.done();
		}
	}
	
	/**
	 * Disables or enables the read timeout.
	 * Does not throw exceptions on failure (safe for use in a finally clause).
	 */
	public void setTimeoutEnabled(boolean enable) {
		try {
			getSocket().setSoTimeout(enable ? 1000 : 0);
		} catch (SocketException e) {
			// we deliberately eat the exception here since it is likely that we
			// will be called from a finally clause and we do not wish to lose
			// the more "interesting" exception
			// if the error is serious (eg. socket closed), we are sure to catch it
			// soon enough anyway
		}
	}
	
	/**
	 * Returns the socket or throws an exception.
	 */
	protected Socket getSocket() {
		if (socket == null) {
			throw new IllegalStateException(Policy.bind("FTPClient.SocketIsNotOpen")); //$NON-NLS-1$
		}
		return socket;
	}
	
	/**
	 * Returns an input stream for the socket or throws an exception.
	 */
	public InputStream getInputStream() throws FTPException {
		if (socketIn == null) {
			throw new IllegalStateException(Policy.bind("FTPClient.SocketIsNotOpen")); //$NON-NLS-1$
		}
		return socketIn;
	}

	/**
	 * Returns an output stream for the socket or throws an exception.
	 */
	public OutputStream getOutputStream() throws FTPException {
		if (socketOut == null) {
			throw new IllegalStateException(Policy.bind("FTPClient.SocketIsNotOpen")); //$NON-NLS-1$
		}
		return socketOut;
	}
}
