/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.client;

import java.util.Date;

/**
 * Represents a file stored remotely.  Due to the imprecise nature of
 * decoding remote directory listings, the information contained herein
 * is of a mostly advisory nature and might be wrong on occasion.  This
 * a common phenomenon among FTP clients, unfortunately.
 */
public class FTPDirectoryEntry {
	public static final FTPDirectoryEntry[] EMPTY_ARRAY = new FTPDirectoryEntry[0];
	public static final long UNKNOWN_SIZE = -1;
	private String name;
	private boolean hasDirectorySemantics;
	private boolean hasFileSemantics;
	private long size;
	private Date modTime;
	
	/**
	 * Creates a new directory entry.
	 * @param name the file or directory name (not its full path)
	 * @param hasDirectorySemantics set to true if CWD might succeed
	 * @param hasFileSemantics set to true if RETR might succeed
	 * @param size the file size in bytes, or UNKNOWN_SIZE if unknown
	 * @param modTime the modification stamp or null if unknown
	 */
	public FTPDirectoryEntry(String name, boolean hasDirectorySemantics, boolean hasFileSemantics,
		long size, Date modTime) {
		this.name = name;
		this.hasDirectorySemantics = hasDirectorySemantics;
		this.hasFileSemantics = hasFileSemantics;
		this.size = size;
		this.modTime = modTime;
	}
	
	/**
	 * Returns the name of this directory entry.
	 * 
	 * @return the entry name
	 */
	public String getName() {
		return name;
	}
	
	/**
	 * Returns true only if changeDirectory() _might_ succeed for this directory entry.
	 * Does not imply hasFileSemantics() == false.
	 * 
	 * @return true if the entry might represent a remote directory
	 */
	public boolean hasDirectorySemantics() {
		return hasDirectorySemantics;
	}

	/**
	 * Returns true only if getFile() _might_ succeed for this directory entry.
	 * Does not imply hasDirectorySemantics() == false.
	 * 
	 * @return true if the entry might represent a remote file
	 */
	public boolean hasFileSemantics() {
		return hasFileSemantics;
	}
	
	/**
	 * Returns the size of the file in bytes, or <code>UNKNOWN_SIZE</code> if unknown.
	 * 
	 * @return the file size, or UNKNOWN_SIZE
	 */
	public long getSize() {
		return size;
	}
	
	/**
	 * Returns the modification stamp of this file, or <code>null</code> if unknown.
	 * <p>
	 * Mod time is unreliable for FTP files since some servers do not send
	 * send results back in GMT, or provide poor granularity for files that
	 * were modified over 6 months ago.
	 * </p>
	 * 
	 * @return the date, or null
	 */
	public Date getModTime() {
		return modTime;
	}
}
