/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.client;

import java.io.IOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.FTPPlugin;
import org.eclipse.team.internal.ftp.Policy;

class PortDataConnection extends DataConnection {
	private InetAddress address;
	private int port;
	private ServerSocket serverSocket;

	/**
	 * Creates a data transfer manager.
	 * @param worker the worker for the thread
	 */
	public PortDataConnection() {
	}
	
	public InetAddress getLocalAddress() {
		return address;
	}
	
	public int getLocalPort() {
		return port;
	}
	
	public void open(IProgressMonitor monitor) throws FTPException {
		serverSocket = openServerSocket();
		super.open(monitor);
	}
	
	public void close(boolean interrupt) throws FTPException {
		try {
			super.close(interrupt);
		} finally {
			if (interrupt) {
				try {
					closeServerSocket();
				} catch (Exception e) {
					// eat any exceptions resulting from forced closure of the socket
				}
			} else {
				closeServerSocket();
			}
		}
	}
	
	protected ServerSocket openServerSocket() throws FTPException {
		try {
			ServerSocket serverSocket = new ServerSocket(0 /*any port*/, 1 /*backlog*/);
			serverSocket.setSoTimeout(FTPPlugin.getPlugin().getPortConnectionTimeout() * 1000);
			address = InetAddress.getLocalHost(); // FIXME: won't work well for multi-homed hosts
			port = serverSocket.getLocalPort();
			return serverSocket;
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorOpeningPortServerSocket"), e); //$NON-NLS-1$
		}
	}
	
	protected synchronized void closeServerSocket() throws FTPException {
		if (serverSocket == null) return;
		try {
			serverSocket.close();
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorClosingPortServerSocket"), e); //$NON-NLS-1$
		} finally {
			serverSocket = null;
		}
	}
	
	protected Socket openSocket() throws FTPException {
		try {
			ServerSocket ss;
			synchronized (this) {
				ss = serverSocket;
				if (ss == null) {
					// this can happen if we interrupt the thread
					throw new FTPCommunicationException(Policy.bind("FTPClient.CannotOpenPortDataConnection"), null); //$NON-NLS-1$
				}
			}
			return ss.accept();
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.CannotOpenPortDataConnection"), e); //$NON-NLS-1$
		} finally {
			closeServerSocket();
		}
	}	
}
