/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.ftp.client;

import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.team.internal.ftp.FTPException;
import org.eclipse.team.internal.ftp.Policy;

public class Util {
	
	private static final String[] months = new String[] {
		"jan", "feb", "mar", "apr", "may", "jun", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
		"jul", "aug", "sep", "oct", "nov", "dec" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
	};
	
	private static byte[] buffer = new byte[256];
	
	/**
	 * Returns the size of an IFile.
	 */
	public static long getFileSize(IFile file) {
		if (! file.exists()) return 0;
		return file.getLocation().toFile().length();
	}
	
	/**
	 * If the number of segments in the path is greater than <code>split</code> then the returned path
	 * is truncated to <code>split</code> number of segments and '...' is shown as the first segment
	 * of the path.
	 */
	public static String toTruncatedPath(IPath path, int split) {
		int segments = path.segmentCount();
		if(segments>split) {				
			IPath last = path.removeFirstSegments(segments - split);
			return "..." + IPath.SEPARATOR + last.toString(); //$NON-NLS-1$
		}
		return path.toString();
	}

	/**
	 * Reads a CR/LF delimiter line from the input stream and discards the delimiter.
	 * @param in the input stream
	 * @return the line, or null on EOF
	 */
	public static String readListLine(InputStream in) throws FTPException {
		int index = 0;
		boolean seenCR = false;
		try {
			for (;;) {
				// read a byte
				int c = in.read();
				if (c == -1) return null;
				if (index == buffer.length) {
					byte[] oldBuffer = buffer;
					buffer = new byte[oldBuffer.length * 2];
					System.arraycopy(oldBuffer, 0, buffer, 0, oldBuffer.length);
				}
				buffer[index++] = (byte) c;
				
				if (c == Constants.CR) {
					seenCR = true;
				} else if (c == Constants.LF && seenCR) {
					return new String(buffer, 0, index - 2);
				} else {
					seenCR = false;
				}
			}
		} catch (IOException e) {
			throw new FTPCommunicationException(Policy.bind("FTPClient.ErrorReceivingDirectoryList"), e); //$NON-NLS-1$
		}
	}
		
	/**
	 * Parses /bin/ls format lines.
	 * <p>
	 * Understands the following formats:
	 * <ul>
	 *   <li>d<permissions> <blocks> <uid> <gid> <size> <date> <dirname></li>
	 *   <li>l<permissions> <blocks> <uid> <gid> <size> <date> <file or dirname> -> <link></li>
	 *   <li>-<permissions> <blocks> <uid> <gid> <size> <date> <filename></li>
	 *   <li>d <permissions> <uid> <size> <date> <filename></li>
	 *   <li>- <permissions> <uid> <size> <date> <filename></li>
	 * </ul>
	 * Recognizes the following date formats:
	 * <ul>
	 *   <li>MMM dd [hh:mm] [yyyy]</li>
	 * </ul></p>
	 * 
	 * @param line the line
	 * @return the file info, or null if the line could not be parsed
	 */
	public static FTPDirectoryEntry parseLSLine(String line) {
		boolean hasDirectorySemantics = false;
		boolean hasFileSemantics = false;
		switch (line.charAt(0)) { // see man page for /bin/ls
			case 'd': // directory
				hasDirectorySemantics = true;
				break;
			case 'l': // symbolic link, don't know if this is a directory or a file
				hasDirectorySemantics = true;
				hasFileSemantics = true;
				break;
			case '-': // regular file
				hasFileSemantics = true;
				break;
			case 'c': // character special file
			case 'b': // block special file
			case 'p': // named pipe
			case 's': // socket
			case 'D': // Solaris door (IPC mechanism?)
				break;
			default:
				return null;
		}
		StringTokenizer tok = new StringTokenizer(line, " \t"); //$NON-NLS-1$
		try {
			
			// read permissions, blocks, uid, gid, size
			String permissions = tok.nextToken();
			if (permissions.length() == 1) {
				// Handle format observed from certain servers
				permissions = tok.nextToken();
				String uid = tok.nextToken();
			} else {
				long blocks = Long.parseLong(tok.nextToken(), 10);
				String uid = tok.nextToken();
				String gid = tok.nextToken();
			}
			long size = Long.parseLong(tok.nextToken(), 10);
			
			// read date (try hard!)
			String token;
			Calendar calendar = GregorianCalendar.getInstance(Constants.DEFAULT_TIMEZONE);
			if (! parseMonth(calendar, token = tok.nextToken())) return null;
			if (! parseDay(calendar, token = tok.nextToken())) return null;
			if (! parseTime(calendar, token = tok.nextToken())) {
				calendar.set(Calendar.HOUR_OF_DAY, 0);
				calendar.set(Calendar.MINUTE, 0);
				calendar.set(Calendar.SECOND, 0);
			} else {
				token = tok.nextToken();
			}
			if (! parseYear(calendar, token)) {
				// some servers don't send the year if the date is within the last 6/12 months
				// assume the date is within the last 12 months or so (cannot handle dates in the future)
				Calendar current = GregorianCalendar.getInstance(Constants.DEFAULT_TIMEZONE);
				current.setTime(new Date(System.currentTimeMillis()));
				int currentYear = current.get(Calendar.YEAR);
				int currentDayOfYear = current.get(Calendar.DAY_OF_YEAR);
				int dayOfYear = calendar.get(Calendar.DAY_OF_YEAR);
				if (dayOfYear > currentDayOfYear + 1) { // +1 for leapyears and subtle discrepancies
					calendar.set(Calendar.YEAR, currentYear - 1);
				} else {
					calendar.set(Calendar.YEAR, currentYear);
				}
			} else {
				token = tok.nextToken();
			}
			
			// read filename
			StringBuffer filename = new StringBuffer(token);
			while (tok.hasMoreTokens()) {
				token = tok.nextToken();
				if (token.equals("->")) break; // followed by the symlink pointer //$NON-NLS-1$
				filename.append(' ');
				filename.append(token);
			}
			
			// done!
			return new FTPDirectoryEntry(filename.toString(), hasDirectorySemantics, hasFileSemantics,
				size, calendar.getTime());
		} catch (NoSuchElementException e) {
			// expected more tokens
			return null;
		} catch (NumberFormatException e) {
			// don't understand the format so ignore the line
			return null;
		}
	}
	
	private static boolean parseMonth(Calendar calendar, String text) {
		for (int i = 0; i < months.length; ++i) {
			if (text.equalsIgnoreCase(months[i])) {
				calendar.set(Calendar.MONTH, i);
				return true;
			}
		}
		return false;
	}
	
	private static boolean parseDay(Calendar calendar, String text) {
		try {
			int day = Integer.parseInt(text, 10);
			if (day > 0 && day < 32) {
				calendar.set(Calendar.DAY_OF_MONTH, day);
				return true;
			}
		} catch (NumberFormatException e) {
		}
		return false;
	}
	
	private static boolean parseYear(Calendar calendar, String text) {
		try {
			int year = Integer.parseInt(text, 10);
			if (year >= 1900 && year <= 2100) {
				calendar.set(Calendar.YEAR, year);
				return true;
			}
		} catch (NumberFormatException e) {
		}
		return false;
	}
	
	private static boolean parseTime(Calendar calendar, String time) {
		try {
			int colonPos = time.indexOf(':');
			if (colonPos != -1) {
				int hour = Integer.parseInt(time.substring(0, colonPos), 10);
				if (hour == 24) hour = 0;
				if (hour >= 0 && hour <= 23) {
					int minute = Integer.parseInt(time.substring(colonPos + 1), 10);
					if (minute >= 0 && minute <= 59) {
						calendar.set(Calendar.HOUR_OF_DAY, hour);
						calendar.set(Calendar.MINUTE, minute);
						calendar.set(Calendar.SECOND, 0);
						return true;
					}
				}
			}
		} catch (NumberFormatException e) {
		}
		return false;
	}
	
	/**
	 * Parses EPLF format lines (can be intermixed with /bin/ls but much more reliable)
	 * <p>
	 * Understands the following formats:
	 * <ul>
	 *   <li>+r,s50,m12345678,\tmyFile.txt
	 * </ul></p>
	 * 
	 * @param line the line
	 * @return the file info, or null if the line could not be parsed
	 */
	public static FTPDirectoryEntry parseEPLFLine(String line) {
		if (line.charAt(0) != '+') return null;
		try {
			boolean hasDirectorySemantics = false;
			boolean hasFileSemantics = false;
			long size = FTPDirectoryEntry.UNKNOWN_SIZE;
			Date modTime = null;
			String filename = null;
			StringTokenizer tok = new StringTokenizer(line.substring(1), ",\t", true); //$NON-NLS-1$
			while (tok.hasMoreTokens()) {
				String token = tok.nextToken();
				if (token.length() == 0) continue;
				switch (token.charAt(0)) {
					case 'r': // can RETR
						hasFileSemantics = true;
						break;
					case '/': // can CWD
						hasDirectorySemantics = true;
						break;
					case 's': // file size in bytes
						size = Long.parseLong(token.substring(1), 10);
						break;
					case 'm': // mod time in seconds since Jan 1, 1970 GMT
						modTime = new Date(Long.parseLong(token.substring(1), 10) * 1000);
						break;
					case ',': // separator between facts
						break;
					case '\t': // separator before file name
						if (tok.hasMoreTokens()) filename = tok.nextToken();
						break;
				}
			}
			if (filename != null) {
				return new FTPDirectoryEntry(filename, hasDirectorySemantics, hasFileSemantics, size, modTime);
			}
		} catch (NumberFormatException e) {
			// don't understand the format so ignore the line
		}
		return null;
	}
}
