/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.core;

import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.core.sync.IRemoteResource;
import org.eclipse.team.internal.core.streams.PollingInputStream;
import org.eclipse.team.internal.core.streams.ProgressMonitorInputStream;
import org.eclipse.team.internal.core.target.IRemoteTargetResource;
import org.eclipse.team.internal.core.target.ITeamStatusConstants;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.core.target.UrlUtil;
import org.eclipse.webdav.ILocator;
import org.eclipse.webdav.IResponse;
import org.eclipse.webdav.client.AbstractResourceHandle;
import org.eclipse.webdav.client.CollectionHandle;
import org.eclipse.webdav.client.DAVClient;
import org.eclipse.webdav.client.PropertyStatus;
import org.eclipse.webdav.client.ResourceHandle;
import org.eclipse.webdav.client.WebDAVFactory;
import org.eclipse.webdav.dom.ElementEditor;
import org.eclipse.webdav.internal.kernel.ClientException;
import org.eclipse.webdav.internal.kernel.DAVException;
import org.eclipse.webdav.internal.kernel.WebDAVException;
import org.eclipse.webdav.internal.kernel.WebDAVPropertyNames;
import org.w3c.dom.Element;

/**
 * A remote target resource is a client-side 'proxy' for the server 
 * resource.
 * <p>
 * It is certainly posible to create a stale or invalid handle in
 * numerous ways.  For example, the existance of a handle does not
 * imply the existance of a  corresponding server resource (indeed
 * resource can be created by sending create() to a handle, or deleted
 * using delete() -- so the life cycles are not coupled.  It is also
 * possible to create, say, a collection handle on a regular resource
 * and invoke invalid WebDAV methods.  These will typically result
 * in an exception.</p>
 */
public class DavRemoteTargetResource extends PlatformObject implements IRemoteTargetResource {

	/*
	 * The WebDAVFactory is used to create new resource handles, locators, and
	 * contexts, etc.
	 */
	protected static WebDAVFactory DAV_FACTORY = new WebDAVFactory();
	
	/*
	 * This is a handle to the remote DAV element
	 */
	private AbstractResourceHandle remoteResource;
	
	/*
	 * The site where this resource handle was created from
	 */
	private Site site;
	 
	 /*
	  * URL to the parent 
	  */
	private URL root;
	
	/*
	 * Remember is this is a collection handle 
	 */
	private boolean isContainer;
	
	public static final int TRANSFER_PROGRESS_INCREMENT = 32768;
	
	/**
	 * Create a remote resource handle to the given URL
	 */
	public DavRemoteTargetResource(DAVClient davClient, Site site, URL root, IPath path, boolean isContainer) {
		String remoteResourceURL = UrlUtil.concatString(root.toExternalForm(), path);

		if(isContainer)
			remoteResourceURL = UrlUtil.makeAbsolute(remoteResourceURL);
			
		ILocator locator = DAV_FACTORY.newLocator(remoteResourceURL);
		this.root = root;
		this.remoteResource = new ResourceHandle(davClient, locator);
		this.site = site;
		this.isContainer = isContainer;
	}
	
	/**
	 * Create a remote resource handle based on an existing dav handle
	 */
	public DavRemoteTargetResource(AbstractResourceHandle remoteResource, Site site, boolean isContainer) {
		this.remoteResource = remoteResource;
		this.site = site;
		this.isContainer = isContainer;
	}
	
	/**
	 * @see IRemoteTargetResource#getURL()
	 */
	public URL getURL() {
		try {
			return new URL(remoteResource.getLocator().getResourceURL());
		} catch (MalformedURLException e) {
			TeamWebDavPlugin.log(IStatus.ERROR, e.getMessage(), e);
			return null;
		}
	}

	/**
	 * @see IRemoteTargetResource#getSize()
	 */
	public int getSize(IProgressMonitor monitor) throws TeamException {
		try {
			if(isContainer()) {
				return 0;
			} else {
				PropertyStatus prop =
					remoteResource.getProperty(WebDAVPropertyNames.DAV_GET_CONTENT_LENGTH);
				if (prop.getStatusCode() != IResponse.SC_OK)
					throw new TeamException(ITeamStatusConstants.IO_FAILED_STATUS);
				Element value = prop.getProperty();
				return new Integer(ElementEditor.getFirstText(value)).intValue();
			}
		} catch (DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
	}

	/**
	 * @see IRemoteTargetResource#getLastModified()
	 */
	public String getLastModified(IProgressMonitor monitor) throws TeamException {
		try {
			PropertyStatus prop =
				remoteResource.getProperty(WebDAVPropertyNames.DAV_GET_LAST_MODIFIED);
			if (prop.getStatusCode() != IResponse.SC_OK)
				throw new TeamException(ITeamStatusConstants.IO_FAILED_STATUS);
			Element value = prop.getProperty();
			return ElementEditor.getFirstText(value);
		} catch (DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
	}

	/**
	 * @see IRemoteResource#getName()
	 */
	public String getName() {
		try {
			// The DAV remote resource's location can have the following forms:
			// http://www.server.com/
			// http://www.server.com/dav/auth/
			URL url = new URL(remoteResource.getLocator().getResourceURL());
			Path path = new Path(url.getFile());
			
			// if the remote resource is at the root of the remote site then
			// return the root path
			if(path.isRoot() || path.isEmpty()) {
				return path.toString();				
			// otherwise, if the resource is not at the root return the last 
			// segment of the location as its name
			} else {
				return path.lastSegment().toString();
			}
		} catch (MalformedURLException e) {
			TeamWebDavPlugin.log(IStatus.ERROR, e.getMessage(), e);
			return Policy.bind("DavRemoteTargetResource.errorName"); // $NON-NLS-1$ //$NON-NLS-1$
		}
	}

	/**
	 * @see IRemoteResource#members(IProgressMonitor)
	 */
	public IRemoteResource[] members(IProgressMonitor progress) throws TeamException {
		progress = Policy.monitorFor(progress);
		try {		
			progress.beginTask(Policy.bind("DavRemoteTargetResource.membersProgress", getName()), 100); //$NON-NLS-1$
			progress.subTask(Policy.bind("DavRemoteTargetResource.membersProgress", getName())); //$NON-NLS-1$
			// Only containers have children.
			if (!isContainer())
				return new IRemoteTargetResource[0];
	
			// Query the DAV server to find the children (i.e., members) of the receiver.
			Set remoteChildren;
			try {
				remoteChildren = remoteResource.asCollectionHandle().getMembers();
				progress.worked(80);
			} catch (DAVException exception) {
				// Problem getting the children.
				throw TeamWebDavPlugin.wrapException(exception);
			}
	
			// Have to contruct resource states from DAV resource handles.
			DavRemoteTargetResource[] result = new DavRemoteTargetResource[remoteChildren.size()];
			int index = 0;
			Iterator remoteChildrenItr = remoteChildren.iterator();
			while (remoteChildrenItr.hasNext()) {
				AbstractResourceHandle remoteChild = (AbstractResourceHandle)remoteChildrenItr.next();
				result[index++] = new DavRemoteTargetResource(remoteChild, site, (remoteChild instanceof CollectionHandle));
			}
			progress.worked(20);
	
			// All children processed.
			return result;
		} finally {
			progress.done();
		}
	}

	/**
	 * @see IRemoteResource#getContents(IProgressMonitor)
	 */
	public InputStream getContents(IProgressMonitor progress) throws TeamException {
		try {
			// number of bytes expected
			final int size = getSize(null) >> 10;
			final String name = UrlUtil.toTruncatedPath(getURL(), 3);
			
			// timeout value for this site
			int timeout = ((WebDavSite)site).getTimeout();
			
			final String prefix = Policy.bind("DavRemoteTargetResource.download", name); //$NON-NLS-1$
			progress.subTask(Policy.bind("DavRemoteTargetResource.transferNoSize", prefix)); //$NON-NLS-1$
			
			// wrap the dav input stream with an input stream that allows progress
			// and cancellation.
			InputStream in = new ProgressMonitorInputStream(
									new PollingInputStream(remoteResource.getContent(), 
									timeout,
									progress), 
								0, 
								TRANSFER_PROGRESS_INCREMENT, 
								progress) {
				// perform updates to the progress message, showing the number of bytes transfered
				protected void updateMonitor(long bytesRead, long bytesTotal, IProgressMonitor monitor) {
					if (bytesRead == 0) return;
					monitor.subTask(Policy.bind("DavRemoteTargetResource.transfer", //$NON-NLS-1$
						new Object[] { prefix,
										Long.toString(bytesRead >> 10),
										Integer.toString(size) }));
				}
			};																			
											
			return in;
		} catch (ClientException exception) {
			if (exception.getStatusCode() == IResponse.SC_NOT_FOUND)
				throw new TeamException(ITeamStatusConstants.NO_REMOTE_RESOURCE_STATUS);
			throw TeamWebDavPlugin.wrapException(exception);
		} catch (DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
	}

	/**
	 * @see IRemoteResource#isContainer()
	 */
	public boolean isContainer() {
		return isContainer;
	}

	/**
	 * Delete the remote resource.
	 * 
	 * @param progress a progress monitor to show the progress of the deletion, or
	 * <code>null</code> if progress is not required.
	 * @see IResourceState#delete(IProgressMonitor)
	 */
	public void delete(IProgressMonitor progress) throws TeamException {
		try {
			remoteResource.delete();
		} catch (WebDAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		} catch (DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
	}
	
	/**
	 * Get the E-Tag of the released remote resource.
	 */
	public String getReleasedIdentifier() throws TeamException {
		try {
			PropertyStatus prop =
				remoteResource.getProperty(WebDAVPropertyNames.DAV_GET_E_TAG);
			if (prop.getStatusCode() != IResponse.SC_OK)
				throw new TeamException(ITeamStatusConstants.IO_FAILED_STATUS);
			Element value = prop.getProperty();
			return "ETag:" + ElementEditor.getFirstText(value); //$NON-NLS-1$
		} catch (DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
	}
		
	public boolean exists(IProgressMonitor monitor) throws TeamException {
		try {
			return remoteResource.exists();
		} catch (DAVException e) {
			throw TeamWebDavPlugin.wrapException(e);
		}
	}
	
	/**
	 * Return the DavResource handle
	 */
	public AbstractResourceHandle getDavHandle() {
		return remoteResource;
	}
	/**
	 * @see IRemoteTargetResource#getSite()
	 */
	public Site getSite() {
		return site;
	}

	/**
	 * @see IRemoteTargetResource#getFile(String)
	 */
	public IRemoteTargetResource getFile(String name) {
		return new DavRemoteTargetResource(remoteResource.getDAVClient(), site, getURL(), new Path(name), false /* not a container */);
	}

	/**
	 * @see IRemoteTargetResource#getFolder(String)
	 */
	public IRemoteTargetResource getFolder(String name) {
		return new DavRemoteTargetResource(remoteResource.getDAVClient(), site, getURL(), new Path(name), true /* container */);
	}

	/**
	 * @see IRemoteTargetResource#mkdir()
	 */
	public void mkdirs(IProgressMonitor monitor) throws TeamException {
		try {
			remoteResource.asCollectionHandle().mkdirs();
		} catch (DAVException e) {
			throw TeamWebDavPlugin.wrapException(e);
		}
	}
	
	public boolean equals(Object obj) {
		if(this == obj)
			return true;
		if(!(obj instanceof DavRemoteTargetResource))
			return false;
		DavRemoteTargetResource otherElement = ((DavRemoteTargetResource)obj);
		return otherElement.getRemoteResource().equals(this.remoteResource);
	}
	
	public int hashCode() {
		return this.remoteResource.hashCode();
	}
	
	/**
	 * Gets the remoteResource.
	 * @return Returns a AbstractResourceHandle
	 */
	protected AbstractResourceHandle getRemoteResource() {
		return remoteResource;
	}

	public void setContent(final long size, InputStream is, IProgressMonitor progress) throws DAVException {
		// number of bytes expected
		final String name = UrlUtil.toTruncatedPath(getURL(), 3);
		
		// timeout value for this site
		int timeout = ((WebDavSite)site).getTimeout();

		final String prefix = Policy.bind("DavRemoteTargetResource.upload", name); //$NON-NLS-1$
		progress.subTask(Policy.bind("DavRemoteTargetResource.transferNoSize", prefix)); //$NON-NLS-1$
				
		// wrap the dav input stream with an input stream that allows progress
		// and cancellation.
		InputStream in = new ProgressMonitorInputStream(
								new PollingInputStream(is, 
								timeout,
								progress), 
							0, 
							TRANSFER_PROGRESS_INCREMENT, 
							progress) {
			// perform updates to the progress message, showing the number of bytes transfered
			protected void updateMonitor(long bytesRead, long bytesTotal, IProgressMonitor monitor) {
				if (bytesRead == 0) return;
				monitor.subTask(Policy.bind("DavRemoteTargetResource.transfer", //$NON-NLS-1$
					new Object[] { prefix, Long.toString(bytesRead >> 10), Long.toString(size >> 10) }));
			}
		};
		remoteResource.setContent(in);
	}
	/**
	 * @see org.eclipse.team.internal.core.target.IRemoteTargetResource#canBeReached(IProgressMonitor)
	 */
	public boolean canBeReached(IProgressMonitor monitor) throws TeamException {
		try {
			return this.remoteResource.canTalkDAV();
		} catch (DAVException e) {
			throw TeamWebDavPlugin.wrapException(e);
		}
	}


}
