/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.core;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.target.ResourceState;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.webdav.client.AbstractResourceHandle;
import org.eclipse.webdav.client.DAVClient;
import org.eclipse.webdav.client.WebDAVFactory;
import org.eclipse.webdav.internal.kernel.DAVException;

/**
 * Instances of <code>DAVState</code> represent the state of locally managed resources
 * that have a corresponding remote resource on a WebDAV server.  Operations on the
 * remote resource are marshalled using the WebDAV protocol.
 */
class DavState extends ResourceState {

	/*
	 * The WebDAVFactory is used to create new resource handles, locators, and
	 * contexts, etc.
	 */
	protected static WebDAVFactory DAV_FACTORY = new WebDAVFactory();
	
	/*
	 * The copy buffer is used to ensure efficient resourc content transfer.
	 */
	private static final byte[] COPY_BUFFER = new byte[4096];

	/*
	 * This is the handle to the remote DAV resource represented by this state information.
	 */
	protected DavRemoteTargetResource remoteResource;
	
	/**
	 * Construct a new DAV resource state based on a local file.
	 */
	public DavState(DAVClient davClient, Site site, URL root, IResource localResource) {
		super(localResource, root);
		this.remoteResource = new DavRemoteTargetResource(davClient, site, root, localResource.getProjectRelativePath(), localResource.getType() != IResource.FILE);
	}
	
	/**
	 * Construct a new DAV resource state based on a local file and an existing
	 * remote resource.
	 */
	public DavState(DAVClient davClient, IResource localResource, DavRemoteTargetResource remote) {
		super(localResource, remote.getURL());
		this.remoteResource = remote;
	}

	/**
	 * Get the E-Tag of the released remote resource.
	 */
	public String getReleasedIdentifier(IProgressMonitor progress) throws TeamException {
		return remoteResource.getReleasedIdentifier();
	}

	/**
	 * Returns the array of children of this resource, if any.
	 */
	public ResourceState[] getRemoteChildren(IProgressMonitor progress) throws TeamException {
		// Only containers have children.
		if (!remoteResource.isContainer())
			return new ResourceState[0];

		// Query the DAV server to find the children (i.e., members) of the receiver.
		DavRemoteTargetResource[] remoteChildren = (DavRemoteTargetResource[])remoteResource.members(progress);

		// We will answer with an array of resource states describing the children.
		ResourceState[] result = new ResourceState[remoteChildren.length];
		
		// Have to contruct resource states from DAV resource handles.
		for (int i = 0; i < remoteChildren.length; i++) {
			DavRemoteTargetResource remoteChild = remoteChildren[i];
			
			// Get the path to the child, this is it's full name.
			String childURL = remoteChild.getURL().toExternalForm();
			String fullChildName;
			String childName = remoteChild.getName();

			// Look-up the child, by name in the local resource, note that it may be missing.
			IContainer localContainer = (IContainer) localResource;
			IResource localChild;
			if (remoteChild.isContainer())
				localChild = localContainer.getFolder(new Path(childName));
			else
				localChild = localContainer.getFile(new Path(childName));
				
			// Add the new DAV state to the result array.
			result[i] = new DavState(remoteResource.getDavHandle().getDAVClient(), remoteResource.getSite(), getRoot(), localChild);
		}

		// All children processed.
		return result;
	}

	/**
	 * Answer the type of the remote resource, in terms of the IResource constants.
	 */
	public int getRemoteType() {
		if (remoteResource.isContainer())
			return IResource.FOLDER;
		return IResource.FILE;
	}

	/**
	 * Answer whether the corresponding remote resource exists.
	 * 
	 * @return <code>true</code> if the remote resource exists, and <code>false</code>
	 * if it does not exist.
	 */
	public boolean hasRemote(IProgressMonitor progress) throws TeamException {
		return remoteResource.exists(progress);
	}

	/**
	 * Copy the remote resource down to the local resource.
	 * 
	 * @param progress a progress monitor to show the progress of the copy, or
	 * <code>null</code> if progress is not required.
	 */
	public void download(IProgressMonitor progress) throws TeamException {
		// In the file system, collections do not have content.
		if (getRemoteType() != IResource.FILE)
			return;

		// We know the resource is a file.
		IFile localFile = (IFile)localResource;

		// Optimize the case where the local and remote file are identical.
//		if (localFile.getModificationStamp() == remoteFile.lastModified())
//			return ITeamStatusConstants.OK_STATUS;

//problems caused by downloading the .project file.  bug id: 17518
if (localFile.getName().equals(".project")) //$NON-NLS-1$
	return;

		// Copy from the remote file to the local file.
		try {
			InputStream source = null;
			try {
				// Get the remote file content.
				source = remoteResource.getContents(progress);
				// Set the local file content to be the same as the remote file.
				if (localFile.exists())
					localFile.setContents(source, false, true, progress);
				else
					localFile.create(source, false, progress);
				// Get the remote base identifier (the last modified time in this instance)  _before_
				// closing the file so that we can ensure it is locked while we determine it.  However,
				// the local timestamp may change on close(), so we have to take the risk.
				remoteBaseIdentifier = getReleasedIdentifier(progress);
			} finally {
				if (source != null)
					source.close();
			}
		} catch (CoreException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		} catch (IOException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
		// Set the base stamp of the local resource.
		localBaseTimestamp = localFile.getModificationStamp();
		// Remember the new base state.
		storeState();
	}

	/**
	 * Copy the local resource up to the remote resource.
	 * 
	 * @param progress a progress monitor to show the progress of the copy, or
	 * <code>null</code> if progress is not required.
	 */
	public void upload(IProgressMonitor progress) throws TeamException {
		// Upload only applies to files as folders are created using mkRemoteDirs()
		if (localResource.getType() != IResource.FILE) return;

		// We know the local resource is a file.
		IFile localFile = (IFile)localResource;
		AbstractResourceHandle davHandle = remoteResource.getDavHandle();
		try {
			// Copy from the local file to the remote resource.
			InputStream in = null;
			try {
				in = localFile.getContents();
				long size = localFile.getLocation().toFile().length();
				
				// To be precise we should set the If-Match: header to ensure we do not overwrite others' changes.
				remoteResource.setContent(size, in, progress);

				// Get the local base timestamp before closing the file so that
				// we can ensure it is locked while we determine it.  However,
				// the remote may change on close(), so we have to take the risk.
				localBaseTimestamp = localFile.getModificationStamp();
			} finally {
				if (in != null)
					in.close();
			}
		} catch (IOException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		} catch (CoreException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		} catch(DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
		
		// Set the remote base modification stamp.
		// It is important that the remote id is determined after the
		// input stream is closed.
		remoteBaseIdentifier = getReleasedIdentifier(progress);
		// Remember the new base state.
		storeState();
	}

	/**
	 * @see ResourceState#delete(IProgressMonitor)
	 */
	public void delete(IProgressMonitor progress) throws TeamException {
		remoteResource.delete(progress);
		removeState();
	}
	
	/**
	 * @see ResourceState#mkRemoteDirs(IProgressMonitor)
	 */
	protected void mkRemoteDirs(IProgressMonitor progress) throws TeamException {
		AbstractResourceHandle davHandle = remoteResource.getDavHandle();
		try {
			if (localResource.getType() == IResource.FILE) {
				davHandle.getParent().mkdirs();
			} else {
				davHandle.asCollectionHandle().mkdirs();
			}
		} catch(DAVException exception) {
			throw TeamWebDavPlugin.wrapException(exception);
		}
		storeState();
	}
}
