/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.core;

import java.net.UnknownHostException;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.ui.TeamUIPlugin;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.webdav.internal.kernel.DAVException;
import org.eclipse.webdav.internal.kernel.SystemException;
import org.eclipse.webdav.internal.kernel.WebDAVException;

/**
 * <code>TeamPlugin</code> is the plug-in runtime class for the Team 
 * resource management plugin.
 * <p>
 * 
 * @see Team
 * @see RepositoryProvider
 * 
 * @since 2.0
 */
final public class TeamWebDavPlugin extends AbstractUIPlugin {

	// The id of the core team plug-in
	public static final String ID = "org.eclipse.team.webdav"; //$NON-NLS-1$

	// The one and only plug-in instance
	private static TeamWebDavPlugin plugin;	

	/** 
	 * Constructs a plug-in runtime class for the given plug-in descriptor.
	 */
	public TeamWebDavPlugin(IPluginDescriptor pluginDescriptor) {
		super(pluginDescriptor);
		plugin = this;
	}
	
	/**
	 * @see Plugin#startup()
	 */
	public void startup() throws CoreException {
		super.startup();
		Policy.localize("org.eclipse.team.internal.webdav.core.messages"); //$NON-NLS-1$
	}
	
	/**
	 * Returns the Team plug-in.
	 *
	 * @return the single instance of this plug-in runtime class
	 */
	public static TeamWebDavPlugin getPlugin() {
		return plugin;
	}

	/*
	 * Static helper methods for creating exceptions
	 */
	public static TeamException wrapException(Exception e) {
		if(e instanceof SystemException)
			return wrapException((SystemException) e);
		if(e instanceof WebDAVException)
			return wrapException((WebDAVException) e);
			
		return new TeamException(new Status(IStatus.ERROR, ID, 0, e.getMessage() != null ? e.getMessage() : "",	e)); //$NON-NLS-1$
	}
	
	public static TeamException wrapException(CoreException e) {
		IStatus status = e.getStatus();
		return new TeamException(new Status(status.getSeverity(), ID, status.getCode(), status.getMessage(), e));
	}

	public static TeamException wrapException(WebDAVException e) {
		int statusCode = e.getStatusCode();
		IStatus status = new Status(IStatus.ERROR, ID, statusCode, Policy.bind("plugin.davStatusError", String.valueOf(statusCode), e.getMessage()), e); //$NON-NLS-1$
		return new TeamException(status);
	}
	
	//Workaround for bug id=17996
	public static TeamException wrapException(SystemException e) {
		Exception we = e.getWrappedException();
		String message;
		if(we instanceof UnknownHostException) {
			message = Policy.bind("TeamWebDavPlugin.unknownHost"); //$NON-NLS-1$
		} else {
			message = we.getMessage();
		}
		return new TeamException(new Status(IStatus.ERROR, ID, 0, message, e));
	}
	
	/**
	 * Log an error
	 */
	public static void logError(String message, Throwable e) {
		plugin.getLog().log(new Status(IStatus.ERROR, ID, 0, message, e));
	}

	/**
	 * Log a problem of given severity
	 */
	public static void log(int severity, String message, Throwable e) {
		plugin.getLog().log(new Status(severity, ID, 0, message, e));
	}
	
	/**
	 * Returns the plug-in's log
	 */
	public static void log(IStatus status) {
		plugin.getLog().log(status);
	}
	
	public static void handle(Throwable t) {
		if (t instanceof DAVException) {
			Shell shell = new Shell(Display.getDefault());
			IStatus error = new Status(IStatus.ERROR, TeamWebDavPlugin.ID, 1, Policy.bind("Internal_error_occured_5"), t); //$NON-NLS-1$
			ErrorDialog.openError(shell, Policy.bind("exception"), null, error); //$NON-NLS-1$
			shell.dispose();
		} else {
			TeamUIPlugin.handle(t);
		}
	}
}
