/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.ui;


import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.team.core.TeamException;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.core.target.TargetManager;
import org.eclipse.team.internal.ui.UIConstants;
import org.eclipse.team.internal.ui.target.ConfigureTargetWizard;
import org.eclipse.team.internal.ui.target.MappingSelectionPage;
import org.eclipse.team.internal.webdav.core.Policy;
import org.eclipse.team.internal.webdav.core.TeamWebDavPlugin;
import org.eclipse.team.ui.IConfigurationWizard;
import org.eclipse.team.ui.TeamImages;
import org.eclipse.ui.IWorkbench;

public class WebDavConfigurationWizard extends Wizard implements IConfigurationWizard {
			
	IProject project;
	WebDavMainPage mainPage = null;
	
	public WebDavConfigurationWizard() {
		// initializes the dialogs settings store
		IDialogSettings workbenchSettings = TeamWebDavPlugin.getPlugin().getDialogSettings();
		IDialogSettings section = workbenchSettings.getSection("WebDavConfigurationWizard");//$NON-NLS-1$
		if (section == null) {
			section = workbenchSettings.addNewSection("WebDavConfigurationWizard");//$NON-NLS-1$
		}
		setDialogSettings(section);
	}
	
	protected String getMainPageTitle() {
		return Policy.bind("webdavMainPage.name"); //$NON-NLS-1$
	}
	
	protected String getMainPageDescription() {
		return Policy.bind("webdavMainPage.description"); //$NON-NLS-1$
	}
	
	public void addPages() {
		mainPage = new WebDavMainPage("webdav-site-creation", getMainPageTitle(), getMainPageDescription(), TeamImages.getImageDescriptor(UIConstants.IMG_WIZBAN_SHARE)); //$NON-NLS-1$
		mainPage.setDialogSettings(getDialogSettings());
		addPage(mainPage);
	}
	
	/**
	 * Finished pressed while on the target pages. Simply create the Site
	 */
	public boolean performFinish() {
		if(project == null) {
			mainPage.finish(null);
			Site site = mainPage.getSite();
			if(ConfigureTargetWizard.validateSite(site, getContainer())) {				
				TargetManager.addSite(site);
			} else {
				try {
					site.dispose();
				} catch (TeamException e) {
					TeamWebDavPlugin.logError(Policy.bind("Site_disposal_error"), e);
				}
				return false;
			}
		}
		return true;
	}
	
	public void init(IWorkbench workbench, IProject project) {
		this.project = project;
	}
	
	/*
	 * Convenience method for running an operation with progress and
	 * error feedback.
	 * 
	 * @param runnable  the runnable to run
	 * @param problemMessage an optional message to display in case of errors
	 */
	protected void run(IRunnableWithProgress runnable, String problemMessage) {
		IStatus errors = null;
		boolean internalError = false;
		try {
			new ProgressMonitorDialog(getShell()).run(false, false, runnable);
		} catch (InvocationTargetException e) {
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				errors = ((CoreException)t).getStatus();
			} else {
				errors = new Status(IStatus.ERROR, TeamWebDavPlugin.ID, 1, problemMessage, t);
				internalError = true;
			}
		} catch (InterruptedException e) {
			errors = null;
		}
		if (errors != null) {
			String msg = internalError
				? Policy.bind("Internal_error_occured_5") //$NON-NLS-1$
				: problemMessage;
			ErrorDialog.openError(getShell(), msg, null, errors);
		}
	}
	
	/**
	 * @see IWizard#canFinish()
	 */
	public boolean canFinish() {
		IWizardPage page = getContainer().getCurrentPage();
		if(page == mainPage && project != null) {
			return false;
		}
		return super.canFinish();
	}
	
	/**
	 * @see IWizard#getNextPage(IWizardPage)
	 */
	public IWizardPage getNextPage(IWizardPage page) {
		if(page == mainPage) {
			Site site = mainPage.getSite();
			if(site != null && project != null) {
				MappingSelectionPage mappingPage = ConfigureTargetWizard.getMappingPage();
				mappingPage.setSite(site);
				mappingPage.setPreviousPage(page);
				return mappingPage;
			}				
		}
		return super.getNextPage(page);
	}
	
	/**
	 * @see org.eclipse.jface.wizard.IWizard#getWindowTitle()
	 */
	public String getWindowTitle() {
		return Policy.bind("webdavMainPage.windowTitle"); //$NON-NLS-1$
	}
}
