/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.team.internal.webdav.ui;


import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.team.internal.core.target.Site;
import org.eclipse.team.internal.core.target.TargetManager;
import org.eclipse.team.internal.ui.target.TargetWizardPage;
import org.eclipse.team.internal.webdav.core.Policy;
import org.eclipse.team.internal.webdav.core.WebDavSite;

/*
 * Wizard page for configuring a project with a WebDav repository location.
 */
public class WebDavMainPage extends TargetWizardPage {
	boolean isCheckout;
	
	// Widgets
	private Combo usernameCombo;
	private Combo hostnameUrlCombo;
	private Text password;
	private Combo proxyCombo;
	private Combo timeoutCombo;
	
	// Dialog store id constants
	private static final String STORE_USERNAME_ID =
		"DavWizardMainPage.STORE_USERNAME_ID";//$NON-NLS-1$
	private static final String STORE_URL_ID =
		"DavWizardMainPage.STORE_HOSTNAME_ID";//$NON-NLS-1$
	private static final String STORE_PROXY_ID =
		"DavWizardMainPage.STORE_PATH_ID";//$NON-NLS-1$
	private static final String STORE_TIMEOUT_ID =
		"DavWizardMainPage.STORE_MODULE_ID";//$NON-NLS-1$
	
	
	// In case the page was launched from a different wizard	
	private IDialogSettings settings;
		
	/*
	 * ConfigurationWizardMainPage constructor.
	 */
	public WebDavMainPage(String pageName, String title, String description, ImageDescriptor titleImage) {
		super(pageName, title, titleImage);
		setDescription(description);
		setTitle(title);
	}	
	
	protected void setDialogSettings(IDialogSettings settings) {
		this.settings = settings;
	}
	
	protected IDialogSettings getDialogSettings() {
		return settings;
	}
	
	/*
	 * Creates the UI part of the page.
	 * 
	 * @param parent  the parent of the created widgets
	 */
	public void createControl(Composite parent) {
		Composite composite = createComposite(parent, 2);

		createLabel(composite, Policy.bind("WebDavWizardMainPage.url")); //$NON-NLS-1$
		hostnameUrlCombo = createEditableCombo(composite);
		hostnameUrlCombo.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				WebDavMainPage.this.validateFields();		
			}
		});

		createLabel(composite, Policy.bind("WebDavWizardMainPage.userName")); //$NON-NLS-1$
		usernameCombo = createEditableCombo(composite);
		
		createLabel(composite, Policy.bind("WebDavWizardMainPage.password")); //$NON-NLS-1$
		password = createTextField(composite);
		password.setEchoChar('*');

		createLabel(composite, Policy.bind("WebDavWizardMainPage.proxy")); //$NON-NLS-1$
		proxyCombo = createEditableCombo(composite);
		proxyCombo.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				WebDavMainPage.this.validateFields();		
			}
		});

		createLabel(composite, Policy.bind("WebDavWizardMainPage.timeout")); //$NON-NLS-1$
		timeoutCombo = createEditableCombo(composite);
		timeoutCombo.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event e) {
				WebDavMainPage.this.validateFields();		
			}
		});

		initializeValues();
		validateFields();
		hostnameUrlCombo.setFocus();
		setControl(composite);
	}
	
	public WebDavSite getSite() {
		if(!isPageComplete()) {
			return null;
		}

		return new WebDavSite(getCurrentValues());
	}

	private Properties getCurrentValues() {
		Properties result = new Properties();
		result.setProperty("location", hostnameUrlCombo.getText()); //$NON-NLS-1$
		result.setProperty("httpClient.username", usernameCombo.getText()); //$NON-NLS-1$
		result.setProperty("httpClient.password", password.getText()); //$NON-NLS-1$
		result.setProperty("httpClient.proxyURL", proxyCombo.getText()); //$NON-NLS-1$
		result.setProperty("httpClient.connectionTimeout", timeoutCombo.getText());  //$NON-NLS-1$
		return result;
	}
			
	/*
	 * @see WizardPage#finish
	 */
	public boolean finish(IProgressMonitor monitor) {
		// Set the result to be the current values
		saveWidgetValues();
		return true;
	}
	
	/*
	 * Initializes states of the controls.
	 */
	private void initializeValues() {
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] hostNames = settings.getArray(STORE_URL_ID);
			if (hostNames != null) {
				for (int i = 0; i < hostNames.length; i++) {
					hostnameUrlCombo.add(hostNames[i]);
				}
			}
			String[] userNames = settings.getArray(STORE_USERNAME_ID);
			if (userNames != null) {
				for (int i = 0; i < userNames.length; i++) {
					usernameCombo.add(userNames[i]);
				}
			}
			String[] proxies = settings.getArray(STORE_PROXY_ID);
			if (proxies != null) {
				for (int i = 0; i < proxies.length; i++) {
					proxyCombo.add(proxies[i]);
				}
			}			
			String[] timeouts = settings.getArray(STORE_TIMEOUT_ID);
			if (timeouts != null) {
				for (int i = 0; i < timeouts.length; i++) {
					timeoutCombo.add(timeouts[i]);
				}
			}
			timeoutCombo.setText(String.valueOf(WebDavSite.DEFAULT_TIMEOUT));
		}
	}
	
	/*
	 * Saves the widget values
	 */
	private void saveWidgetValues() {
		// Update history
		IDialogSettings settings = getDialogSettings();
		if (settings != null) {
			String[] userNames = settings.getArray(STORE_USERNAME_ID);
			if (userNames == null) userNames = new String[0];
			userNames = addToHistory(userNames, usernameCombo.getText());
			settings.put(STORE_USERNAME_ID, userNames);
	
			String[] hostNames = settings.getArray(STORE_URL_ID);
			if (hostNames == null) hostNames = new String[0];
			hostNames = addToHistory(hostNames, hostnameUrlCombo.getText());
			settings.put(STORE_URL_ID, hostNames);
	
			String[] paths = settings.getArray(STORE_PROXY_ID);
			if (paths == null) paths = new String[0];
			paths = addToHistory(paths, proxyCombo.getText());
			settings.put(STORE_PROXY_ID, paths);

			String[] modules = settings.getArray(STORE_TIMEOUT_ID);
			if (modules == null) modules = new String[0];
			modules = addToHistory(modules, timeoutCombo.getText());
			settings.put(STORE_TIMEOUT_ID, modules);	
		}
	}

	/*
	 * Validates the contents of the editable fields and set page completion 
	 * and error messages appropriately.
	 */
	private void validateFields() {
		// Validate the server URL
		String host = hostnameUrlCombo.getText();
		if (host.length() == 0) {
			setPageComplete(false);
			return;
		}
		try {
			URL url = new URL(host);
			Site site = TargetManager.getSite(WebDavSite.TYPE, url);
			if(site != null) {
				setMessage(Policy.bind("WebDAVWizardMainPage.locationAlreadyExists"), WARNING); //$NON-NLS-1$
				setPageComplete(false);
				return;
			}
		} catch(MalformedURLException e) {
			setMessage(Policy.bind("WebDAVWizardMainPage.invalidServerURL"), WARNING); //$NON-NLS-1$
			setPageComplete(false);
			return;
		}

		// Validate the proxy URL.
		String proxy = proxyCombo.getText();
		if (proxy.length() != 0) {
			try {
				new URL(proxy);
			} catch(MalformedURLException e) {
				setMessage(Policy.bind("WebDAVWizardMainPage.invalidProxyURL"), WARNING); //$NON-NLS-1$
				setPageComplete(false);
				return;
			}
		}
		
		// validate the timeout
		String timeout = timeoutCombo.getText();
		try {
			int timeoutInt = Integer.parseInt(timeout);
			if(timeoutInt <= 0) {
				setMessage(Policy.bind("WebDAVWizardMainPage.invalidTimeout"), WARNING); //$NON-NLS-1$
				setPageComplete(false);
				return;		
			}
		} catch(NumberFormatException e) {
			setMessage(Policy.bind("WebDAVWizardMainPage.invalidTimeout"), WARNING); //$NON-NLS-1$
			setPageComplete(false);
			return;		
		}
		
		setMessage(null);
		setPageComplete(true);
	}
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if (visible) {
			hostnameUrlCombo.setFocus();
		}
	}
}
