#include "windows.h"
#include "stdio.h"

#include "PIFormat.h"
#include "PIAbout.h"

#define gResult				(*(globals->result))
#define gParams				(globals->formatParamBlock)
#define gPixelBuffer		(globals->pixelBuffer)
#define gPixelData			(globals->pixelData)

#define PIUnlockHandle(x)		_PIUnlockHandle(gParams->handleProcs, (x))
#define PILockHandle(x)			_PILockHandle(gParams->handleProcs, (x))
#define PIDisposeHandle(x)		_PIDisposeHandle(gParams->handleProcs, (x))
#define PINewHandle(x)			_PINewHandle(gParams->handleProcs, (x))

#define PIAllocateBuffer(x,y)	_PIAllocateBuffer(gParams->bufferProcs, (x), (y))
#define PILockBuffer(x)			_PILockBuffer(gParams->bufferProcs, (x))
#define PIUnlockBuffer(x)		_PIUnlockBuffer(gParams->bufferProcs, (x))
#define PIFreeBuffer(x)			_PIFreeBuffer(gParams->bufferProcs, (x))

#define PIAdvanceState()		gParams->advanceState();

#define DLLExport extern "C" __declspec(dllexport)
extern HANDLE hDllInstance;

typedef struct Globals
{
	short				*result;			// Must always be first in Globals.
	FormatRecord		*formatParamBlock;	// Must always be second in Globals.

	BufferID			pixelBuffer;
	Ptr					pixelData;
	
	int16				leftX, rightX, aboveY, belowY;
} Globals, *GPtr, **GHdl;
typedef void (* FProcP)(Ptr globals);


DLLExport MACPASCAL void PluginMain (const short selector,
					  	             FormatRecord *formatParamBlock,
						             long *data,
						             short *result);

Ptr AllocateGlobals(const uint32 resultAddr,
					const uint32 paramBlockAddr,
					HandleProcs *procs, 
					const size_t size, 
					long *data, 
					FProcP InitGlobals);


void InitGlobals (Ptr globalPtr);
void DoAbout (AboutRecordPtr about);
void ValidateParameters (GPtr globals);


static Handle _PINewHandle(HandleProcs* procs, int32 size);
static void _PIDisposeHandle(HandleProcs* procs, Handle h);
static Ptr _PILockHandle(HandleProcs* procs, Handle h);
static void _PIUnlockHandle(HandleProcs* procs, Handle h);
static OSErr _PIAllocateBuffer(BufferProcs* procs, int32 size, BufferID* buffer);
static Ptr _PILockBuffer(BufferProcs* procs, BufferID buffer);
static void _PIUnlockBuffer(BufferProcs* procs, BufferID buffer);
static void _PIFreeBuffer(BufferProcs* procs, BufferID buffer);

static void DoReadPrepare (GPtr globals);
static void DoReadStart (GPtr globals);
static void DoReadContinue (GPtr globals);
static void DoReadFinish (GPtr globals);
static void DoOptionsPrepare (GPtr globals);
static void DoOptionsStart (GPtr globals);
static void DoOptionsContinue (GPtr globals);
static void DoOptionsFinish (GPtr globals);
static void DoEstimatePrepare (GPtr globals);
static void DoEstimateStart (GPtr globals);
static void DoEstimateContinue (GPtr globals);
static void DoEstimateFinish (GPtr globals);
static void DoWritePrepare (GPtr globals);
static void DoWriteStart (GPtr globals);
static void DoWriteContinue (GPtr globals);
static void DoWriteFinish (GPtr globals);
static void DoFilterFile (GPtr globals);

static Boolean CheckForServices (GPtr globals);
static Boolean AllocatePixelBuffer (GPtr globals);
static void DisposePixelBuffer (GPtr globals);

static void ReadSome (GPtr globals, int32 count, void *buffer);
static void WriteSome (GPtr globals, int32 count, void *buffer);
static void ReadShape (GPtr globals);
static void WriteShape (GPtr globals);
static void DisposeImageResources (GPtr globals);


static const unsigned char redLUT[256] = {
0x00, 0xF8, 0xF4, 0xF0, 0xEC, 0xEC, 0xD8, 0xC4, 
0xB0, 0x9C, 0x88, 0x74, 0x60, 0x4C, 0x38, 0xF8, 
0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xDC, 
0xC0, 0xA4, 0x88, 0x6C, 0x50, 0x34, 0x18, 0xFC, 
0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xE0, 
0xC0, 0xA4, 0x88, 0x6C, 0x50, 0x34, 0x18, 0xFC, 
0xF4, 0xEC, 0xE4, 0xDC, 0xC0, 0xA4, 0x88, 0x6C, 
0x50, 0x34, 0x18, 0xD8, 0xB0, 0x8C, 0x6C, 0x50, 
0x38, 0x28, 0x1C, 0x14, 0x0C, 0x04, 0x00, 0x00, 
0x00, 0xD4, 0xB8, 0x98, 0x7C, 0x5C, 0x3C, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xE8, 0xD4, 0xC4, 0xB0, 0xA0, 0x8C, 0x7C, 0x6C, 
0x60, 0x50, 0x44, 0x34, 0x24, 0x18, 0xF4, 0xEC, 
0xE4, 0xE0, 0xD8, 0xD0, 0xC8, 0xC4, 0xAC, 0x98, 
0x80, 0x6C, 0x54, 0x3C, 0x28, 0x10, 0xEC, 0xDC, 
0xCC, 0xBC, 0xAC, 0x9C, 0x8C, 0x7C, 0x6C, 0x60, 
0x50, 0x44, 0x34, 0x24, 0x18, 0x08, 0xE8, 0xD8, 
0xC8, 0xB8, 0xA8, 0x98, 0x88, 0x7C, 0x6C, 0x5C, 
0x4C, 0x3C, 0x2C, 0x20, 0xEC, 0xDC, 0xCC, 0xBC, 
0xAC, 0x9C, 0x8C, 0x7C, 0x6C, 0x60, 0x50, 0x44, 
0x34, 0x24, 0x18, 0xE0, 0xC8, 0xB4, 0x9C, 0x88, 
0x70, 0x5C, 0x4C, 0x40, 0x38, 0x30, 0x28, 0x20, 
0x18, 0x0C, 0xEC, 0xDC, 0xCC, 0xBC, 0xAC, 0x9C, 
0x8C, 0x7C, 0x6C, 0x60, 0x50, 0x44, 0x34, 0x24, 
0xF0, 0xE4, 0xD8, 0xCC, 0xC0, 0xB4, 0xA8, 0x9C, 
0x84, 0x84, 0x7C, 0x68, 0x50, 0x3C, 0x2C, 0x1C, 
0x20, 0xEC, 0xDC, 0xCC, 0xBC, 0xAC, 0x9C, 0x8C, 
0x7C, 0x6C, 0x60, 0x50, 0x44, 0x34, 0x24, 0x18, 
0xFC, 0xAC, 0x70, 0x34, 0x00, 0x30, 0x6C, 0xB0, 
0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xC8, 0x98, 0x68, 
0x7C, 0x44, 0x18, 0x00, 0xF8, 0xFC, 0xFC, 0xCC, 
0xFC, 0x00, 0x00, 0xFC, 0xFC, 0x61, 0xC0, 0xFC 
};

static const unsigned char greenLUT[256] = {
0x00, 0xF0, 0xE4, 0xDC, 0xD0, 0xC8, 0xAC, 0x94, 
0x80, 0x68, 0x54, 0x44, 0x30, 0x24, 0x14, 0xFC, 
0xD8, 0xB8, 0x98, 0x78, 0x58, 0x38, 0x1C, 0x14, 
0x0C, 0x08, 0x04, 0x00, 0x00, 0x00, 0x00, 0xEC, 
0xDC, 0xCC, 0xBC, 0xAC, 0x9C, 0x8C, 0x7C, 0x6C, 
0x60, 0x50, 0x44, 0x34, 0x24, 0x18, 0x08, 0xFC, 
0xF4, 0xEC, 0xE4, 0xDC, 0xC0, 0xA4, 0x88, 0x6C, 
0x50, 0x34, 0x18, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 
0xFC, 0xDC, 0xC0, 0xA4, 0x88, 0x6C, 0x50, 0x34, 
0x18, 0xD8, 0xB8, 0x98, 0x7C, 0x5C, 0x3C, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xC8, 0x98, 0x6C, 0x48, 0x28, 0x10, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE8, 0xDC, 
0xCC, 0xC0, 0xB0, 0xA4, 0x98, 0x8C, 0x7C, 0x6C, 
0x5C, 0x4C, 0x3C, 0x2C, 0x1C, 0x0C, 0xEC, 0xDC, 
0xCC, 0xBC, 0xAC, 0x9C, 0x8C, 0x7C, 0x6C, 0x60, 
0x50, 0x44, 0x34, 0x24, 0x18, 0x08, 0xE0, 0xC8, 
0xB0, 0x98, 0x84, 0x70, 0x5C, 0x4C, 0x3C, 0x34, 
0x2C, 0x24, 0x1C, 0x14, 0xE8, 0xD4, 0xC4, 0xB0, 
0xA0, 0x90, 0x80, 0x70, 0x60, 0x54, 0x48, 0x3C, 
0x30, 0x20, 0x14, 0xE8, 0xD4, 0xC0, 0xAC, 0x98, 
0x84, 0x70, 0x5C, 0x50, 0x44, 0x3C, 0x30, 0x24, 
0x1C, 0x10, 0xD8, 0xB8, 0x98, 0x80, 0x64, 0x50, 
0x3C, 0x2C, 0x24, 0x20, 0x1C, 0x14, 0x10, 0x0C, 
0xF0, 0xE4, 0xD8, 0xCC, 0xC0, 0xB4, 0xA8, 0x9C, 
0xD0, 0xB0, 0x94, 0x78, 0x58, 0x40, 0x24, 0x08, 
0x00, 0xD8, 0xC0, 0xB4, 0x9C, 0x90, 0x84, 0x74, 
0x64, 0x54, 0x48, 0x40, 0x34, 0x2C, 0x18, 0x10, 
0xF8, 0xD4, 0xAC, 0x8C, 0x6C, 0x8C, 0xB0, 0xD4, 
0xFC, 0xEC, 0xC0, 0x8C, 0x50, 0x38, 0x28, 0x18, 
0xDC, 0xB4, 0x90, 0x6C, 0xB8, 0x64, 0x00, 0x00, 
0xFC, 0x00, 0xFC, 0x00, 0xFC, 0x61, 0xC0, 0x00
};

static const unsigned char blueLUT[256] = {
0x00, 0xCC, 0xA4, 0x78, 0x50, 0x28, 0x20, 0x18, 
0x10, 0x0C, 0x08, 0x04, 0x00, 0x00, 0x00, 0xFC, 
0xD8, 0xB8, 0x9C, 0x80, 0x64, 0x4C, 0x34, 0x28, 
0x1C, 0x14, 0x0C, 0x04, 0x00, 0x00, 0x00, 0xD8, 
0xB8, 0x98, 0x7C, 0x5C, 0x3C, 0x1C, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD8, 
0x9C, 0x60, 0x2C, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xD8, 0xAC, 0x80, 0x54, 0x28, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 
0xE0, 0xC0, 0xA4, 0x88, 0x6C, 0x50, 0x34, 0x18, 
0xE8, 0xD4, 0xC4, 0xB0, 0xA0, 0x8C, 0x7C, 0x6C, 
0x60, 0x50, 0x44, 0x34, 0x24, 0x18, 0xE4, 0xD4, 
0xC0, 0xB0, 0xA0, 0x90, 0x80, 0x74, 0x64, 0x58, 
0x4C, 0x3C, 0x30, 0x24, 0x14, 0x08, 0xEC, 0xDC, 
0xCC, 0xBC, 0xAC, 0x9C, 0x8C, 0x7C, 0x6C, 0x60, 
0x50, 0x44, 0x34, 0x24, 0x18, 0x08, 0xD4, 0xB0, 
0x90, 0x70, 0x58, 0x40, 0x2C, 0x18, 0x0C, 0x0C, 
0x0C, 0x0C, 0x08, 0x08, 0xE4, 0xD0, 0xBC, 0xAC, 
0x98, 0x88, 0x78, 0x68, 0x5C, 0x50, 0x44, 0x38, 
0x2C, 0x20, 0x14, 0xD4, 0xB4, 0x98, 0x7C, 0x60, 
0x4C, 0x38, 0x28, 0x20, 0x1C, 0x18, 0x14, 0x10, 
0x08, 0x04, 0xCC, 0xA0, 0x7C, 0x5C, 0x3C, 0x24, 
0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xC4, 0xB4, 0xA0, 0x94, 0x80, 0x74, 0x64, 
0x58, 0x4C, 0x44, 0x38, 0x2C, 0x24, 0x18, 0x10, 
0xFC, 0xF0, 0xE4, 0xD8, 0xD0, 0xD8, 0xE4, 0xF0, 
0xF8, 0x40, 0x28, 0x10, 0x00, 0x00, 0x00, 0x00, 
0x7C, 0x44, 0x18, 0x00, 0xFC, 0xEC, 0xB4, 0x70, 
0x00, 0xff, 0x00, 0x00, 0xFC, 0x61, 0xC0, 0xF1
};

HANDLE hDllInstance = NULL;



// entrypoint
DLLExport MACPASCAL void PluginMain (const short selector,
						             FormatRecord *formatParamBlock,
						             long *data,
						             short *result)
{
	
	if (selector == formatSelectorAbout) {
		DoAbout((AboutRecordPtr)formatParamBlock);
	} else {
	 	Ptr globalPtr = NULL;
		GPtr globals = NULL;

		globalPtr = AllocateGlobals ((uint32)result,
									 (uint32)formatParamBlock,
									 formatParamBlock->handleProcs,
									 sizeof(Globals),
						 			 data,
						 			 InitGlobals);
		
		if (globalPtr == NULL) {
		  *result = memFullErr;
		  return;
		}
		globals = (GPtr)globalPtr;

		switch (selector) {
		case formatSelectorAbout: break;
		case formatSelectorReadPrepare:		DoReadPrepare(globals); break;
		case formatSelectorReadStart:		DoReadStart(globals); break;
		case formatSelectorReadContinue:	DoReadContinue(globals); break;
		case formatSelectorReadFinish:		DoReadFinish(globals); break;
			
		case formatSelectorOptionsPrepare:	DoOptionsPrepare(globals); break;
		case formatSelectorOptionsStart:	DoOptionsStart(globals); break;
		case formatSelectorOptionsContinue:	DoOptionsContinue(globals); break;
		case formatSelectorOptionsFinish:	DoOptionsFinish(globals); break;
			
		case formatSelectorEstimatePrepare:	DoEstimatePrepare(globals); break;
		case formatSelectorEstimateStart:	DoEstimateStart(globals); break;
		case formatSelectorEstimateContinue:DoEstimateContinue(globals); break;
		case formatSelectorEstimateFinish:	DoEstimateFinish(globals); break;
			
		case formatSelectorWritePrepare:	DoWritePrepare(globals); break;
		case formatSelectorWriteStart:		DoWriteStart(globals); break;
		case formatSelectorWriteContinue:	DoWriteContinue(globals); break;
		case formatSelectorWriteFinish:		DoWriteFinish(globals); break;
			
		case formatSelectorFilterFile:		DoFilterFile(globals); break;
		default:
			gResult = formatBadParameters;
		}

		if ((Handle)*data != NULL)
			PIUnlockHandle((Handle)*data);
			
	}
} // PluginMain


void InitGlobals (Ptr globalPtr)
{	
	GPtr globals = (GPtr)globalPtr;
	
	globals->pixelBuffer = NULL;
	globals->pixelData = NULL;
	globals->leftX = 0;
	globals->rightX = 0;
	globals->aboveY = 0;
	globals->belowY = 0;
} // InitGlobals


void DoAbout(AboutRecordPtr about)
{
	//TODO: About box!
}

Ptr AllocateGlobals(const uint32 resultAddr,
					const uint32 paramBlockAddr,
					HandleProcs *procs,
					const size_t size,
					long *data, 
					FProcP InitGlobals)

{
	Ptr globalPtr = NULL;
	
	if (!*data) {
		// Data is empty, so initialize our globals
		Handle h = _PINewHandle(procs, size);
		
		if (h != NULL) {
			globalPtr = _PILockHandle(procs, h);
			
			if (globalPtr != NULL) {
				uint32 *result = (uint32 *)globalPtr;
				uint32 *address = (uint32 *)(globalPtr + sizeof(short *));
				
				*result = resultAddr;
				*((short *)resultAddr) = noErr;
				
				*address = paramBlockAddr;
				
				InitGlobals (globalPtr);
				
				*data = (long)h;
				h = NULL;
			} else {
				_PIDisposeHandle(procs, h);
				h = NULL;
			}
		}		
	} else {
		// we've already got a valid structure pointed to by *data
		globalPtr = _PILockHandle(procs, (Handle)*data);

		if (globalPtr != NULL) {
			uint32 *result = (uint32 *)globalPtr;
			uint32 *address = (uint32 *)(globalPtr + sizeof(short *));
			
			*result = resultAddr;
			*((short *)resultAddr) = noErr;
			
			*address = paramBlockAddr;
		}
	}
	
	return globalPtr;
}


#define TSR(x) TestAndStoreResult(&gResult, (x))
Boolean TestAndStoreResult (short *res, OSErr result)
{
	if (result != noErr)
		{
		if (*res == noErr)
			*res = result;
		return FALSE;
		}
	else
		return TRUE;	
}
	
#define TSC(x) TestAndStoreCancel(&gResult, (x))
Boolean TestAndStoreCancel (short *res, Boolean tocancel)
{	
	if (tocancel)
		{
		if (*res == noErr)
			*res = userCanceledErr;
		return FALSE;
		}
	else
		return TRUE;	
}


static Handle _PINewHandle(HandleProcs* procs, int32 size)
{
	if (procs && procs->handleProcsVersion == kPIHandleSuiteVersion &&
			procs->numHandleProcs >= 1)
		return procs->newProc(size);
	else
		return 0;
}

static void _PIDisposeHandle(HandleProcs* procs, Handle h)
{
	if (procs && procs->handleProcsVersion == kPIHandleSuiteVersion &&
			procs->numHandleProcs >= 2)
		procs->disposeProc(h);
}

static Ptr _PILockHandle(HandleProcs* procs, Handle h)
{
	if (procs && procs->handleProcsVersion == kPIHandleSuiteVersion &&
			procs->numHandleProcs >= 5)
		return procs->lockProc(h, FALSE);
	else
		return 0;
}

static void _PIUnlockHandle(HandleProcs* procs, Handle h)
{
	if (procs && procs->handleProcsVersion == kPIHandleSuiteVersion &&
			procs->numHandleProcs >= 6)
		procs->unlockProc(h);
}

static OSErr _PIAllocateBuffer(BufferProcs* procs, int32 size, BufferID* buffer)
{
	if (procs && procs->bufferProcsVersion == kPIBufferSuiteVersion &&
			procs->numBufferProcs >= 1)
		return procs->allocateProc(size, buffer);
	else
		return memFullErr;
}

static Ptr _PILockBuffer(BufferProcs* procs, BufferID buffer)
{
	if (procs && procs->bufferProcsVersion == kPIBufferSuiteVersion &&
			procs->numBufferProcs >= 2)
		return procs->lockProc(buffer, FALSE);
	else
		return 0;
}

static void _PIUnlockBuffer(BufferProcs* procs, BufferID buffer)
{
	if (procs && procs->bufferProcsVersion == kPIBufferSuiteVersion &&
			procs->numBufferProcs >= 3)
		procs->unlockProc(buffer);
}

static void _PIFreeBuffer(BufferProcs* procs, BufferID buffer)
{
	if (procs && procs->bufferProcsVersion == kPIBufferSuiteVersion &&
			procs->numBufferProcs >= 4)
		procs->freeProc(buffer);
}
	
static Boolean CheckForServices (GPtr globals)
{
	return (gParams->bufferProcs && 
			gParams->bufferProcs->bufferProcsVersion == kPIBufferSuiteVersion &&
			gParams->bufferProcs->numBufferProcs >= 4 &&
		gParams->handleProcs &&
			gParams->handleProcs->handleProcsVersion == kPIHandleSuiteVersion &&
			gParams->handleProcs->numHandleProcs >= 6);
}


static Boolean AllocatePixelBuffer (GPtr globals)
{
	
	BufferID buffer;
	
	if (gResult != noErr) return FALSE;

	gPixelBuffer = 0;

	if (!TSR (PIAllocateBuffer (gParams->imageSize.v * gParams->imageSize.h, &buffer))) return FALSE;
	
	gPixelBuffer = buffer;
	gPixelData = PILockBuffer (gPixelBuffer);
	
	return TRUE;
}


static void DisposePixelBuffer (GPtr globals)
{
	if (gPixelBuffer != 0) {
		PIFreeBuffer (gPixelBuffer);
		gPixelBuffer = 0;
		gPixelData = 0;
	}
	
}

OSErr FSWrite(int32 refNum, long *count,void *buffPtr)
{
	/* Note:  this routine doesn't work for data larger than 64k. */

	WORD bytes;

	bytes = (WORD)*count;
	if ((*count = _lwrite((int)refNum,(LPSTR)buffPtr,bytes))==0xffff)
		return writErr;

	return noErr;
}

OSErr SetFPos (int32 refNum, short posMode, long posOff)
{
	_llseek((int)refNum, posOff, posMode);
	return noErr;
}

OSErr FSRead(int32 refNum, long *count	, void *buffPtr)
{
	WORD bytes;

	bytes = (WORD) *count;

	if ((*count = _lread((int)refNum,(LPSTR)buffPtr,bytes))==0xffff)
		return readErr;

	return noErr;
}


static void ReadSome (GPtr globals, int32 count, void *buffer)
{
	int32 readCount = count;
	
	if (gResult != noErr)
		return;
	
	gResult = FSRead (gParams->dataFork, &readCount, buffer);
	
	if (gResult == noErr && readCount != count)
		gResult = eofErr;
}


static void WriteSome (GPtr globals, int32 count, void *buffer)
{
	int32 writeCount = count;
	
	if (gResult != noErr)
		return;
	
	gResult = FSWrite (gParams->dataFork, &writeCount, buffer);
	
	if (gResult == noErr && writeCount != count)
		gResult = dskFulErr;
}


static unsigned int read1(GPtr globals)
{
	unsigned char b0;
	ReadSome(globals, 1, &b0);
	return b0;
}

static unsigned int read2(GPtr globals)
{
	unsigned char b0, b1;
	ReadSome(globals, 1, &b0);
	ReadSome(globals, 1, &b1);
	return (b0 + (b1<<8));
}

static signed int read2signed(GPtr globals)
{
	unsigned char b0, b1;
	signed int i0;
	ReadSome(globals, 1, &b0);
	ReadSome(globals, 1, &b1);
	i0 = b0 + (b1<<8);
	if (i0 >= 32768) i0 -= 65536;
	return i0;
}

static unsigned int read4(GPtr globals)
{
	unsigned char b0, b1, b2, b3;
	ReadSome(globals, 1, &b0);
	ReadSome(globals, 1, &b1);
	ReadSome(globals, 1, &b2);
	ReadSome(globals, 1, &b3);
	return (b0 + (b1<<8) + (b2<<16) + (b3<<24));
}


static void write1(GPtr globals, unsigned char b)
{
	WriteSome(globals, 1, &b);
}

static void write2(GPtr globals, unsigned int b)
{
	unsigned char b0,b1;
	b0 = b & 0xFF;
	b1 = (b >> 8) & 0xFF;
	WriteSome(globals, 1, &b0);
	WriteSome(globals, 1, &b1);
}

static void write4(GPtr globals, unsigned int b)
{
	unsigned char b0,b1,b2,b3;
	b0 = b & 0xFF;
	b1 = (b >> 8) & 0xFF;
	b2 = (b >> 16) & 0xFF;
	b3 = (b >> 24) & 0xFF;
	WriteSome(globals, 1, &b0);
	WriteSome(globals, 1, &b1);
	WriteSome(globals, 1, &b2);
	WriteSome(globals, 1, &b3);
}

static unsigned char *out1(unsigned char *p, unsigned char b)
{
    *p++ = b;
    return p;
}

static unsigned char *out2(unsigned char *p, unsigned int b)
{
    *p++ = b & 0xFF;
    *p++ = (b >> 8) & 0xFF;
    return p;
}

static unsigned char *out4(unsigned char *p, unsigned int b)
{
    *p++ = b & 0xFF;
    *p++ = (b >> 8) & 0xFF;
    *p++ = (b >> 16) & 0xFF;
    *p++ = (b >> 24) & 0xFF;
    return p;
}

static unsigned char *res_out1(unsigned char *p, unsigned char b)
{
	*p++ = b;
	return p;
}

static unsigned char *res_out2(unsigned char *p, unsigned int b)
{
	*p++ = (b >> 8) & 0xFF;
	*p++ = b & 0xFF;
	return p;
}

static unsigned char *res_out4(unsigned char *p, unsigned int b)
{
	*p++ = (b >> 24) & 0xFF;
	*p++ = (b >> 16) & 0xFF;
	*p++ = (b >> 8) & 0xFF;
	*p++ = b & 0xFF;
	return p;
}

static unsigned char *res_in1(unsigned char *p, unsigned char &t)
{
	t = *p++;
	return p;
}

static unsigned char *res_in2(unsigned char *p, unsigned int &t)
{
	t = *p++;
	t <<= 8; t += *p++;
	return p;
}

static unsigned char *res_in4(unsigned char *p, unsigned int &t)
{
	t = *p++;
	t <<= 8; t+= *p++;
	t <<= 8; t+= *p++;
	t <<= 8; t+= *p++;
	return p;
}

static void ReadShape (GPtr globals)
{
	int16 slice, slice_type, slice_length;
	int16 offsetX, offsetY;
	int16 block_type, block_length;
	unsigned char block, pix;
	int width, height;
	int temp_int;
	int j;

	unsigned char *pixptr, *eod, *pixeldata;

	width = gParams->imageSize.h;
	height = gParams->imageSize.v;

	pixeldata = (unsigned char*)gPixelData;

	for (int i=0; i < width * height; i++)
		pixeldata[i] = 255; //set everything transparent

	pixptr = pixeldata;
	eod = pixeldata + width*height;

	while((slice=read2(globals))!=0) {
		slice_type = slice & 0x1;
		slice_length = slice >> 1;

		offsetX = read2signed(globals);
		offsetY = read2signed(globals);

        temp_int = (globals->aboveY + offsetY)*width +
                                           (globals->leftX + offsetX);

        pixptr = pixeldata;
        pixptr = pixptr + temp_int;

		if(pixptr<pixeldata)
			pixptr = pixeldata;
		if(slice_type) {	// Compressed
			while(slice_length>0) {
				block = read1(globals);
				block_type = block & 0x1;
				block_length = block >> 1;
				if(block_type) {
					pix = read1(globals);
					for(j=0;j<block_length;j++) {
						*pixptr++ = pix;
					}
				} else {
					for(j=0;j<block_length;j++) {
						pix = read1(globals);
						*pixptr++ = pix;
					}
				}
				slice_length -= block_length;
			}
		} else {		// Uncompressed
			// Just read the pixels
			for(j=0;j<slice_length;j++) {
				pix = read1(globals);
				*pixptr++ = pix;
			}
		}
	}
}

static int find_runs(short *runs, unsigned char *pixptr, int x,	int w, 
					 unsigned char transindex)
{
    int runcnt = 0;
    while (x < w && pixptr[0] != transindex) {	// Stop at first transparent pixel.
		int run = 0;		// Look for repeat.
		while (x < w - 1 && pixptr[0] == pixptr[1]) {
		    x++;
			pixptr++;
			run++;
		}
		if (run) {		// Repeated?  Count 1st, shift, flag.
		    run = ((run + 1)<<1)|1;
			x++;		// Also pass the last one.
			pixptr++;
		} else {
			do {			/* Pass non-repeated run of */
				x++;
				pixptr++;
				run += 2;	// So we don't have to shift.
			} while (x < w && pixptr[0] != transindex &&
				(x == w - 1 || pixptr[0] != pixptr[1]));
		}
		// Store run length.
		runs[runcnt++] = run;
	}
    runs[runcnt] = 0;		/* 0-delimit list. */
    return x;
}


static int skip_transparent(unsigned char **pixptr, int x, int w,
							unsigned char transindex)
{
    unsigned char *pixel = *pixptr;
    while (x < w && pixel[0] == transindex) {
		x++;
		pixel++;
    }
    *pixptr = pixel;
    return x;
}

static void WriteShape (GPtr globals)
{
	int32 shape_size;
	int32 hdr_size;
	int32 width;
	int32 height;
	unsigned char *pixptr;
	unsigned char *pix;
	unsigned char *out;
	unsigned char *outptr;
	int i, j;
	int newx, x, y, hotx, hoty;
	int16 leftX, rightX, aboveY, belowY;
	int datalen;
	unsigned char transindex;
	
	hotx = 0; hoty = 0;		// set to (0,0) in case there are no guides

	if (gParams->imageRsrcSize && gParams->imageRsrcData) {
		unsigned char *rsrc = (unsigned char*)PILockHandle(gParams->imageRsrcData);

		int totsize = 0;
		unsigned int restype;
		unsigned int ressize;
		unsigned int nguides;
		unsigned int guidecoord;
		unsigned char guidedir;
		unsigned int dummy;
		unsigned char strlength;
		unsigned char dummy2;

		while (totsize < gParams->imageRsrcSize) {
			rsrc = res_in4(rsrc, dummy);		// '8BIM'
			rsrc = res_in2(rsrc, restype);		// resource type
			totsize += 6;
			rsrc = res_in1(rsrc, strlength);    // string length
			totsize += 1;
			if (strlength % 2 == 0) strlength++;// pad to even length
			for (i = 0; i < strlength; i++) {
				rsrc = res_in1(rsrc, dummy2);   // skip name string
				totsize += 1;
			}
			rsrc = res_in4(rsrc, ressize);		// resource size
			totsize += 4;	
			if (restype == 0x0408) {
				rsrc = res_in4(rsrc, dummy);	// version
				rsrc = res_in4(rsrc, dummy);	// grid coords
				rsrc = res_in4(rsrc, dummy);
				rsrc = res_in4(rsrc, nguides);	// nr. of guides
				totsize += 16;
				for (unsigned int i=0; i < nguides; i++) {
					rsrc = res_in4(rsrc, guidecoord);
					rsrc = res_in1(rsrc, guidedir);
					totsize += 5;

					if (guidedir == 0)
						hotx = guidecoord >> 5;
					else
						hoty = guidecoord >> 5;
				}
			} else {
				rsrc += ressize + (ressize & 1 ? 1 : 0);
				totsize += ressize + (ressize & 1 ? 1 : 0);
			}
		}

		PIUnlockHandle(gParams->imageRsrcData);
	}
	
	transindex = 255;
	if (gParams->transparentIndex >= 0 && gParams->transparentIndex <= 255)
		transindex = (unsigned char)(gParams->transparentIndex);

	width = gParams->imageSize.h;
	height = gParams->imageSize.v;

	hdr_size = 8;
	
	leftX = hotx;
	aboveY = hoty;
	rightX = width - leftX - 1;
	belowY = height - aboveY - 1;
	
	pix = (unsigned char*)gPixelData;
	pixptr = pix;

	BufferID buffer;
	PIAllocateBuffer (width*height*8, &buffer);
	out = (unsigned char*)PILockBuffer (buffer);
	outptr = out;

	newx = 0;
	for (y = 0; y < height; y++) {
		gParams->theRect.top = y;
		gParams->theRect.bottom = y+1;
		PIAdvanceState();
		pix = (unsigned char*)gPixelData;
		pixptr = pix;

	    for (x = 0; (x = skip_transparent(&pixptr,x,width,transindex)) < width;
																		x = newx) {
			short runs[100];
			newx = find_runs(runs, pixptr, x, width, transindex);
			if (!runs[1] && !(runs[0]&1)) {
			    int len = runs[0] >> 1;
				outptr = out2(outptr, runs[0]);
				outptr = out2(outptr, x - leftX);
				outptr = out2(outptr, y - aboveY);
				for(i=0; i<len; i++)
					*outptr++ = *pixptr++;
			} else {
			    outptr = out2(outptr, ((newx - x)<<1)|1);
				outptr = out2(outptr, x - leftX);
				outptr = out2(outptr, y - aboveY);
				for (i = 0; runs[i]; i++) {
					int len = runs[i]>>1;
					if (runs[i]&1) {
						while (len) {
							int c = len > 127 ? 127 : len;
							*outptr++ = (c<<1)|1;
							*outptr++ = *pixptr;
							pixptr += c;
							len -= c;
						}
					} else while (len > 0) {
						int c;
						c = len > 127 ? 127 : len;
						*outptr++ = c<<1;
						for(j=0; j<c; j++)
							*outptr++ = *pixptr++;
						len -= c;
					}
				}
			}
	    }
	}

	outptr = out2(outptr, 0);			// End with 0 length.
	datalen = outptr - out;

	shape_size = datalen + hdr_size + 8;
	write4(globals, shape_size);
	write4(globals, hdr_size);
	
	write2(globals, rightX);
	write2(globals, leftX);
	write2(globals, aboveY);
	write2(globals, belowY);
	WriteSome(globals, datalen, out);

	PIUnlockBuffer(buffer);
	PIFreeBuffer(buffer);
}


static void DisposeImageResources (GPtr globals)
	{
	
	if (gParams->imageRsrcData) {
		PIDisposeHandle (gParams->imageRsrcData);
		gParams->imageRsrcData = NULL;
		gParams->imageRsrcSize = 0;
	}
	
}

static void DoReadPrepare (GPtr globals)
{
	gParams->maxData = 65536;
}

static void DoReadStart (GPtr globals)
	{
	//TODO: 8x8 tiles!!!

	int32 filelength;
	int32 headersize;
	int16 rightX, leftX, aboveY, belowY;
	int16 height, width;

    if (gResult != noErr) return;
		
	if (!TSC ((Boolean)(!CheckForServices (globals)))) return;

	if (!TSR (SetFPos (gParams->dataFork, 0 /* fsFromStart */, 0))) return;

	filelength = read4(globals);
	headersize = read4(globals);

	if (headersize > 8)
		MessageBox(NULL, 
			"This shape contains multiple frames. Only the first will be read.",
			"Warning", MB_OK);
		
	if (gResult != noErr) return;

	if (!TSR (SetFPos (gParams->dataFork, 0 /* fsFromStart */, headersize))) return;

	rightX = read2(globals);
	leftX = read2(globals);
	aboveY = read2(globals);
	belowY = read2(globals);
	
	height = aboveY + belowY + 1;
	width = leftX + rightX + 1;

	globals->leftX = leftX;
	globals->rightX = rightX;
	globals->aboveY = aboveY;
	globals->belowY = belowY;

	gParams->imageMode = plugInModeIndexedColor;
	gParams->imageSize.h = width;
	gParams->imageSize.v = height;
	gParams->depth = 8;
	gParams->planes = 1;
	gParams->transparentIndex = 255;	// doesn't seem to work in PSP7 :-(

	Handle x = PINewHandle(39);

	unsigned char* rsrc = (unsigned char*)PILockHandle(x);

	rsrc = res_out4(rsrc, '8BIM');
	rsrc = res_out2(rsrc, 0x0408);				// type of resource	
	rsrc = res_out2(rsrc, 0);
	rsrc = res_out4(rsrc, 26);					// data size
	rsrc = res_out4(rsrc, 1);
	rsrc = res_out4(rsrc, 0x0240);				// hor. & ver. resolutions
	rsrc = res_out4(rsrc, 0x0240);
	rsrc = res_out4(rsrc, 2);					// nr. of guides
	rsrc = res_out4(rsrc, 32 * leftX + 16);		// coords
	rsrc = res_out1(rsrc, 0);					// vertical
	rsrc = res_out4(rsrc, 32 * aboveY + 16);	// coords
	rsrc = res_out1(rsrc, 1);					// horizontal
	rsrc = res_out1(rsrc, 0);					// terminator
	
	PIUnlockHandle(x);

	gParams->imageRsrcData = x;
	gParams->imageRsrcSize = 39;

	for (int i=0; i < 256; i++) {
		gParams->redLUT[i] = redLUT[i];
		gParams->greenLUT[i] = greenLUT[i];
		gParams->blueLUT[i] = blueLUT[i];
	}
}


static void DoReadContinue (GPtr globals)
{
	DisposeImageResources (globals);
	
	AllocatePixelBuffer (globals);
	
	gParams->theRect.left = 0;
	gParams->theRect.right = gParams->imageSize.h;
	gParams->colBytes = 1;
	gParams->rowBytes = gParams->imageSize.h;
	gParams->planeBytes = 0;
	gParams->data = gPixelData;

	gParams->loPlane = gParams->hiPlane = 0;
		
	gParams->theRect.top = 0;
	gParams->theRect.bottom = gParams->imageSize.v;
			
	ReadShape (globals);
			
	if (gResult == noErr)
		gResult = PIAdvanceState();
				
	gParams->data = NULL;
	
	DisposePixelBuffer (globals);
}


static void DoReadFinish (GPtr globals)
{
	DisposeImageResources (globals);
}

static void DoOptionsPrepare (GPtr globals)
{
	gParams->maxData = 0;
}

static void DoOptionsStart (GPtr globals)
{
	gParams->data = NULL;
}


static void DoOptionsContinue (GPtr globals)
{

}


static void DoOptionsFinish (GPtr globals)
{

}


static void DoEstimatePrepare (GPtr globals)
{
	gParams->maxData = 0;
}


static void DoEstimateStart (GPtr globals)
{
	
	int32 dataBytes;
	
	dataBytes = 16 + gParams->imageSize.h * gParams->imageSize.v;
					  
	gParams->minDataBytes = dataBytes;
	gParams->maxDataBytes = dataBytes;
	
	gParams->data = NULL;
}


static void DoEstimateContinue (GPtr globals)
{

}


static void DoEstimateFinish (GPtr globals)
{

}


static void DoWritePrepare (GPtr globals)
{
	gParams->maxData = 0;	
}


static void DoWriteStart (GPtr globals)
{
	AllocatePixelBuffer (globals);
		
	gParams->theRect.left = 0;
	gParams->theRect.right = gParams->imageSize.h;
	gParams->colBytes = 1;
	gParams->rowBytes = gParams->imageSize.h;
	gParams->planeBytes = 0;
	gParams->data = gPixelData;
	gParams->loPlane = gParams->hiPlane = 0;
	
	WriteShape (globals);
			
	gParams->data = NULL;
	
	DisposePixelBuffer (globals);
}


static void DoWriteContinue (GPtr globals)
{

}


static void DoWriteFinish (GPtr globals)
{

}


static void DoFilterFile (GPtr globals)
{
	// TODO: check some stuff
	gResult == noErr;
}