#include <iostream>
#include <algorithm>
#include <string>
#include <SDL.h>

#include "controls.hpp"
#include "filesystem.hpp"
#include "formatter.hpp"
#include "preferences.hpp"
#include "sound.hpp"
#include "wml_node.hpp"
#include <time.h>

#include "wml_parser.hpp"
#include "wml_utils.hpp"
#include "wml_writer.hpp"

namespace preferences {
	const std::string& version() {
		static const std::string Version = "1.0.3";
		return Version;
	}

	namespace {
		int unique_user_id = 0;

		int screen_editor_mode = 0;

		bool no_sound_ = false;
		bool no_music_ = false;
		bool show_debug_hitboxes_ = false;
		bool use_pretty_scaling_ = false;
		bool fullscreen_ = false;
		bool resizable_ = false;
		bool debug_ = true;
		bool reverse_ab_ = false;

		std::string level_path_ = "data/level/";

		bool send_stats_ = true;
		
#if TARGET_OS_IPHONE || TARGET_IPHONE_SIMULATOR
		
#ifndef PREFERENCES_PATH
#define PREFERENCES_PATH "../Documents/"
#endif

		bool sim_iphone_ = true;

		int virtual_screen_width_ = 960;
		int virtual_screen_height_ = 640;
		
		int actual_screen_width_ = 320;
		int actual_screen_height_ = 480;

		bool screen_rotated_ = true;

		bool show_fps_ = false;
		bool use_joystick_ = false;
		
		bool load_compiled_ = true;

		bool use_16bpp_textures_ = true;
#else
		bool sim_iphone_ = false;

#ifndef PREFERENCES_PATH
#define PREFERENCES_PATH "~/.frogatto/"
#endif
		int virtual_screen_width_ = 800;
		int virtual_screen_height_ = 600;
		
		int actual_screen_width_ = 800;
		int actual_screen_height_ = 600;
		
		bool screen_rotated_ = false;

		bool show_fps_ = false;
		bool use_joystick_ = true;

		bool load_compiled_ = false;

		bool use_16bpp_textures_ = false;
#endif

#define SAVE_FILENAME					"save.cfg"
#define AUTOSAVE_FILENAME				"autosave.cfg"

		std::string preferences_path_ = PREFERENCES_PATH;
		std::string save_file_path_ = PREFERENCES_PATH SAVE_FILENAME;
		std::string auto_save_file_path_ = PREFERENCES_PATH AUTOSAVE_FILENAME;
		
		bool force_no_npot_textures_ = false;
	}

	int get_unique_user_id() {
		if(unique_user_id == 0) {
			time_t t1;
			time(&t1);
			int tm = t1;
			unique_user_id = tm^rand();
		}

		return unique_user_id;
	}

	int xypos_draw_mask = actual_screen_width_ < virtual_screen_width_ ? ~1 : ~0;
	bool double_scale() {
		return xypos_draw_mask&1 != 0;
	}
	bool compiling_tiles = false;

	namespace {
	void recalculate_draw_mask() {
		xypos_draw_mask = actual_screen_width_ < virtual_screen_width_ ? ~1 : ~0;
	}
	}
	
	bool no_sound() {
		return no_sound_;
	}

	bool no_music() {
		return no_music_;
	}

	bool setup_preferences_dir()
	{
		return !sys::get_dir(user_data_path()).empty();
	}

	void set_preferences_path(const std::string& path)
	{
		preferences_path_ = path;
		if(preferences_path_[preferences_path_.length()-1] != '/') {
			preferences_path_ += '/';
		}

		save_file_path_ = preferences_path_ + SAVE_FILENAME;
		auto_save_file_path_ = preferences_path_ + AUTOSAVE_FILENAME;	
	}

	const std::string& level_path() {
		return level_path_;
	}
	
	const char *save_file_path() {
		return save_file_path_.c_str();
	}

	const char *auto_save_file_path() {
		return auto_save_file_path_.c_str();
	}

	const char *user_data_path() {
		return preferences_path_.c_str();
	}

	namespace {
	void expand_path(std::string& str) {
		if(!str.empty() && str[0] == '~') {
			str = std::string(getenv("HOME")) + std::string(str.begin()+1, str.end());
		}
	}
	}

	void expand_data_paths() {
		expand_path(level_path_);
		expand_path(save_file_path_);
		expand_path(auto_save_file_path_);
		expand_path(preferences_path_);
	}
	
	bool show_debug_hitboxes() {
		return show_debug_hitboxes_;
	}
	
	bool use_pretty_scaling() {
		return use_pretty_scaling_;
	}
	
	void set_use_pretty_scaling(bool value) {
		use_pretty_scaling_ = value;
	}
	
	bool fullscreen() {
		return fullscreen_;
	}
	
	void set_fullscreen(bool value) {
		fullscreen_ = value;
	}

	bool resizable() {
		return resizable_;
	}
	
	bool reverse_ab() {
		return reverse_ab_;
	}
	
	void set_reverse_ab(bool value) {
		reverse_ab_ = value;
	}
	
	void set_widescreen()
	{
		virtual_screen_width_ = (virtual_screen_height_*16)/9;
		actual_screen_width_ = (actual_screen_height_*16)/9;
		recalculate_draw_mask();
	}
	
	int virtual_screen_width()
	{
		return virtual_screen_width_;
	}
	
	int virtual_screen_height()
	{
		return virtual_screen_height_;
	}
	
	void set_virtual_screen_width (int width)
	{
		virtual_screen_width_ = width;
		recalculate_draw_mask();
	}
	
	void set_virtual_screen_height (int height)
	{
		virtual_screen_height_ = height;
	}
	
	int actual_screen_width()
	{
		return actual_screen_width_;
	}
	
	int actual_screen_height()
	{
		return actual_screen_height_;
	}
	
	void set_actual_screen_width(int width)
	{
		actual_screen_width_ = width;
		if(screen_editor_mode) {
			virtual_screen_width_ = actual_screen_width_;
		}
		recalculate_draw_mask();
	}
	
	void set_actual_screen_height(int height)
	{
		actual_screen_height_ = height;
		if(screen_editor_mode) {
			virtual_screen_height_ = actual_screen_height_;
		}
	}

	bool load_compiled()
	{
		return load_compiled_;
	}

	void set_load_compiled(bool value)
	{
		load_compiled_ = value;
	}
	
	bool force_no_npot_textures()
	{
		return force_no_npot_textures_;
	}
	
	bool screen_rotated()
	{
		return screen_rotated_;
	}

	bool debug()
	{
		return debug_;
	}

	bool show_fps()
	{
		return show_fps_;
	}

	bool use_joystick()
	{
		return use_joystick_;
	}

	void load_preferences()
	{
		std::string path = PREFERENCES_PATH;
		expand_path(path);
		if(!sys::file_exists(path + "preferences.cfg")) {
			return;
		}

		const wml::const_node_ptr node = wml::parse_wml_from_file(path + "preferences.cfg");
		if(node.get() == NULL) {
			return;
		}

		unique_user_id = wml::get_int(node, "user_id", 0);

		use_joystick_ = wml::get_bool(node, "joystick", use_joystick_);

		no_sound_ = wml::get_bool(node, "no_sound", no_sound_);
		no_music_ = wml::get_bool(node, "no_music", no_music_);
		reverse_ab_ = wml::get_bool(node, "reverse_ab", reverse_ab_);

		sound::set_music_volume(wml::get_int(node, "music_volume", 1000)/1000.0);
		sound::set_sound_volume(wml::get_int(node, "sound_volume", 1000)/1000.0);

		controls::set_sdlkey(controls::CONTROL_UP, static_cast<SDLKey>(wml::get_int(node, "key_up", SDLK_UP)));
		controls::set_sdlkey(controls::CONTROL_DOWN, static_cast<SDLKey>(wml::get_int(node, "key_down", SDLK_DOWN)));
		controls::set_sdlkey(controls::CONTROL_LEFT, static_cast<SDLKey>(wml::get_int(node, "key_left", SDLK_LEFT)));
		controls::set_sdlkey(controls::CONTROL_RIGHT, static_cast<SDLKey>(wml::get_int(node, "key_right", SDLK_RIGHT)));
		controls::set_sdlkey(controls::CONTROL_ATTACK, static_cast<SDLKey>(wml::get_int(node, "key_attack", SDLK_d)));
		controls::set_sdlkey(controls::CONTROL_JUMP, static_cast<SDLKey>(wml::get_int(node, "key_jump", SDLK_a)));
		controls::set_sdlkey(controls::CONTROL_TONGUE, static_cast<SDLKey>(wml::get_int(node, "key_tongue", SDLK_s)));
	}

	void save_preferences()
	{
		wml::node_ptr node(new wml::node("preferences"));
		node->set_attr("user_id", formatter() << get_unique_user_id());
		node->set_attr("no_sound", no_sound_ ? "true" : "false");
		node->set_attr("no_music", no_music_ ? "true" : "false");
		node->set_attr("reverse_ab", reverse_ab_ ? "true" : "false");
		node->set_attr("joystick", use_joystick_ ? "true" : "false");
		node->set_attr("sound_volume", formatter() << static_cast<int>(sound::get_sound_volume()*1000));
		node->set_attr("music_volume", formatter() << static_cast<int>(sound::get_music_volume()*1000));
		node->set_attr("key_up", formatter() << controls::get_sdlkey(controls::CONTROL_UP));
		node->set_attr("key_down", formatter() << controls::get_sdlkey(controls::CONTROL_DOWN));
		node->set_attr("key_left", formatter() << controls::get_sdlkey(controls::CONTROL_LEFT));
		node->set_attr("key_right", formatter() << controls::get_sdlkey(controls::CONTROL_RIGHT));
		node->set_attr("key_attack", formatter() << controls::get_sdlkey(controls::CONTROL_ATTACK));
		node->set_attr("key_jump", formatter() << controls::get_sdlkey(controls::CONTROL_JUMP));
		node->set_attr("key_tongue", formatter() << controls::get_sdlkey(controls::CONTROL_TONGUE));
		sys::write_file(preferences_path_ + "preferences.cfg", wml::output(node));
	}

	editor_screen_size_scope::editor_screen_size_scope() : width_(virtual_screen_width_), height_(virtual_screen_height_) {
		++screen_editor_mode;
		virtual_screen_width_ = actual_screen_width_;
		virtual_screen_height_ = actual_screen_height_;
	}

	editor_screen_size_scope::~editor_screen_size_scope() {
		virtual_screen_width_ = width_;
		virtual_screen_height_ = height_;
		--screen_editor_mode;
	}
	
	bool parse_arg(const char* arg) {
		const std::string s(arg);

		std::string arg_name, arg_value;
		std::string::const_iterator equal = std::find(s.begin(), s.end(), '=');
		if(equal != s.end()) {
			arg_name = std::string(s.begin(), equal);
			arg_value = std::string(equal+1, s.end());
		}

		if(arg_name == "--level-path") {
			level_path_ = arg_value + "/";
		} else if(s == "--show-hitboxes") {
			show_debug_hitboxes_ = true;
		} else if(s == "--scale") {
			set_use_pretty_scaling(true);
		} else if(s == "--no-sound") {
			no_sound_ = true;
		} else if(s == "--no-music") {
			no_music_ = true;
		} else if(s == "--sound") {
			no_sound_ = false;
		} else if(s == "--music") {
			no_music_ = false;
		} else if(s == "--fullscreen") {
			fullscreen_ = true;
		} else if(s == "--resizable") {
			resizable_ = true;
		} else if(s == "--no-resizable") {
			resizable_ = false;
		} else if(s == "--widescreen") {
			set_widescreen();
		} else if(s == "--bigscreen") {
			virtual_screen_width_ = actual_screen_width_;
			virtual_screen_height_ = actual_screen_height_;
		} else if(s == "--potonly") {
			force_no_npot_textures_ = true;
		} else if(s == "--textures16") {
			use_16bpp_textures_ = true;
		} else if(s == "--textures32") {
			use_16bpp_textures_ = false;
		} else if(s == "--debug") {
			debug_ = true;
		} else if(s == "--no-debug") {
			debug_ = false;
		} else if(s == "--simiphone") {
			sim_iphone_ = true;

			virtual_screen_width_ = 960;
			virtual_screen_height_ = 640;
		
			actual_screen_width_ = 480;
			actual_screen_height_ = 320;
			use_16bpp_textures_ = true;

			recalculate_draw_mask();
		} else if(s == "--simipad") {
			sim_iphone_ = true;

			virtual_screen_width_ = 1024;
			virtual_screen_height_ = 768;
		
			actual_screen_width_ = 1024;
			actual_screen_height_ = 768;

			recalculate_draw_mask();
		} else if(s == "--wvga") {
			virtual_screen_width_ = 800;
			virtual_screen_height_ = 480;

			actual_screen_width_ = 800;
			actual_screen_height_ = 480;

			recalculate_draw_mask();
		} else if(s == "--fps") {
			show_fps_ = true;
		} else if(s == "--no-fps") {
			show_fps_ = false;
		} else if(arg_name == "--config-path" && !arg_value.empty()) {
			set_preferences_path(arg_value);
		} else if(s == "--no-send-stats") {
			send_stats_ = false;
		} else if(s == "--joystick") {
			use_joystick_ = true;
		} else if(s == "--no-joystick") {
			use_joystick_ = false;
		} else {
			return false;
		}
		
		return true;
	}

	bool use_16bpp_textures() {
		return use_16bpp_textures_;
	}

	bool sim_iphone() {
		return sim_iphone_;
	}

	bool send_stats() {
		return send_stats_;
	}
}
