/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.cli.commands;

import com.sun.appserv.management.client.prefs.LoginInfo;
import com.sun.appserv.management.client.prefs.LoginInfoStore;
import com.sun.appserv.management.client.prefs.LoginInfoStoreFactory;
import com.sun.enterprise.cli.framework.*;

import java.io.IOException;
import javax.management.MBeanServerConnection;

public class LoginCommand extends S1ASCommand 
{

    /**
     *  A method that validates the options/operands 
     *  @return boolean returns true if success else returns false
     *  @throws CommandValidationException
     */
    public boolean validateOptions() throws CommandValidationException
    {
    	return super.validateOptions();
    }

    
    /**
     *  Method that Executes the command
     *  @throws CommandException, CommandValidationException
     */
    public void runCommand() throws CommandException, CommandValidationException
    {
        validateOptions();
        String userName = getUser();
        String password = getPassword();
        final String host = getHost();
        final int port = getPort();
        authenticate(host, port, userName, password);
        saveLogin(host, port, userName, password);
    }
    
    protected String getUser() throws CommandValidationException
    {
        try {
            InputsAndOutputs.getInstance().getUserOutput().print(
                                        getLocalizedString("AdminUserPrompt"));
            return InputsAndOutputs.getInstance().getUserInput().getLine();
        }
        catch (IOException ioe)
        {
            throw new CommandValidationException(getLocalizedString("CannotReadOption",
                                                    new Object[]{"user"}));
        }
    }


    protected String getPassword() throws CommandValidationException
    {
        String passwordValue;
        
        try 
        {
            InputsAndOutputs.getInstance().getUserOutput().print(
                                getLocalizedString("AdminPasswordPrompt"));
            passwordValue = new CliUtil().getPassword();
        }
        catch (java.lang.NoClassDefFoundError e)
        {
            passwordValue = readInput();
        }
        catch (java.lang.UnsatisfiedLinkError e)
        {
            passwordValue = readInput();
        }
        catch (Exception e)
        {
            throw new CommandValidationException(e);
        }
        return passwordValue;
    }
    
    
    private void authenticate(final String host, final int port, final String user, final String password)
    throws CommandException, CommandValidationException {
        try {
            final Object[] params = new Object[] {host, ""+port};
            CLILogger.getInstance().printMessage(getLocalizedString("AuthenticatingMsg", params));
            final MBeanServerConnection mbsc = super.getMBeanServerConnection(host, port, user, password);
            final String dd = mbsc.getDefaultDomain(); //calls a dummy method to make sure authentication is OK
            final String msg = "Authentication succeeded to: " + host + "and port: " + port;
            CLILogger.getInstance().printDebugMessage(msg);
        }
        catch(final IOException ioe) {
            throw new CommandException(ioe);
        }
    }
    
    
    private void saveLogin(final String host, final int port, final String user, final String password)
    throws CommandException {
        try {
            final LoginInfoStore store = LoginInfoStoreFactory.getStore(null);
            final LoginInfo login      = new LoginInfo(host, port, user, password);
            String msg                 = null;
            final boolean storeIt      = handleExists(store, login);
            if (storeIt) {
                store.store(login, true);
                final Object[] params = new Object[] {login.getUser(), login.getHost(), ""+login.getPort(), store.getName()};
                msg = getLocalizedString("LoginInfoStored", params);
            }
            else {
                final Object[] params = new Object[] {login.getHost(), ""+login.getPort()};
                msg = getLocalizedString("LoginInfoNotStored", params);
            }
            CLILogger.getInstance().printMessage(msg);
        }
        catch(final Exception e) {
            throw new CommandException(e);
        }
    }
    
    
    private boolean handleExists(final LoginInfoStore store, final LoginInfo login) throws Exception {
        boolean storeIt = true;
        if (store.exists(login.getHost(), login.getPort())) {
            storeIt = promptOnce(login);
        }
        return ( storeIt );
    }
    
    
    private boolean promptOnce(final LoginInfo login) throws Exception {
        boolean userPressedYes = false;
        final String YES = "y";
        final Object[] loginId = new Object[] {login.getHost(), ""+login.getPort()};
        final String prompt = getLocalizedString("ShouldOverwriteLogin", loginId);
        final InputsAndOutputs io = InputsAndOutputs.getInstance();
        io.getUserOutput().print(prompt);
        final String in = io.getUserInput().getLine();
        userPressedYes = YES.equalsIgnoreCase(in);
        return ( userPressedYes );
    }
}