/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.appserv.management.j2ee.statistics;

import java.util.Map;

import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;

import javax.management.j2ee.statistics.Statistic;
import javax.management.j2ee.statistics.CountStatistic;
import javax.management.j2ee.statistics.RangeStatistic;
import javax.management.j2ee.statistics.BoundaryStatistic;
import javax.management.j2ee.statistics.BoundedRangeStatistic;
import javax.management.j2ee.statistics.TimeStatistic;

import com.sun.appserv.management.util.jmx.OpenMBeanUtil;
import com.sun.appserv.management.util.misc.ClassUtil;
import com.sun.appserv.management.util.misc.TypeCast;

/**
	Creates Statistic implementations based on CompositeData or Map.
 */
public final class StatisticFactory 
{
	private	StatisticFactory()	{}
	
	
	/**
		Create a new Statistic using the specified CompositeData
		
		@param theInterface		interface which the Statistic should implement, must extend Statistic
	 */
		public static Statistic
	create( Class<? extends Statistic> theInterface, final CompositeData data )
	{
		return( create( theInterface, OpenMBeanUtil.compositeDataToMap( data ) ) );
	}
	
	private static final String	COUNT_STATISTIC			= CountStatistic.class.getName();
	private static final String	TIME_STATISTIC			= TimeStatistic.class.getName();
	private static final String	RANGE_STATISTIC			= RangeStatistic.class.getName();
	private static final String	BOUNDARY_STATISTIC		= BoundaryStatistic.class.getName();
	private static final String	BOUNDED_RANGE_STATISTIC	= BoundedRangeStatistic.class.getName();
	private static final String	STRING_STATISTIC		= StringStatistic.class.getName();
	private static final String	MAP_STATISTIC			= MapStatistic.class.getName();
	private static final String	NUMBER_STATISTIC                = NumberStatistic.class.getName();
	
	
	
	/**
		Create a new Statistic using the specified CompositeData.
		The CompositeType of data must be an appropriate Statistic class.
	 */
		public static Statistic
	create( final CompositeData data )
	{
		final String	typeName	= data.getCompositeType().getTypeName();
		Class<? extends Statistic>	theInterface	= null;
		if ( typeName.equals( COUNT_STATISTIC ) )
		{
			theInterface	= CountStatistic.class;
		}
		else if ( typeName.equals( TIME_STATISTIC ) )
		{
			theInterface	= TimeStatistic.class;
		}
		else if ( typeName.equals( RANGE_STATISTIC ) )
		{
			theInterface	= RangeStatistic.class;
		}
		else if ( typeName.equals( BOUNDARY_STATISTIC ) )
		{
			theInterface	= BoundaryStatistic.class;
		}
		else if ( typeName.equals( BOUNDED_RANGE_STATISTIC ) )
		{
			theInterface	= BoundedRangeStatistic.class;
		}
		else if ( typeName.equals( STRING_STATISTIC ) )
		{
			theInterface	= StringStatistic.class;
		}
		else if ( typeName.equals( NUMBER_STATISTIC ) )
		{
			theInterface	= NumberStatistic.class;
		}
		else if ( typeName.equals( MAP_STATISTIC ) )
		{
			theInterface	= MapStatistic.class;
		}
		else
		{
			try
			{
				theInterface	= TypeCast.asClass( ClassUtil.classForName( typeName ) );
			}
			catch( Exception e )
			{
				theInterface	= Statistic.class;
			}
		}
		
		return( create( theInterface, data ) );
	}
	
	
	/**
		Create a new Statistic using the specified map.  The standard JSR 77
		types are handled appropriately. Custom (non-standard) Statistics
		may also be used; in this case a proxy is returned which implements
		the interface specified by theClass.
		
		@param	theInterface	the interface which the resulting statistic implements
		@param	mappings		a Map containing keys of type String and their Object values
	 */
		public static Statistic
	create( Class<? extends Statistic> theInterface, final Map<String,?> mappings )
	{
		Statistic	result	= null;
		
		// hopefully specific classes are faster than a proxy...
		if ( theInterface == CountStatistic.class )
		{
			result	= new CountStatisticImpl( mappings );
		}
		else if ( theInterface == RangeStatistic.class )
		{
			result	= new RangeStatisticImpl( mappings );
		}
		else if ( theInterface == BoundaryStatistic.class )
		{
			result	= new BoundaryStatisticImpl( mappings );
		}
		else if ( theInterface == BoundedRangeStatistic.class )
		{
			result	= new BoundedRangeStatisticImpl( mappings );
		}
		else if ( theInterface == TimeStatistic.class )
		{
			result	= new TimeStatisticImpl( mappings );
		}
		else if ( theInterface == StringStatistic.class )
		{
			result	= new StringStatisticImpl( mappings );
		}
		else if ( theInterface == NumberStatistic.class )
		{
			result	= new NumberStatisticImpl( mappings );
		}
		else if ( theInterface == MapStatistic.class )
		{
			result	= new MapStatisticImpl( mappings );
		}
		else
		{
			throw new IllegalArgumentException(
				"Unsupported Statistic interface: " + theInterface.getName() );
		}
		
		return( result );
	}
	

	
	private static final Class<? extends Statistic>[] KNOWN_STATISTICS = TypeCast.asArray( new Class[]
	{
		CountStatistic.class,
		TimeStatistic.class,
		BoundedRangeStatistic.class,	// must come before RangeStatistic,BoundaryStatistic
		RangeStatistic.class,
		BoundaryStatistic.class,
		StringStatistic.class,
		NumberStatistic.class,
	} );
	
	private static final String	INTERNAL_STRING_STATISTIC_CLASSNAME	=
		"com.sun.enterprise.admin.monitor.stats.StringStatisticImpl";
		
		public static Class<? extends Statistic>
	getInterface( final Statistic s )
	{
		final Class<? extends Statistic>	implClass	= s.getClass();
		
		Class<? extends Statistic>	theInterface	= MapStatistic.class;
		
		for( int i = 0; i < KNOWN_STATISTICS.length; ++i )
		{
			final Class<? extends Statistic>	candidateInterface	= KNOWN_STATISTICS[ i ];
			
			if ( candidateInterface.isAssignableFrom( implClass )  )
			{
				theInterface	= candidateInterface;
				break;
			}
		}
		
		if ( theInterface == MapStatistic.class && ! (s instanceof MapStatisticImpl) )
		{
			if ( s.getClass().getName().equals( INTERNAL_STRING_STATISTIC_CLASSNAME ) )
			{
				theInterface	= StringStatistic.class;
			}
			else
			{
				throw new IllegalArgumentException( "Unknown statistic class: " + s.getClass().getName() );
			}
		}
		
		return( theInterface );
	}
	
		

}





























