/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
 

package com.sun.enterprise.management.ext.lb;

import com.sun.appserv.management.config.LBConfig;
import com.sun.enterprise.server.ApplicationServer;
import com.sun.enterprise.server.pluggable.PluggableFeatureFactory;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Date;
import java.util.logging.Level;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;

import javax.management.ObjectName;

import com.sun.appserv.management.base.AMX;
import static com.sun.appserv.management.base.AMX.*;
import static com.sun.appserv.management.base.XTypes.*;
import com.sun.appserv.management.ext.lb.LoadBalancer;
import com.sun.appserv.management.config.LoadBalancerConfig;
import com.sun.appserv.management.client.ProxyFactory;
import com.sun.enterprise.management.support.AMXImplBase;
import com.sun.enterprise.server.pluggable.LBFeatureFactory;
/**
 * Implmentation class for LoadBalancer AMX MBean
 * @author Harsha R A
 * @since Appserver 9.0
 */
public final class LoadBalancerImpl extends AMXImplBase implements LoadBalancer {
    
    private static final String COMMA = ",";
    private static final String COLON = ":";
    private static final String EQUAL = "=";

    public static final String	NAME_PROP_VALUE	= "load-balancer";
    public static final String  LAST_APPLIED_PROPERTY = "last-applied";    
    public static final String  LAST_APPLIED_HASH_PROPERTY = "last-applied-message-digest";    
    public static final String  LAST_EXPORTED_PROPERTY = "last-exported";    

    public static final String LOADBALANCER_CONFIG_OBJECT_NAME=
                    JMX_DOMAIN+COLON+J2EE_TYPE_KEY+EQUAL+
                    LOAD_BALANCER_CONFIG+COMMA+NAME_KEY+EQUAL;

    public static final String LBCONFIG_OBJECT_NAME=
                    JMX_DOMAIN+COLON+J2EE_TYPE_KEY+EQUAL+
                    LB_CONFIG+COMMA+NAME_KEY+EQUAL;

    public static final String LOADBALANCER_OBJECT_NAME = 
                    JMX_DOMAIN+COLON+J2EE_TYPE_KEY+EQUAL+
                    LOAD_BALANCER+COMMA+NAME_KEY+EQUAL;

    private LBFeatureFactory lbFactory = null;
    private String lbName = null;
    private String lbConfigName = null;
    private LoadBalancerConfig loadBalancerConfig = null;
    private static final String XML_COMMENT_START = "<!--";
    private static final String MD5 = "MD5";
    private final MBeanServerConnection	mServer;

    public LoadBalancerImpl( final MBeanServerConnection server) {
        super( );
        mServer			= server;
        PluggableFeatureFactory featureFactory =
                ApplicationServer.getServerContext().getPluggableFeatureFactory();
        
        lbFactory = featureFactory.getLBFeatureFactory();
        
    }
	
    /**
     * Applies changes in the corresponding configuration to this LB
     */
    public void applyLBChanges() {
        lbName = getName();
        loadBalancerConfig = getLoadBalancerConfig(lbName);
        if(loadBalancerConfig!=null) {
            lbConfigName = loadBalancerConfig.getLbConfigName();
        }
        lbFactory.applyChanges(lbConfigName,lbName);
        if(!loadBalancerConfig.existsProperty(LAST_APPLIED_PROPERTY))
            loadBalancerConfig.createProperty(LAST_APPLIED_PROPERTY,String.valueOf(new Date().getTime()));
        else
            loadBalancerConfig.setPropertyValue(LAST_APPLIED_PROPERTY,String.valueOf(new Date().getTime()));
        String digest = getMessageDigest();
        if(!loadBalancerConfig.existsProperty(LAST_APPLIED_HASH_PROPERTY))
            loadBalancerConfig.createProperty(LAST_APPLIED_HASH_PROPERTY,digest);
        else
            loadBalancerConfig.setPropertyValue(LAST_APPLIED_HASH_PROPERTY,digest);
    }        

    /**
     * checks if apply change is required
     * @return true if there are pending changes for this LB
     */
    public boolean isApplyChangeRequired() {
        boolean isRequired = true;
        try{
        String digest = getMessageDigest();
        String lastDigest = getLastAppliedMessageDigest();
        if(lastDigest!=null)
            isRequired = !digest.equals(lastDigest);
        }catch (Throwable t){
            getMBeanLogger().warning(t.getMessage());
            if(getMBeanLogger().isLoggable(Level.FINE))
                t.printStackTrace();
        }
        return isRequired;
                
    }  	

    /**
     * Returns the timestamp of the most recent application of referenced LBConfig
     * @return Date the timestamp when the changes were applied to the load balancer
     */
    public Date getLastApplied() {
        lbName = getName();
        loadBalancerConfig = getLoadBalancerConfig(lbName);
        if(loadBalancerConfig.existsProperty(LAST_APPLIED_PROPERTY))
            return new Date(Long.valueOf(loadBalancerConfig.getPropertyValue(LAST_APPLIED_PROPERTY)));
        return null;
    }
    
    /**
     * method to return the message digest of the load balaner.xml that was last
     * applied
     * @return String last-applied-digest property of the load balancer from domain.xml
     */
    public String getLastAppliedMessageDigest() {
        lbName = getName();
        loadBalancerConfig = getLoadBalancerConfig(lbName);
        if(loadBalancerConfig.existsProperty(LAST_APPLIED_HASH_PROPERTY))
            return loadBalancerConfig.getPropertyValue(LAST_APPLIED_HASH_PROPERTY);
        return null;
    }

    /**
     * Returns the timestamp of the most recent export of referenced LBConfig
     * @return Date timestamp
     */
    public Date getLastExported() {
        lbName = getName();
        getLoadBalancerConfig(lbName);
        if(loadBalancerConfig!=null) {
            lbConfigName = loadBalancerConfig.getLbConfigName();
        }
        LBConfig lbConfig = getLBConfig(lbConfigName);
        if(lbConfig.existsProperty(LAST_EXPORTED_PROPERTY))
            return new Date(Long.valueOf(lbConfig.getPropertyValue(LAST_EXPORTED_PROPERTY)));
        return null;
    }

    /**
     * Exports the corresponding LBConfig information and returns the contents as a string.
     * @see com.sun.appserv.management.config.LBConfig
     * @return String the loadbalancer.xml as a string
     */    
    public String getLoadBalancerXML() {
        return getLoadBalancerXML(true);
    }
    

    private LoadBalancerConfig getLoadBalancerConfig(String lbName){
        ObjectName loadBalancerConfigObjName = null;
        try{
            loadBalancerConfigObjName =
                    new ObjectName(LOADBALANCER_CONFIG_OBJECT_NAME+lbName);
        } catch ( MalformedObjectNameException e ){
            if(getMBeanLogger().isLoggable(Level.FINE))
                e.printStackTrace();
        }
        loadBalancerConfig =
                getProxy(loadBalancerConfigObjName, LoadBalancerConfig.class);        
        return loadBalancerConfig;
    }
    
    private LBConfig getLBConfig(String lbConfigName){
        ObjectName lbConfigObjName = null;
        try{
            lbConfigObjName =
                    new ObjectName(LBCONFIG_OBJECT_NAME+lbConfigName);
        } catch ( MalformedObjectNameException e ){
            e.printStackTrace();
        }
        LBConfig lbConfig = getProxy(lbConfigObjName, LBConfig.class);
        return lbConfig;
    }
    
    private String getLoadBalancerXML(boolean updateTimeStamp) {
        lbName = getName();
        getLoadBalancerConfig(lbName);
        if(loadBalancerConfig!=null) {
            lbConfigName = loadBalancerConfig.getLbConfigName();
        }
        LBConfig lbConfig = getLBConfig(lbConfigName);
        if(updateTimeStamp) {
            if(!lbConfig.existsProperty(LAST_EXPORTED_PROPERTY))
                lbConfig.createProperty(LAST_EXPORTED_PROPERTY,String.valueOf(new Date().getTime()));
            else
                lbConfig.setPropertyValue(LAST_EXPORTED_PROPERTY,String.valueOf(new Date().getTime()));
        }

        return lbFactory.getLoadBalancerXML(lbConfigName,lbName);
                
    }
    
    private String getMessageDigest(){
        try {
            String lbxml = getLoadBalancerXML(false).split(XML_COMMENT_START)[0];
            MessageDigest md = MessageDigest.getInstance(MD5);
            md.update(lbxml.getBytes());
            String hash = new BigInteger(md.digest()).toString(16);        
            return hash;
        }catch(NoSuchAlgorithmException e){
            getMBeanLogger().warning(e.getMessage());
            if(getMBeanLogger().isLoggable(Level.FINE))
                e.printStackTrace();            
        }
        return "";
    }


    /**
      Returns the timestamp of the last time the stats on this loadbalancer were reset
     */
    public Date getLastResetTime() {
        return null;
    }            

    /**
      Reset the monitoring stats on this loadbalancer.
     */
    public void resetStats() {
        String lbName = getName();
        String lbConfigName = null;
        loadBalancerConfig = getLoadBalancerConfig(lbName);
        if(loadBalancerConfig!=null) {
            lbConfigName = loadBalancerConfig.getLbConfigName();
        }
        lbFactory.resetStats(lbConfigName,lbName);        
    }               

    /**
      Test the LB and Domain Application Server setup 
     */
    public boolean testConnection() {
        String lbName = getName();
        String lbConfigName = null;
        loadBalancerConfig = getLoadBalancerConfig(lbName);
        if(loadBalancerConfig!=null) {
            lbConfigName = loadBalancerConfig.getLbConfigName();
        }
        return lbFactory.testConnection(lbConfigName,lbName);
    }
            
    /**
     * Returns the uhealthy/healthy/quiesced status for an insatnce load balanced
     * by this load balancer.
     */
    public String getStatus(String instanceName) {
        return null;
    }     
}









