/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.management.support;


import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.Collection;
import java.io.IOException;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.JMException;
import javax.management.InstanceNotFoundException;

import com.sun.appserv.management.base.AMXDebug;
import com.sun.appserv.management.base.Container;
import com.sun.appserv.management.base.Util;
import com.sun.appserv.management.client.ProxyFactory;
import com.sun.enterprise.management.support.ObjectNames;

import com.sun.appserv.management.monitor.LoadBalancerMonitor;  
import com.sun.appserv.management.monitor.LoadBalancerClusterMonitor;
import com.sun.appserv.management.monitor.LoadBalancerServerMonitor;
import com.sun.appserv.management.monitor.LoadBalancerApplicationMonitor;
import com.sun.appserv.management.monitor.LoadBalancerContextRootMonitor;

import static com.sun.appserv.management.base.XTypes.DEPLOYED_ITEM_REF_CONFIG;
import com.sun.appserv.management.config.DeployedItemRefConfig;
import com.sun.appserv.management.config.ClusterConfig;
import com.sun.appserv.management.config.ServerConfig;
import com.sun.appserv.management.config.LBConfig;

/**
 * A class that triggers forming LB monitoring tree whenever a new <application-ref>
 * gets added to a <server> or <cluster> elements. The registration logic 
 * Determines parent object <server> or <cluster>.
 *
 * 0. If <server>  
 * 1. locate the referring <lb-config> elements (via <server-ref> under these 
 *     <lb-config> elements 
 * 2. locate the referring <load-balancer> elements and fetch corresponding 
 *    root LoadBalancerMonitors
 * 3. locate LoadBalancerServerMonitors under these root LoadBalancerMonitors
 *    (each LoadBalancerServerMonitor will reside under LoadBalancerClusterMonitor
 *     with name equal to the name of the <server> in step 0.)
 * 4. create LoadBalancerApplicationMonitor the rest of the monitoring tree under 
 *    this LoadBalancerApplicationMonitor as governed by context roots 
 *    associated with this <application> element
 *
 * 0. If <cluster> 
 * 1. locate the referring <lb-config> elements (via <cluster-ref> under these 
 *     <lb-config> elements 
 * 2. locate the referring <load-balancer> elements and fetch corresponding 
 *    root LoadBalancerMonitors 
 * 3. locate LoadBalancerServerMonitors under these root LoadBalancerMonitors
 *    (each LoadBalancerServerMonitor will reside under LoadBalancerClusterMonitor
 *     with name equal to the name of the <server> in step 0.)
 * 4. create LoadBalancerApplicationMonitor the rest of the monitoring tree 
 *    under this LoadBalancerApplicationMonitor as governed by context roots 
 *    associated with this <application> element
 *
 * @see LoadBalancerClusterRefRegistrationListener
 * @see LoadBalancerServerRefRegistrationListener
 * @see LoadBalancerRegistrationListener
 */

public class LoadBalancerApplicationRefRegistrationListener 
    extends LBBaseMBeanRegistrationListener {

    public LoadBalancerApplicationRefRegistrationListener(MBeanServer mBeanServer)
        throws InstanceNotFoundException, IOException {
        super(mBeanServer);
    }

    protected void mbeanRegistered(final ObjectName objectName) {
        try {
            if (DEPLOYED_ITEM_REF_CONFIG.equals(Util.getJ2EEType(objectName))) {
                DeployedItemRefConfig appRefConfig = 
                    ProxyFactory.getInstance(getConn()).getProxy(objectName, DeployedItemRefConfig.class);
                String appName = appRefConfig.getRef();
                Container parent = appRefConfig.getContainer();

                if (parent instanceof ServerConfig) {
                    Collection<LBConfig> affectedLBConfigArr = 
                        fetchLBConfigs(((ServerConfig)parent).getName(), false);
                    for (LBConfig lbConfig : affectedLBConfigArr) {
                        List<LoadBalancerMonitor> monitoringRoots = 
                            fetchLBMonitoringRoots(lbConfig);
                        List<LoadBalancerServerMonitor> serverMonitors = 
                            fetchLoadBalancerServerMonitors(monitoringRoots, 
                                ((ServerConfig)parent).getName(), false);
                        for (LoadBalancerServerMonitor lbsm : serverMonitors) 
                            registerLoadBalancerApplicationMonitorTree(lbsm, appName);
                    }
                } //end if (parent instanceof ServerConfig)

                else  if (parent instanceof ClusterConfig) {
                    Collection<LBConfig> affectedLBConfigArr = 
                        fetchLBConfigs(((ClusterConfig)parent).getName(), true);
                    for (LBConfig lbConfig : affectedLBConfigArr) {
                        List<LoadBalancerMonitor> monitoringRoots = 
                            fetchLBMonitoringRoots(lbConfig);
                        List<LoadBalancerServerMonitor> serverMonitors = 
                            fetchLoadBalancerServerMonitors(monitoringRoots, 
                                ((ClusterConfig)parent).getName(), true);
                        for (LoadBalancerServerMonitor lbsm : serverMonitors) 
                            registerLoadBalancerApplicationMonitorTree(lbsm, appName);
                    }
                }//end if (parent instanceof ClusterConfig)

            }
	} catch (Exception ex) {
            logWarning(
                "LoadBalancerApplicationRefRegistrationListener" +
                " mbeanRegistered failed. Exception = ", ex);	
	}
    }
    
    private List<LoadBalancerServerMonitor> fetchLoadBalancerServerMonitors(
        List<LoadBalancerMonitor> monitoringRoots, String targetName, boolean isCluster) {
        
        List<LoadBalancerServerMonitor> serverMonitors = new ArrayList<LoadBalancerServerMonitor>();
        
        if (isCluster) {
            List<LoadBalancerClusterMonitor> clusterMonitors = 
                fetchLoadBalancerClusterMonitors(monitoringRoots, targetName);
         
            for (LoadBalancerClusterMonitor lbcm : clusterMonitors) {
                Map<String, LoadBalancerServerMonitor> lbsmMap = 
                    lbcm.getLoadBalancerServerMonitorMap();
                serverMonitors.addAll(lbsmMap.values());
            }
        } else {
         /* its a serverName which means we have to find LoadBalancerServerMonitors 
         1. for standalone server references with the same name  
         2. for clustered server references with the same name 
         The logic below will take care of both the cases as every standalone server 
         monitor is wrapped in its single cluster monitor.*/
            for (LoadBalancerMonitor lbm : monitoringRoots) {
                Map<String, LoadBalancerClusterMonitor> lbcmMap =  
                    lbm.getLoadBalancerClusterMonitorMap();
                for (LoadBalancerClusterMonitor lbcm : lbcmMap.values()) {
                    Map<String, LoadBalancerServerMonitor> lbsmMap = 
                        lbcm.getLoadBalancerServerMonitorMap();
                    for (LoadBalancerServerMonitor lbsm : lbsmMap.values()) 
                        if (lbsm.getName().equals(targetName))
                            serverMonitors.add(lbsm);
                }
            }
        }
        return serverMonitors;
    }
}