/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.mbeans;

import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.ConfigContext;

import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.util.i18n.StringManagerBase;

import com.sun.enterprise.admin.config.BaseConfigMBean;
import com.sun.enterprise.admin.common.constant.AdminConstants;

import com.sun.enterprise.security.store.PasswordAdapter;
import com.sun.enterprise.security.store.IdentityManager;

import java.util.logging.Logger;
import java.util.Enumeration;
import java.util.ArrayList;

import java.io.IOException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;

public class PasswordAliasConfigMBean extends BaseConfigMBean 
{              
    private static final StringManager _strMgr = 
        StringManager.getManager(PasswordAliasConfigMBean.class);

    private static Logger _logger = null;   
        
    private static Logger getLogger() 
    {
        if (_logger == null) {
            _logger = Logger.getLogger(AdminConstants.kLoggerName);
        }
        return _logger;
    }           

   private static ExceptionHandler _handler = null;
    
    //The exception handler is used to parse and log exceptions
    protected static ExceptionHandler getExceptionHandler() 
    {
        if (_handler == null) {
            _handler = new ExceptionHandler(getLogger());
        }
        return _handler;
    }
    
    public PasswordAliasConfigMBean() {
        super();        
    }       
                
    private PasswordAdapter getPasswordAdapter()
        throws CertificateException, IOException, KeyStoreException, NoSuchAlgorithmException
    {
        //The masterPassword in the IdentityManager is available only through
        //a running DAS, server instance, or node agent.
        String masterPassword = IdentityManager.getMasterPassword();
        return new PasswordAdapter(masterPassword.toCharArray());
    }
    
    /**
     * Add a non-existing password alias
     * @param aliasName the name of the alias   
     * @param password the password of the alias
     * @throws ConfigException
     */    
    public void addPasswordAlias(String aliasName, String password) throws ConfigException        
    {
        try {            
            PasswordAdapter p = getPasswordAdapter();
            if (p.aliasExists(aliasName)) {
                throw new ConfigException(_strMgr.getString("passwordAliasExists", aliasName));
            } 
            p.setPasswordForAlias(aliasName, password.getBytes());           
        } catch (Exception ex) {
            throw getExceptionHandler().handleConfigException(
                ex, "addPasswordAliasException", aliasName);
        }
    }
    
    /**
     * Remove an existing password alias
     * @param aliasName the name of the password alias to remove
     * @throws ConfigException
     */    
    public void removePasswordAlias(String aliasName) throws ConfigException        
    {
        try {            
            PasswordAdapter p = getPasswordAdapter();
            if (!p.aliasExists(aliasName)) {
                throw new ConfigException(_strMgr.getString("passwordAliasDoesNotExist", aliasName));
            }
            p.removeAlias(aliasName);
        } catch (Exception ex) {
            throw getExceptionHandler().handleConfigException(
                ex, "removePasswordAliasException", aliasName);
        }
    }
    
    /**
     * Update the password for an existing alias
     * @param aliasName the name of the alias whose password is to be updated
     * @param password the new password
     * @throws ConfigException
     */    
    public void updatePasswordAlias(String aliasName, String password) throws ConfigException        
    {
        try {
            PasswordAdapter p = getPasswordAdapter();
            if (!p.aliasExists(aliasName)) {
                throw new ConfigException(_strMgr.getString("passwordAliasDoesNotExist", aliasName));
            }
            p.setPasswordForAlias(aliasName, password.getBytes());
        } catch (Exception ex) {
            throw getExceptionHandler().handleConfigException(
                ex, "updatePasswordAliasException", aliasName);
        }
    }
    
    /**
     * Get all the password aliases
     * @throws ConfigException
     * @return The list of password aliases
     */    
    public String[] getPasswordAliases() throws ConfigException        
    {
        try {   
            ArrayList result = new ArrayList();
            Enumeration en = getPasswordAdapter().getAliases();
            while (en.hasMoreElements()) {                            
                result.add((String)en.nextElement());                               
            }            
            return (String[])result.toArray(new String[result.size()]);
        } catch (Exception ex) {
            throw getExceptionHandler().handleConfigException(
                ex, "listPasswordAliasException", "");
        }
    }
}
