/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.mbeans.custom;
import java.util.List;
import javax.management.ObjectName;

/** Defines the behavior for querying the various MBean definitions. This is to factor
 * out the MBean related operations. The implementing classes could have specific behaviors
 * depending upon product edition, for example.
 * @since SJSAS9.0
 */
public interface CustomMBeanConfigQueries {
    
    /** Returns the collection of all the custom MBeans for a target as a List of Strings. Each element
     * in the list the <i> name </i> of the custom MBean created. Thus for a standalone instance
     * this method would return all the MBeans that are referenced from it.
     * @param target String representing the cluster or server instance
     * @return a List of Strings representing the referenced MBeans
     * @throws CustomMBeanException
     */
    public List<String> listMBeanNames(String target) throws CustomMBeanException;
    
    /** Returns access to get more information about the custom MBeans referenced from a
     * server instance. Every Config MBean is a wrapper over the domain.xml element and
     * the ObjectName of this MBean can allow clients to get more information such as ObjectType of
     * the custom MBean, by standard JMX calls. This method returns such ObjectNames.  This method
     * is more useful with a caveat that getting more information from the server results
     * in additional calls to the server.
     * @see #listMBeanNames(String) which returns only the names of these MBeans
     * @param target String representing the cluster or server instance
     * @return a List of ObjectNames representing the Config MBeans representing referenced MBeans
     * @throws CustomMBeanException
     */
    public List<ObjectName> listMBeanConfigObjectNames(String target) throws CustomMBeanException;
    
    /** A convenience method to take care of various client requests. It resembles the function
     * of a filter. Useful in getting the List of MBeans that for example are of object-type "USER_DEFINED"
     * and are not enabled. This is a more general purpose query interface.
     * @param target String representing the cluster or server instance
     * @param type an integer representing the type of MBean
     * @state boolean representing if or not the MBean is enabled
     * @see CustomMBeanConstants
     * @see #listMBeanConfigObjectNames(String)
     * @return the List of Config MBeans that represent an MBean definition referenced from a server instance or cluster
     */ 
    public List<ObjectName> listMBeanConfigObjectNames(String target, int type, boolean state) throws CustomMBeanException;
    
    /** Returns if the given MBean is referenced from a given target.
     * @param target String representing the cluster or server instance
     * @param name String representing the <i> name </name> of the MBean
     * @return a boolean representing if the given MBean exists in the given target
     * @throws CustomMBeanException
     */
    public boolean existsMBean(String target, String name) throws CustomMBeanException;
    
    /** Returns if the given MBean is referenced from a given target and is enabled.
     * @param target String representing the cluster or server instance
     * @param name String representing the <i> name </name> of the MBean
     * @return a boolean representing if the given MBean exists in the given target and is enabled
     * @throws CustomMBeanException
     */
    public boolean isMBeanEnabled(String target, String name) throws CustomMBeanException;
}