/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * JndiMBeanManager.java
 *
 * Created on March 9, 2004, 1:46 PM
 */

package com.sun.enterprise.admin.monitor.jndi;

import com.sun.enterprise.admin.common.constant.AdminConstants;
import com.sun.enterprise.util.i18n.StringManager;
import java.util.logging.Logger;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;

/**
 * The JndiMBeanManager is responsible for registering and 
 * unregistering the JndiMBean.  
 *
 * @author  Rob Ruyak
 */
public class JndiMBeanManager {
    
    /** MBeanServer for registering/unregistering JndiMBean **/
    private MBeanServer server;
    private static final Logger logger = 
        Logger.getLogger(AdminConstants.kLoggerName);
    private static final StringManager sm = 
        StringManager.getManager(JndiMBeanManager.class);
    
    /** Creates a new instance of JndiMBeanManager */
    public JndiMBeanManager() {
        server = this.getMBeanServer();
    }
    
    /**
     * Registers the JndiMBean.
     *
     * @param objectName The objectName of the JndiMBean.
     */
    public void registerMBean(ObjectName objectName) {
        try {
            if (server.isRegistered(objectName)) {
                logger.fine(sm.getString("monitor.jndi.already_registered", 
                    new Object[] {objectName.toString()}));
                return;
            }            
            server.registerMBean(new JndiMBeanImpl(), objectName);
            logger.finer(sm.getString("monitor.jndi.registered", 
                    new Object[]{objectName.toString()}));
        } catch(Exception e) {
            logger.fine(sm.getString("monitor.jndi.register_exception", 
                    new Object[]{objectName}));
            logger.throwing(JndiMBeanManager.class.getName(),
                    "registerMBean()", e);
        }
    }
    
    /**
     * Unegisters the JndiMBean.
     *
     * @param objectName The objectName of the JndiMBean.
     */
    public void unregisterMBean(ObjectName objectName) {
        try {
            if (server.isRegistered(objectName)) {
                server.unregisterMBean(objectName);
                logger.fine(sm.getString("monitor.jndi.unregistered",
                        new Object[]{objectName.toString()}));
            } else {
                logger.fine(sm.getString("monitor.jndi.never_registered",
                        new Object[]{objectName.toString()}));
            } 
        } catch(Exception e) {
            logger.fine(sm.getString("monitor.jndi.register_failed", 
                    new Object[]{objectName.toString()}));
        }
    }
    
    /**
     * Gets the MBeanServer for use within the JndiMBeanManager.
     *
     * @return The MBeanServer used to register the mbean.
     */
    MBeanServer getMBeanServer() {
        MBeanServer server = null;
        java.util.ArrayList servers = MBeanServerFactory.findMBeanServer(null);
        if(!servers.isEmpty()){
            server = (MBeanServer)servers.get(0);
        }
        return server;
    }
}
