/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * RuntimeStatus.java
 *
 * Created on February 27, 2004, 12:26 PM
 */

package com.sun.enterprise.admin.servermgmt;

import com.sun.enterprise.admin.common.Status;

import java.io.Serializable;
import java.util.ArrayList;

/**
 * This class holds a list of runtime status objects which corresponds to a list of server
 * instances (e.g. in a cluster) or a list of node agents.
 * @author  kebbs
 */
public class RuntimeStatusList extends ArrayList implements Serializable {           
       
    public RuntimeStatusList() {         
        super();
    }
        
    public RuntimeStatusList(int capacity) 
    {
        super(capacity);
    }
    
    /**
     * Return the RuntimeStatus at the specified index
     * @param index
     * @throws IndexOutOfBoundsException
     * @return
     */    
    public RuntimeStatus getStatus(int index) throws IndexOutOfBoundsException
    {
        return (RuntimeStatus)super.get(index);
    }
    
    
    /**
     * 
     * @return true if at least one of the servers / node agents in the list is in a running state.
     */    
    public boolean anyRunning()
    {
        return numRunning() > 0 ? true : false;
    }
    
    /**
     *
     * @return true if all of the server or node agents in the list is in a running state.
     * Note that there must be at least one server or node agent in the list.
     */    
    public boolean allRunning()
    {   
        if (isEmpty()) {
            return false;
        } else {
            return numRunning() == size() ? true : false;
        }
    }
       
    /**
     *
     * @return the number of running instances / node agents.
     */    
    public int numRunning()
    {
        int count = 0;
        for (int i = 0; i < size(); i++) {
            if (((RuntimeStatus)get(i)).isRunning()) {              
                count++;
            }
        }
        return count;
    }
    
    /**
     *
     * @return the number of instances / node agents needing a restart. Note that 
     * stopped instances (i.e. not running) will not be counted.
     */    
    public int numNeedingRestart()
    {
        int count = 0;
        for (int i = 0; i < size(); i++) {
            //Only running instances are considered for restart required
            if (((RuntimeStatus)get(i)).isRunning() &&
                ((RuntimeStatus)get(i)).isRestartNeeded())
            {              
                count++;
            }
        }
        return count;
    }
        
    /**
     *
     * @return the number of stopped (not running) server instances
     */    
    public int numStopped() 
    {
        return size() - numRunning();
    }
    
    /**
     *
     * @return true if there are no instances / agents in the list
     */    
    public boolean isEmpty()
    {
        return size() == 0 ? true : false;
    }
    
    
    /**
     *
     * @return a string version of the list which can be: "running" if allRunning(), 
     * "partially running" if anyRunning(), or "stopped" everywhere else.
     */    
    public String toString()          
    {
       if (allRunning()) {
           return Status.getStatusString(Status.kInstanceRunningCode);
       } else if (anyRunning()) {
           return Status.getStatusString(Status.kClusterPartiallyRunningCode);
       } else {
           return Status.getStatusString(Status.kInstanceNotRunningCode);
       }
    }
}
    
