/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.admin.servermgmt.pe;

import java.util.Map;
import java.util.ArrayList;
import java.io.File;
import java.lang.reflect.Method;

import com.sun.enterprise.util.ProcessExecutor;
import com.sun.enterprise.util.SystemPropertyConstants;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.admin.servermgmt.InstancesManager;
import com.sun.enterprise.admin.servermgmt.InstanceException;
import com.sun.enterprise.admin.servermgmt.pe.InstanceTimer;
import com.sun.enterprise.admin.servermgmt.pe.TimerCallback;

import com.sun.enterprise.admin.servermgmt.RepositoryConfig;
import com.sun.enterprise.admin.servermgmt.RepositoryManager;
import com.sun.enterprise.admin.servermgmt.DomainConfig;
import com.sun.enterprise.admin.common.Status;
import com.sun.enterprise.admin.server.core.channel.RMIClient;

/**
 */
public class PEInstancesManager extends RepositoryManager implements InstancesManager
{
    /**
     * i18n strings manager object
     */
    private static final StringManager _strMgr =
        StringManager.getManager(PEInstancesManager.class);

    private final RepositoryConfig _config;

    public PEInstancesManager(RepositoryConfig config)
    {
        super();
        _config = config;
    }

    protected RepositoryConfig getConfig()
    {
        return _config;
    }

    public void createInstance()
        throws InstanceException
    {
        throw new UnsupportedOperationException(
            _strMgr.getString("notSupported"));
    }

    public void deleteInstance() throws InstanceException
    {
        throw new UnsupportedOperationException(
            _strMgr.getString("notSupported"));
    }


    /**
     * startInstance method to use if there aren't any interactiveOptions or
     * args to append to the executing process
     *
     * @return The Process that is being excecuted
     * @throws InstanceException
     */
    public Process startInstance()  throws InstanceException {
        return startInstance(null, null);
    }


    /**
     * startInstance method to use if there are interactiveOption, but no
     * args to append to the executing process
     *
     * @param interativeOptions that are passed into the executing process' standard input
     * @return The Process that is being excecuted
     * @throws InstanceException
     */
    public Process startInstance(String[] interativeOptions)  throws InstanceException {
        String[] commandLineArgs=null;
        return startInstance(interativeOptions, commandLineArgs);
    }


    /**
     * startInstance method to use if there are interactiveOption and
     * args to append to the executing process
     *
     * @param interativeOptions that are passed into the executing process' standard input
     * @param commandLine arguments to be appended to the executing process' commandline
     * @return The Process that is being excecuted
     * @throws InstanceException
     */
    public Process startInstance(String[] interativeOptions, String[] commandLineArgs)
        throws InstanceException
    {
        preStart();

	// check if --debug or --verbose options were given to asadmin
	Boolean v = (Boolean)getConfig().get(DomainConfig.K_VERBOSE);
	boolean verbose = false;
	if ( v != null ) {
	    verbose = v.booleanValue();
	}

	Boolean d = (Boolean)getConfig().get(DomainConfig.K_DEBUG);
        boolean debug = false;
	if ( d != null ) {
	    debug = d.booleanValue();
	}

        String[] command=null;
        File script = getFileLayout(getConfig()).getStartServ();
        String nativeLauncher=System.getProperty(SystemPropertyConstants.NATIVE_LAUNCHER);
        Process process=null;
        ArrayList alCmd=new ArrayList();
        // append required arguments to start command
        alCmd.add(script.getAbsolutePath());

        //
        // add temporary switch for new ProcessLauncher. Will optimize this section
        // better once the commons-launcher is removed ???

        // This will be removed once PE using the new invocation classes
	if ( System.getProperty("com.sun.aas.processLauncher") == null && verbose ) {
	    try {
		// Invoke launcher directly without running startserv script.
		// This saves a JVM, and will allow CTRL-C and CTRL-\ on asadmin
		// to reach the server JVM.

		// Set system props needed by launcher
		System.setProperty(SystemPropertyConstants.INSTANCE_ROOT_PROPERTY,
                    getConfig().getRepositoryRoot() + File.separator +
                    getConfig().getRepositoryName());
		ArrayList args = new ArrayList();
		args.add("s1as-server");
                //FIXTHIS: The com.sun.aas.instanceName probably needs to be dynamically set, but for
                //now this is not important as it is not being used.
		args.add("-Dcom.sun.aas.instanceName=server");
		args.add("start");
		if ( debug ) {
		    args.add("debug");
		}
		if ( verbose ) {
		    args.add("verbose");
		}

                // addin command line args
                if (commandLineArgs != null) {
                    for(int ii=0; ii < commandLineArgs.length; ii++) {
                        args.add(commandLineArgs[ii]);
                    }
                }

                // extract full command
		String[] argStrings = (String[])args.toArray(
                    new String[args.size()]);

		/* This doesnt work with the JDK1.4.2 javac
		LauncherBootstrap.main(argStrings);
		*/
		Class launcherClass = Class.forName("LauncherBootstrap");
		Class[] paramClasses = new Class[] { String[].class };
		Object[] argsArray = new Object[] { argStrings };
		Method mainMethod = launcherClass.getMethod("main", paramClasses);

		// If verbose, LauncherBootstrap.main() returns only after
		// the appserver JVM exits.
		mainMethod.invoke(null, argsArray);

	    } catch ( Exception ex ) {
		throw new InstanceException(
		    getMessages().getInstanceStartupExceptionMessage(
			getConfig().getDisplayName()), ex);
	    }

    } else if (System.getProperty("com.sun.aas.processLauncher") != null && verbose) {
        // add arguments to main command, native must come first
        if (nativeLauncher != null && nativeLauncher.equals("true")) {
            // use native launcher, add in argument
            alCmd.add("native");
        }
        if (debug) alCmd.add("debug");
        if (verbose) alCmd.add("verbose");
        // addin command line args
        if (commandLineArgs != null) {
            for(int ii=0; ii < commandLineArgs.length; ii++) {
                alCmd.add(commandLineArgs[ii]);
            }
        }

        // extract full command
        command=new String[alCmd.size()];
        command=(String[])alCmd.toArray(command);

        try {
            // exec process directly to exercise needed control
            ProcessExecutor exec = new ProcessExecutor(command, interativeOptions);
            // set verbose flag so process error stream get redirected to stderr
            exec.setVerbose(verbose);
            // call execute so it will not be timed out
            exec.execute(false, false);
            process=exec.getSubProcess();
            // this will force process to wait for executing process
            int exitValue=process.waitFor();
            System.exit(exitValue);
        } catch (Exception e) {
            throw new InstanceException(_strMgr.getString("procExecError"), e);
        }

    } else {
        // add arguments to main command, native must come first
        if (nativeLauncher != null && nativeLauncher.equals("true")) {
            // use native launcher, add in argument
            alCmd.add("native");
        }

        // check to see if debug is enabled
        if (debug) {
            alCmd.add("debug");
        }

        // addin command line args
        if (commandLineArgs != null) {
            for(int ii=0; ii < commandLineArgs.length; ii++) {
                alCmd.add(commandLineArgs[ii]);
            }
        }

        // extract full command
        command=new String[alCmd.size()];
        command=(String[])alCmd.toArray(command);

        // call method for executing so can be overriden in descendants
        // also, keep executor for any error information
        ProcessExecutor processExec=startInstanceExecute(command, interativeOptions);
        process=processExec.getSubProcess();
	    waitUntilStarting(processExec);
	    waitUntilStarted();
        postStart();
	}

        return process;
    }

    /**
    * This method is called internally from the startInstance method
    * and was needed so SE could execute a process that doesn't return
    */
    protected ProcessExecutor startInstanceExecute(String[] command, String[] interativeOptions) throws InstanceException {
        return execute(command, interativeOptions);
    }



    public void stopInstance() throws InstanceException
    {
        preStop();
        execute(getFileLayout(getConfig()).getStopServ());
        waitUntilStopped();
        postStop();
    }

    public String[] listInstances() throws InstanceException
    {
        throw new UnsupportedOperationException(
            _strMgr.getString("notSupported"));
    }

    public boolean isInstanceStarting() throws InstanceException
    {
        return (getInstanceStatus() ==
             Status.kInstanceStartingCode);
    }

    public boolean isInstanceRunning() throws InstanceException
    {
        return (Status.kInstanceRunningCode ==
                getInstanceStatus());
    }

    /**
     * Returns whether the server is in the startup failed state or
     * not.
     */
    public boolean isInstanceFailed() throws InstanceException
    {
        return (Status.kInstanceFailedCode ==
                getInstanceStatus());
    }

    public boolean isInstanceNotRunning() throws InstanceException
    {
        return (Status.kInstanceNotRunningCode ==
                getInstanceStatus());
    }

    public boolean isRestartNeeded() throws InstanceException
    {
        boolean isRestartNeeded = false;
        try
        {
            isRestartNeeded = getRMIClient().isRestartNeeded();
        }
        catch (Exception e)
        {
            throw new InstanceException(e.getMessage(), e);
        }
        return isRestartNeeded;
    }

    /**
        To start an instance the instance must be in not running state.
     */
    protected void preStart() throws InstanceException
    {
        final int state = getInstanceStatus();
        if (Status.kInstanceNotRunningCode != state)
        {
            throw new InstanceException(
                getMessages().getCannotStartInstanceInvalidStateMessage(
                    getConfig().getDisplayName(),
                    Status.getStatusString(state)));
        }
    }

    void postStart() throws InstanceException
    {
        if (isInstanceFailed()) {
            int port = getConflictedPort();
            abortServer();
            throw new InstanceException(
                getMessages().getStartupFailedMessage(
                    getConfig().getDisplayName(), port ));
        }
        if (!isInstanceRunning() &&
            !isInstanceNotRunning())
        {
            /*
                Instance could not be started in TIME_OUT_SECONDS.
                Trying to stop.
             */
            try {
                stopInstance();
            } catch (Exception e) {
                throw new InstanceException(
                    getMessages().getStartInstanceTimeOutMessage(
                        getConfig().getDisplayName()), e);
            }
            throw new InstanceException(
               getMessages().getStartInstanceTimeOutMessage(
                   getConfig().getDisplayName()));
        }
        if (isInstanceNotRunning()) {
            throw new InstanceException(
                getMessages().getStartupFailedMessage(
                    getConfig().getDisplayName()));
        }
        setRMIClient(null);
    }

    /**
        To stop an instance the instance must be in (running | stopping |
        starting) state.
     */
    void preStop() throws InstanceException
    {
        final int state = getInstanceStatus();
        /*
        if (!((state == Status.kInstanceRunningCode) ||
              (state == Status.kInstanceStoppingCode) ||
              (state == Status.kInstanceStartingCode)))
         */
        //For now we can only allow running instances to be stopped. Stopping the 
        //node agent while it is starting is a bad idea.
        if (state != Status.kInstanceRunningCode)
        {
            throw new InstanceException(
                getMessages().getCannotStopInstanceInvalidStateMessage(
                    getConfig().getDisplayName(),
                    Status.getStatusString(state)));
        }
    }

    void postStop() throws InstanceException
    {
        if (!isInstanceNotRunning())
        {
            throw new InstanceException(
                getMessages().getCannotStopInstanceMessage(
                    getConfig().getDisplayName()));
        }
        setRMIClient(null);
    }


    // method to over ride timeout time for server to go to starting state
    // This value should be configurable by end user ???
    protected void waitUntilStarting(ProcessExecutor processExec) throws InstanceException
    {
        waitUntilStarting(processExec, 180);
    }

    protected void waitUntilStarting(ProcessExecutor processExec, int timeoutSeconds) throws InstanceException
    {
        InstanceTimer timer = new InstanceTimer(timeoutSeconds, 0,
            new TimerCallback()
            {
                public boolean check() throws Exception
                {
                    return isInstanceStarting() ||
                           isInstanceRunning()  ||
                           isInstanceFailed();
                }
            }
        );
        timer.run(); //synchronous


        if (getInstanceStatus() == Status.kInstanceNotRunningCode)
        {
            throw new InstanceException(
                getMessages().getTimeoutStartingMessage(
                    getConfig().getDisplayName()));
        }
    }

    // method to over ride timeout time for server to started state
    // Also this value should be configurable by end user ???
    protected void waitUntilStarted() throws InstanceException
    {
        // PE set to 20 minutes
        waitUntilStarted(1200);
    }


    protected void waitUntilStarted(int timeoutSeconds) throws InstanceException
    {
        InstanceTimer timer = new InstanceTimer(timeoutSeconds, 0,
            new TimerCallback()
            {
                public boolean check() throws Exception
                {
                    return (isInstanceRunning() ||
                            isInstanceFailed()  ||
                            isInstanceNotRunning());
                }
            }
        );
        timer.run(); //synchronous
    }

    void waitUntilStopped() throws InstanceException
    {
        final int timeOutSeconds = 60;
        InstanceTimer timer = new InstanceTimer(timeOutSeconds, 0,
            new TimerCallback()
            {
                public boolean check() throws Exception
                {
                    return isInstanceNotRunning();
                }
            }
        );
        timer.run(); //synchronous
    }

    void execute(File script) throws InstanceException
    {
        try
        {
            ProcessExecutor exec = new ProcessExecutor(
                                   new String[] {script.getAbsolutePath()});
            exec.execute();
        }
        catch (Exception e)
        {
            throw new InstanceException(_strMgr.getString("procExecError"), e);
        }
    }

    ProcessExecutor execute(String[] command) throws InstanceException {
        return execute(command, null);
    }

    ProcessExecutor execute(String[] command, String[] interativeOptions) throws InstanceException
    {
        try
        {
            ProcessExecutor exec = new ProcessExecutor(command, interativeOptions);
            String nativeLauncher=System.getProperty(SystemPropertyConstants.NATIVE_LAUNCHER);
            if (nativeLauncher != null && nativeLauncher.equals("true")) {
                // native processes don't return, so don't wait
                // this follows the methodology for se, but should be revisted to make
                // sure timeouts for state transitions are reasonable
                exec.execute(false, false);
            } else {
                // expect the process to return
                exec.execute();
            }


            // this signature for execute will terminiate the process
            // if it goes on too long, reason for return signature is for SE ProcessManager watchdog
            return exec;
        }
        catch (Exception e)
        {
            throw new InstanceException(_strMgr.getString("procExecError"), e);
        }
    }

    protected PEFileLayout getFileLayout()
    {
        return super.getFileLayout(getConfig());
    }

    /**
     * Get the port on which conflict occured.
     */
    int getConflictedPort() {
        int port = 0;
        try {
            port = getRMIClient().getConflictedPort();
        }
        catch (Exception ex) {
            // Not interested!!
        }
        return port;
    }

    /**
     * AS PE will wait before exiting, until client executes this method.
     */
    void abortServer() {
        try {
            getRMIClient().triggerServerExit();
        }
        catch (Throwable t) {
            //Even if there is an error, that should not affect client performance. 
            //Ignore the error.
        }
    }

    public int getInstanceStatus() throws InstanceException
    {
        int status = Integer.MIN_VALUE;
        try {
            status = getRMIClient().getInstanceStatusCode();
        }
        catch (Exception ex) {
            /**
             * There is a timimg issue here while reading the admsn file.
             * If the client attempts to rmi the server before the admsn
             * is created an exception will be thrown. Supressing the exception
             * so that the client will be able to rmi the server in the next
             * attempt once the admsn was created.
             */
            //throw new InstanceException(ex);
            //log this??
        }
        return status;
    }

    private RMIClient rmiClient = null;

    private RMIClient getRMIClient()
    {
        if (rmiClient == null)
        {
            String stubPath = getFileLayout().getStubFile().getAbsolutePath();
            String seedPath = getFileLayout().getSeedFile().getAbsolutePath();
            rmiClient = new RMIClient(false, stubPath, seedPath);
        }
        return rmiClient;
    }

    private void setRMIClient(RMIClient client)
    {
        this.rmiClient = client;
    }
}
