/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment;

import com.sun.enterprise.deployment.Role;

/**
 * Represents a method permission. A method permission can be associated to 
 * a role, be unchecked or excluded.
 * 
 * @author  Jerome Dochez
 * @version 
 */
public class MethodPermission extends Descriptor {

    private static MethodPermission unchecked;
    private static MethodPermission excluded;
    private boolean isUnchecked = false;
    private boolean isExcluded = false;
    private Role role;
    
    /**
     * construct a new MethodPermission based on a security role
     * 
     * @param role the security role associated to the method permission
     */
    public MethodPermission(Role role) {
        this.role = role;
    }

    // We don't want uninitialized method permissins
    private MethodPermission() {
    }
    
    /**
     * @return an unchecked method permission. Methods associated with such a 
     * method permission can be invoked by anyone
     */
    public static MethodPermission getUncheckedMethodPermission() {
        if (unchecked==null) {
            unchecked = new MethodPermission();
            unchecked.isUnchecked=true;   
        }
        return unchecked;
    }
    
    /**
     * @return an ecluded method permission. Methods associated with such a 
     * method permission cannot be invoked by anyone.
     */
    public static MethodPermission getExcludedMethodPermission() {
        if (excluded==null) {
            excluded = new MethodPermission();
            excluded.isExcluded=true;   
        }
        return excluded;
    }    
    
    /**
     * @return true if the method permission is based on a security role
     */
    public boolean isRoleBased() {
        return role!=null;
    }
    
    /**
     * @return true if the method permission is unchecked
     */
    public boolean isUnchecked() {
        return isUnchecked;
    }
    
    /** 
     * @return true if the method permission is excluded
     */
    public boolean isExcluded() {
        return isExcluded;
    }
    
    /**
     * @return the security role associated with this method permission when
     * applicable (role based method permission)
     */
    public Role getRole() {
        return role;
    }
    
    // For Map storage
    public int hashCode() {
        if (role!=null)
            return role.hashCode();
        else
            return super.hashCode();
    }
    
    // for Map storage
    public boolean equals(Object other) {
	boolean ret = false;
	if(other instanceof MethodPermission) {
            MethodPermission o = (MethodPermission) other;
            if (isRoleBased()) {
	        ret = role.equals(o.getRole());
            } else {
                ret = (isExcluded == o.isExcluded()) && (isUnchecked == o.isUnchecked());
            }
	}	
	return ret;
    }    
    
    public void print(StringBuffer toStringBuffer) {
        if (isRoleBased()) {
            toStringBuffer.append(role.toString());
        } else {
            if (isExcluded) 
                toStringBuffer.append("excluded");
            else 
                toStringBuffer.append("unchecked");
        }
    }
}
        
