/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.deployment;

import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Collection;
import java.util.Map;
import java.util.HashMap;

import java.io.*;
import java.net.*;


/**
 * Information about a single webservice-description in webservices.xml
 *
 * @author Kenneth Saks
 * @author Jerome Dochez
 */

public class WebService extends Descriptor {    

    private String wsdlFileUri;

    /**
     * Derived, non-peristent location of wsdl file.
     * Only used at deployment/runtime.
     */
    private URL wsdlFileUrl;

    private String mappingFileUri;

    /**
     * Derived, non-peristent location of mapping file.
     * Only used at deployment/runtime.
     */
    private File mappingFile;

    private HashMap<String, WebServiceEndpoint> endpoints;

    // The set of web services to which this service belongs.
    private WebServicesDescriptor webServicesDesc;

    //
    // Runtime info
    //

    // Optional file URL to which final wsdl should be published.
    // This represents a directory on the file system from which deployment
    // is initiated.  URL schemes other than file: are legal but ignored.
    private URL publishUrl;
    
    /** 
     * Default constructor. 
     */
    public WebService() {
        this("");
    }

    /** 
     * copy constructor. 
     */
    public WebService(WebService other) {
        super(other);
	wsdlFileUri = other.wsdlFileUri; // String
        wsdlFileUrl = other.wsdlFileUrl;
	mappingFileUri = other.mappingFileUri; // String
	mappingFile = other.mappingFile; 
        publishUrl = other.publishUrl;
	webServicesDesc = other.webServicesDesc; // copy as-is
	if (other.endpoints != null) {
	    endpoints = new HashMap<String, WebServiceEndpoint>();
	    for (WebServiceEndpoint wsep : other.endpoints.values()) {
        	wsep.setWebService(this);
        	endpoints.put(wsep.getEndpointName(), wsep);
	    }
	} else {
	    endpoints = null;
	}
    }

    public WebService(String name) {
        setName(name);
        endpoints = new HashMap();
    }
    
    public void setWebServicesDescriptor(WebServicesDescriptor webServices) {
        webServicesDesc = webServices;
    }

    public WebServicesDescriptor getWebServicesDescriptor() {
        return webServicesDesc;
    }

    public BundleDescriptor getBundleDescriptor() {
        return webServicesDesc.getBundleDescriptor();
    }

    public boolean hasWsdlFile() {
        return (wsdlFileUri != null);
    }

    public void setWsdlFileUri(String uri) {
        wsdlFileUri = uri;
        super.changed();
    }

    public String getWsdlFileUri() {
        if (wsdlFileUri==null && wsdlFileUrl!=null) {
            // no wsdl file was provided but we generated one, we can 
            // return this path.
            return getBundleDescriptor().getWsdlDir() + "/" 
                    + (new File(wsdlFileUrl.getFile())).getName();
        }
        return wsdlFileUri;
    }

    public URL getWsdlFileUrl() {
        return wsdlFileUrl;
    }

    public void setWsdlFileUrl(URL url) {
        wsdlFileUrl = url;
        super.changed();
    }
    
    public String getGeneratedWsdlFilePath() {
        if (hasWsdlFile()) {
            String xmlDir = getBundleDescriptor().getApplication().getGeneratedXMLDirectory();
            if(!getBundleDescriptor().getModuleDescriptor().isStandalone()) {
                String uri = getBundleDescriptor().getModuleDescriptor().getArchiveUri();
                xmlDir = xmlDir + File.separator + uri.replaceAll("\\.", "_");
            }
            if(xmlDir == null) {
                return null;
            }
            return  xmlDir + File.separator + wsdlFileUri;
        } else {
            return getWsdlFileUrl().getPath();
        }
    }    
    
    public boolean hasMappingFile() {
        return (mappingFileUri != null);
    }

    public void setMappingFileUri(String uri) {
        mappingFileUri = uri;
        super.changed();
    }

    public String getMappingFileUri() {
        return mappingFileUri;
    }

    public File getMappingFile() {
        return mappingFile;
    }

    public void setMappingFile(File file) {
        mappingFile = file;
        super.changed();
    }

    public void addEndpoint(WebServiceEndpoint endpoint) {
        endpoint.setWebService(this);
        endpoints.put(endpoint.getEndpointName(), endpoint);
        super.changed();
    }

    public void removeEndpointByName(String endpointName) {
        WebServiceEndpoint endpoint = (WebServiceEndpoint)
            endpoints.remove(endpointName);
        endpoint.setWebService(null);
        super.changed();
    }

    public void removeEndpoint(WebServiceEndpoint endpoint) {
        removeEndpointByName(endpoint.getEndpointName());
        super.changed();
    }

    public Collection<WebServiceEndpoint> getEndpoints() {
        HashMap shallowCopy = new HashMap(endpoints);
        return shallowCopy.values();
    }

    public boolean hasClientPublishUrl() {
        return (publishUrl != null);
    }

    public void setClientPublishUrl(URL url) {
        publishUrl = url;
        super.changed();
    }

    public URL getClientPublishUrl() {
        return publishUrl;
    }

    public boolean hasUrlPublishing() {
        return (!hasFilePublishing());
    }

    public boolean hasFilePublishing() {
        return (hasClientPublishUrl() && 
                publishUrl.getProtocol().equals("file"));
    }

    /**
     * Select one of this webservice's endpoints to use for converting 
     * relative imports.  
     */
    public WebServiceEndpoint pickEndpointForRelativeImports() {
        WebServiceEndpoint pick = null;

        // First secure endpoint takes precedence.  
        for(WebServiceEndpoint wse : endpoints.values()) {
            if( wse.isSecure() ) {
                pick = wse;
                break;
            } 
            pick = wse;
        }
        return pick;
    }

    /**
     * Returns a formatted String of the attributes of this object.
     */
    public void print(StringBuffer toStringBuffer) {
	super.print(toStringBuffer);
        toStringBuffer.append( "\n wsdl file : ").append( wsdlFileUri);
        toStringBuffer.append( "\n mapping file ").append(mappingFileUri);
        toStringBuffer.append( "\n publish url ").append(publishUrl);
        toStringBuffer.append( "\n final wsdl ").append(wsdlFileUrl);
        toStringBuffer.append( "\n endpoints ").append(endpoints);        
    }
    
}    
