/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.enterprise.deployment.io;

import java.io.File;
import javax.xml.parsers.*;
import org.w3c.dom.*;

import com.sun.enterprise.deployment.*;
import com.sun.enterprise.deployment.xml.*;
import com.sun.enterprise.deployment.node.SaxParserHandler;
import com.sun.enterprise.deployment.node.SaxParserHandlerFactory;
import javax.enterprise.deploy.shared.ModuleType;

/**
 * Factory for DeploymentDescriptorFile implementations
 *
 * @author Jerome Dochez
 */
public class DeploymentDescriptorFileFactory {
    
    /** no instance necessary for this factory **/
    private DeploymentDescriptorFileFactory() {
    }
    
    /**
     * Creates and return an appropriate DeploymentDescriptorFile
     * capable of handling the passed descriptor
     *
     * @param descriptor used to identify the associated DeploymentDescriptorFile
     * @return the created DeploymentDescriptorFile
     */
    public static DeploymentDescriptorFile getDDFileFor(RootDeploymentDescriptor descriptor) {
        if (descriptor instanceof Application) {
            return new ApplicationDeploymentDescriptorFile();
        }
        if (descriptor instanceof EjbBundleDescriptor) {
            return new EjbDeploymentDescriptorFile();
        }
        if (descriptor instanceof WebBundleDescriptor) {
            return new WebDeploymentDescriptorFile();
        }
        if (descriptor instanceof ConnectorDescriptor) {
            return new ConnectorDeploymentDescriptorFile();
        } 
        if (descriptor instanceof ApplicationClientDescriptor) {
            return new AppClientDeploymentDescriptorFile();
        }
        return null;
    }
    
    /**
     * Creates and return an appropriate DeploymentDescriptorFile
     * capable of handling the passed descriptor
     *
     * @param descriptor used to identify the associated DeploymentDescriptorFile
     * @return the created DeploymentDescriptorFile
     */
    public static DeploymentDescriptorFile getDDFileFor(ModuleType type) {
        if (type==null) {
            return null;
        }
        if (type.equals(ModuleType.EAR)) {
            return new ApplicationDeploymentDescriptorFile();
        }
        if (type.equals(ModuleType.EJB)) {
            return new EjbDeploymentDescriptorFile();
        }
        if (type.equals(ModuleType.WAR)) {
            return new WebDeploymentDescriptorFile();
        }
        if (type.equals(ModuleType.RAR)) {
            return new ConnectorDeploymentDescriptorFile();
        } 
        if (type.equals(ModuleType.CAR)) {
            return new AppClientDeploymentDescriptorFile();
        }
        return null;
    } 
    
    /**
     * Creates and return the appropriate DeploymentDescriptorFile 
     * depending on the XML file passed.
     * 
     * @param input xml file
     * @return the DeploymentDescriptorFile responsible for handling
     * the xml file
     */
    public static DeploymentDescriptorFile getDDFileFor(File xmlFile) 
        throws Exception {
            
        // this is higly unefficient but we read the xml file as a DOM
        // tree, figure out the top xml element name and return the 
        // appropriate DeploymentDescriptorFile
        
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setValidating(false);
        DocumentBuilder docBuilder = factory.newDocumentBuilder();
        docBuilder.setEntityResolver(SaxParserHandlerFactory.newInstance());
        Document document = docBuilder.parse(xmlFile);
        Element element = document.getDocumentElement();
        if (element.getTagName().equals(ApplicationTagNames.APPLICATION)) {
            return new ApplicationDeploymentDescriptorFile();
        }
        if (element.getTagName().equals(EjbTagNames.EJB_BUNDLE_TAG)) {
            return new EjbDeploymentDescriptorFile();
        }
        if (element.getTagName().equals(WebTagNames.WEB_BUNDLE)) {
            return new WebDeploymentDescriptorFile();
        }
        if (element.getTagName().equals(ConnectorTagNames.CONNECTOR)) {
            return new ConnectorDeploymentDescriptorFile();
        } 
        if (element.getTagName().equals(ApplicationClientTagNames.APPLICATION_CLIENT_TAG)) {
            return new AppClientDeploymentDescriptorFile();
        }
        if (element.getTagName().equals(PersistenceTagNames.PERSISTENCE)) {
            return new PersistenceDeploymentDescriptorFile();
        }
        return null;
    }
}
