/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.admin.server.core.jmx;

//JDK imports
import java.lang.reflect.Method;
import java.util.Set;
import java.util.HashSet;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;

//JMX imports
import javax.management.MBeanServer;
import javax.management.QueryExp;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InvalidAttributeValueException;
import javax.management.InstanceNotFoundException;
import javax.management.AttributeNotFoundException;
import javax.management.ReflectionException;
import javax.management.MBeanException;
import javax.management.IntrospectionException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.NotificationListener;
import javax.management.NotificationFilter;
import javax.management.NotCompliantMBeanException;
import javax.management.OperationsException;
import javax.management.MBeanRegistrationException;
import javax.management.NotCompliantMBeanException;

//Admin imports
import com.sun.enterprise.admin.common.ObjectNames;

//i18n import
import com.sun.enterprise.util.i18n.StringManager;

public class MBeanServerImpl implements MBeanServer
{
    protected IRepository mMBeanRepository                        = null;

    private static boolean		kDynamicMBeansOnly		= true;
    private static String		kGetMBeanInfoMethodName	= "getMBeanInfo";
	
	private static String		kAllMBeansPattern		= "*";

	// i18n StringManager
	private static StringManager localStrings =
		StringManager.getManager( MBeanServerImpl.class );
	
    /**
        The private constructor so that only the factory method is called.
        Also makes sure that there is single instance of MBeanServer in the JVM.

        @param defaultDomainName represents the domain of the MBeanServer.
        @throws InitException in case the System Mbeans can't be registered. The
        System MBeans are ServerController and GenericConfigurator.
    */

	protected MBeanServerImpl(String defaultDomainName) throws InitException
    {
        mMBeanRepository = new DomainRepository();
    }

    /**
        The private constructor so that only the factory method is called.
        Also makes sure that there is single instance of MBeanServer in the JVM.
        The default domain of this MBeanServer will be "ias".

        @throws InitException in case the System Mbeans can't be registered. The
        System MBeans are ServerController and GenericConfigurator.
    */

        protected MBeanServerImpl() throws InitException
    {
        this(ServiceName.DOMAIN);
    }
    
    public ObjectInstance registerMBean(Object object, ObjectName objectName) 
        throws InstanceAlreadyExistsException, MBeanRegistrationException,
                NotCompliantMBeanException
    {
        if (object == null || objectName == null)
        {
            throw new IllegalArgumentException();
        }
        if (mMBeanRepository.contains(objectName))
        {
            throw new InstanceAlreadyExistsException(objectName.toString());
        }
        mMBeanRepository.add(objectName, object);
        
        return ( new ObjectInstance(objectName, object.getClass().getName()) );
    }
    
    public Object invoke(ObjectName objectName, String operationName, 
        Object[] params, String[] signature) 
        throws ReflectionException, InstanceNotFoundException, MBeanException
    {
	Object implObject = mMBeanRepository.find(objectName);
        if (implObject == null )
        {
            throw new InstanceNotFoundException(objectName.toString());
        }
        Class           implClass       = implObject.getClass();
        Introspector    reflector       = new Introspector(implClass);
        Object value = null;
        try
        {
            Method method = reflector.getMethod(operationName, signature);
            value = reflector.invokeMethodOn(method, implObject, params);
        }
        catch (java.lang.ClassNotFoundException cnfe)
        {
            throw new javax.management.ReflectionException(cnfe);
        }
        catch (java.lang.NoSuchMethodException nsme)
        {
            throw new javax.management.ReflectionException(nsme);
        }
        catch (java.lang.SecurityException se)
        {
            throw new javax.management.ReflectionException(se);
        }
        catch (java.lang.reflect.InvocationTargetException ite)
        {
            Throwable t = ite.getTargetException();
            if (t instanceof MBeanException)
            {
                throw (MBeanException)t;
            }
            else
                if (t instanceof Exception)
                {
                    throw new MBeanException((Exception) t);
                }
                else //if an error
                {
                    String msg = localStrings.getString( "admin.server.core.jmx.error_from_mbean", t.getMessage() );
                    RuntimeException re = new RuntimeException( msg );
                    throw new MBeanException(re);
                    //Do what?
                }
        }
        catch (java.lang.IllegalAccessException iae)
        {
            throw new javax.management.ReflectionException(iae);
        }
        catch (Exception e)
        {
            throw new MBeanException(e);
        }
        return value;
    }
    
    public Object getAttribute(ObjectName objectName, String attributeName) 
        throws InstanceNotFoundException, AttributeNotFoundException, 
               MBeanException, ReflectionException
    {
	    Object implObject = mMBeanRepository.find(objectName);
        if (implObject == null )
        {
            throw new InstanceNotFoundException(objectName.toString());
        }
        Class implClass = implObject.getClass();
        Object value = null;
        try
        {
            MBeanIntrospector introspector = new MBeanIntrospector(implClass);
            if (introspector.isDynamicMBean())
            {
                Method method = introspector.getMethod("getAttribute", 
                                    new String[] {"java.lang.String"});
                value = introspector.invokeMethodOn(method, implObject, 
                                                 new Object[] {attributeName});
            }
        }
        catch (javax.management.NotCompliantMBeanException e)
        {
            /*
             * This check is not necessary as a noncompliant mbean should not 
             * be allowed to register by the MBeanServer.
             */
            value = null;
        }
        catch (java.lang.ClassNotFoundException cnfe)
        {
            throw new javax.management.ReflectionException(cnfe);
        }
        catch (java.lang.NoSuchMethodException nsme)
        {
            throw new javax.management.ReflectionException(nsme);
        }
        catch (java.lang.SecurityException se)
        {
            throw new javax.management.ReflectionException(se);
        }
        catch (java.lang.reflect.InvocationTargetException ite)
        {
            Throwable t = ite.getTargetException();
            if (t instanceof javax.management.AttributeNotFoundException)
            {
                throw (javax.management.AttributeNotFoundException) t;
            }
            else if (t instanceof Exception)
            {
                throw new MBeanException((Exception) t);
            }
            else //if an error
            {
                //Do what?
            }
        }
        catch (java.lang.IllegalAccessException iae)
        {
            throw new javax.management.ReflectionException(iae);
        }
        catch (Exception e)
        {
            throw new MBeanException(e);
        }
        return value;
    }
    
    public void setAttribute(ObjectName objectName, Attribute attribute) throws 
        InstanceNotFoundException, AttributeNotFoundException, 
        MBeanException, ReflectionException, InvalidAttributeValueException
    {
		Object implObject = mMBeanRepository.find(objectName);
        if (implObject == null )
        {
            throw new InstanceNotFoundException(objectName.toString());
        }
        Class implClass = implObject.getClass();
        Object value = null;
        try
        {
            MBeanIntrospector introspector = new MBeanIntrospector(implClass);
            if (introspector.isDynamicMBean())
            {
                Method method = introspector.getMethod("setAttribute", 
                                new String[] {"javax.management.Attribute"});
                value = introspector.invokeMethodOn(method, implObject, 
                                                 new Object[] {attribute});
            }
        }
        catch (NotCompliantMBeanException e)
        {
            /*
             * This check is not necessary as a noncompliant mbean should not 
             * be allowed to register by the MBeanServer.
             */
            value = null;
        }
        catch (java.lang.ClassNotFoundException cnfe)
        {
            throw new javax.management.ReflectionException(cnfe);
        }
        catch (java.lang.NoSuchMethodException nsme)
        {
            throw new javax.management.ReflectionException(nsme);
        }
        catch (java.lang.SecurityException se)
        {
            throw new javax.management.ReflectionException(se);
        }
        catch (java.lang.reflect.InvocationTargetException ite)
        {
            Throwable t = ite.getTargetException();
            if (t instanceof javax.management.AttributeNotFoundException)
            {
                throw (javax.management.AttributeNotFoundException) t;
            }
            if (t instanceof javax.management.InvalidAttributeValueException)
            {
                throw (javax.management.InvalidAttributeValueException) t;
            }
            else if (t instanceof Exception)
            {
                throw new MBeanException((Exception) t);
            }
            else //if an error
            {
                //Do what?
            }
        }
        catch (IllegalAccessException iae)
        {
            throw new ReflectionException(iae);
        }
        catch (Exception e)
        {
            throw new MBeanException(e);
        }
    }
    
    public void unregisterMBean(ObjectName objectName) 
        throws InstanceNotFoundException, MBeanRegistrationException
    {
        if (! mMBeanRepository.contains(objectName))
        {
            throw new InstanceNotFoundException(objectName.toString());
        }
        mMBeanRepository.remove(objectName);
    }
	
	public Integer getMBeanCount()
	{
		return ( new Integer(mMBeanRepository.getTotalCount()) );
	}
	
    /**
     * Note: Currently the QueryExp is being ignored.
    */
	public Set queryMBeans(ObjectName name, QueryExp exp)
	{
		Set mbeans = new HashSet();
        /*
		if (! name.getDomain().equals(kAllMBeansPattern) )
		{
			String msg = localStrings.getString( "admin.server.core.jmx.yet_to_implement" );
			throw new RuntimeException( msg );
		}
		mbeans = mMBeanRepository.getAllMBeans();
        */
        mbeans = mMBeanRepository.query(name);
		return ( mbeans );
	}
	
	public MBeanInfo getMBeanInfo(ObjectName objName) throws
		InstanceNotFoundException, IntrospectionException, ReflectionException
	{
		MBeanInfo info = null;
		
		if (! kDynamicMBeansOnly)
		{
			String msg = localStrings.getString( "admin.server.core.jmx.assert" );
			throw new IntrospectionException( msg );
		}
		try
		{
			String operationName	= kGetMBeanInfoMethodName;
			String[] signature		= null;
			Object[] params			= null;
			info = (MBeanInfo) this.invoke(objName, operationName, params, signature);
		}
		catch(MBeanException e)
		{
			throw new IntrospectionException(e.getMessage());
		}
		
		return ( info );
	}

	public boolean isRegistered(ObjectName name)
	{
		return mMBeanRepository.contains(name);
	}

	public void addNotificationListener(ObjectName objectName, 
		NotificationListener notificationListener, 
		NotificationFilter notificationFilter, Object obj) 
		throws InstanceNotFoundException
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_addnotificationlistener" );
		throw new UnsupportedOperationException( msg );
	}

	public void addNotificationListener (ObjectName objectName, 
		ObjectName objectName1, NotificationFilter notificationFilter, 
		Object obj) throws InstanceNotFoundException
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_addnotificationlistener" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInstance createMBean (String str, ObjectName objectName) 
		throws ReflectionException, InstanceAlreadyExistsException, 
		MBeanRegistrationException, MBeanException, 
		NotCompliantMBeanException
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_creatembean" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInstance createMBean (String str, ObjectName objectName, 
		ObjectName objectName2) throws ReflectionException, 
		InstanceAlreadyExistsException, MBeanRegistrationException, javax.management.MBeanException, javax.management.NotCompliantMBeanException, javax.management.InstanceNotFoundException
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_creatembean" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInstance createMBean (String str, ObjectName objectName, 
		Object[] obj, String[] str3) 
		throws ReflectionException, InstanceAlreadyExistsException, 
		MBeanRegistrationException, MBeanException, NotCompliantMBeanException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_creatembean" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInstance createMBean (String str, ObjectName objectName, 
		ObjectName objectName2, Object[] obj, String[] str4) 
		throws ReflectionException, InstanceAlreadyExistsException, 
		MBeanRegistrationException, MBeanException, 
		NotCompliantMBeanException, InstanceNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_creatembean" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInputStream deserialize (String str, byte[] values) 
		throws OperationsException, ReflectionException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_deserialize" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInputStream deserialize (ObjectName objectName, byte[] values) 
		throws InstanceNotFoundException, OperationsException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_deserialize" );
		throw new UnsupportedOperationException( msg );
	}

	public ObjectInputStream deserialize (String str, ObjectName objectName, 
		byte[] values) throws InstanceNotFoundException, OperationsException, 
		ReflectionException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_deserialize" );
		throw new UnsupportedOperationException( msg );
	}

    public AttributeList  getAttributes(ObjectName objectName, String[] attrNames) 
        throws InstanceNotFoundException, ReflectionException //, RuntimeOperationsException
    {
        //Debug.println("*MBeanServerImpl.getAttributes");
	    Object implObject = mMBeanRepository.find(objectName);
        if (implObject == null )
        {
            throw new InstanceNotFoundException(objectName.toString());
        }
        Class implClass = implObject.getClass();
        AttributeList list = new AttributeList();
        try
        {
            MBeanIntrospector introspector = new MBeanIntrospector(implClass);
            if (introspector.isDynamicMBean())
            {
                Method method = introspector.getMethod("getAttributes", 
                                    new String[] {"[Ljava.lang.String;"});
                list = (AttributeList)introspector.invokeMethodOn(method, implObject, 
                                                 new Object[] {attrNames});
            }
        }
        catch (javax.management.NotCompliantMBeanException e)
        {
            /*
             * This check is not necessary as a noncompliant mbean should not 
             * be allowed to register by the MBeanServer.
             */
        }
        catch (java.lang.ClassNotFoundException cnfe)
        {
            throw new javax.management.ReflectionException(cnfe);
        }
        catch (java.lang.NoSuchMethodException nsme)
        {
            throw new javax.management.ReflectionException(nsme);
        }
        catch (java.lang.SecurityException se)
        {
            throw new javax.management.ReflectionException(se);
        }
        catch (java.lang.IllegalAccessException iae)
        {
            throw new javax.management.ReflectionException(iae);
//            throw new javax.management.RuntimeOperationsException(iae);
        }
        catch (java.lang.reflect.InvocationTargetException ite)
        {
            Throwable t = ite.getTargetException();
			String msg = localStrings.getString( "admin.server.core.jmx.getattributes_error_from_mbean", t.getMessage() );
            throw new RuntimeException( msg );
        }
        return list;

    }

	public String getDefaultDomain () 
	{
		return ( ServiceName.DOMAIN );
	}

	public ObjectInstance getObjectInstance (ObjectName objectName) 
		throws InstanceNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_getobjectinstance" );
		throw new UnsupportedOperationException( msg );
	}

	public Object instantiate (String str) throws ReflectionException, 
		MBeanException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_instantiate" );
		throw new UnsupportedOperationException( msg );
	}

	public Object instantiate (String str, ObjectName objectName) 
		throws ReflectionException, MBeanException, InstanceNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_instantiate" );
		throw new UnsupportedOperationException( msg );
	}

	public Object instantiate (String str, Object[] obj, String[] str2) 
		throws ReflectionException, MBeanException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_instantiate" );
		throw new UnsupportedOperationException( msg );
	}

	public Object instantiate (String str, ObjectName objectName, 
		Object[] obj, String[] str3) 
		throws ReflectionException, MBeanException, InstanceNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_instantiate" );
		throw new UnsupportedOperationException( msg );
	}

	public boolean isInstanceOf (ObjectName objectName, String str) 
	throws InstanceNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_isinstanceof" );
		throw new UnsupportedOperationException( msg );
	}

	public Set queryNames (ObjectName objectName, QueryExp queryExp) 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_querynames" );
		throw new UnsupportedOperationException( msg );
	}

	public void removeNotificationListener (ObjectName objectName, ObjectName objectName1) 
		throws InstanceNotFoundException, ListenerNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_removenotificationlistener" );
		throw new UnsupportedOperationException( msg );
	}

	public void removeNotificationListener (ObjectName objectName, 
		NotificationListener notificationListener) 
		throws InstanceNotFoundException, ListenerNotFoundException 
	{
		String msg = localStrings.getString( "admin.server.core.jmx.mbeanserver_removenotificationlistener" );
		throw new UnsupportedOperationException( msg );
	}

	public AttributeList setAttributes (ObjectName objectName, AttributeList attributeList) 
        throws InstanceNotFoundException, ReflectionException //, RuntimeOperationsException
    {
        //Debug.println("*MBeanServerImpl.setAttributes");
	    Object implObject = mMBeanRepository.find(objectName);
        if (implObject == null )
        {
            throw new InstanceNotFoundException(objectName.toString());
        }
        Class implClass = implObject.getClass();
        AttributeList list = new AttributeList();
        try
        {
            MBeanIntrospector introspector = new MBeanIntrospector(implClass);
            if (introspector.isDynamicMBean())
            {
                Method method = introspector.getMethod("setAttributes", 
                                    new String[] {list.getClass().getName()});
                list = (AttributeList)introspector.invokeMethodOn(method, implObject, 
                                                 new Object[] {attributeList});
            }
        }
        catch (javax.management.NotCompliantMBeanException e)
        {
            /*
             * This check is not necessary as a noncompliant mbean should not 
             * be allowed to register by the MBeanServer.
             */
        }
        catch (java.lang.ClassNotFoundException cnfe)
        {
            throw new javax.management.ReflectionException(cnfe);
        }
        catch (java.lang.NoSuchMethodException nsme)
        {
            throw new javax.management.ReflectionException(nsme);
        }
        catch (java.lang.SecurityException se)
        {
            throw new javax.management.ReflectionException(se);
        }
        catch (java.lang.IllegalAccessException iae)
        {
            throw new javax.management.ReflectionException(iae);
//            throw new javax.management.RuntimeOperationsException(iae);
        }
        catch (java.lang.reflect.InvocationTargetException ite)
        {
            Throwable t = ite.getTargetException();
			String msg = localStrings.getString( "admin.server.core.jmx.setattributes_error_from_mbean", t.getMessage() );
            throw new RuntimeException( msg );
        }
        return list;

    }

    /*New JMX 1.2 MBeanServer Interface Methods  - not supported. */
    public String[] getDomains() {
        final String msg = "getDomains"; 

        throw new UnsupportedOperationException(msg);
    }
    
    public void removeNotificationListener(ObjectName name,
            ObjectName listener, NotificationFilter filter, Object handback) {
        final String msg = "removeNotificationListener";

        throw new UnsupportedOperationException(msg);
    }

    public void removeNotificationListener(ObjectName name,
            NotificationListener listener, NotificationFilter filter, 
            Object handback) {
        final String msg = "removeNotificationListener";

        throw new UnsupportedOperationException(msg);
    }
    
    public ClassLoader getClassLoaderFor(ObjectName oName) {
        final String msg = "getClassLoaderFor";

        throw new UnsupportedOperationException(msg);
    }

    public ClassLoader getClassLoader(ObjectName oName) {
        final String msg = "getClassLoader";

        throw new UnsupportedOperationException(msg);
    }
    
    public javax.management.loading.ClassLoaderRepository getClassLoaderRepository() {
        final String msg = "getClassLoaderRepository";

        throw new UnsupportedOperationException(msg);
    }
            
    /*New JMX 1.2 MBeanServer Interface Methods - not supported. */
}
