/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */


package com.sun.enterprise.appclient;

import com.sun.enterprise.deployment.*;
import com.sun.enterprise.loader.InstrumentableClassLoader;
import com.sun.enterprise.server.PersistenceUnitInfoImpl;
import com.sun.logging.LogDomains;

import javax.persistence.EntityManagerFactory;
import javax.persistence.spi.PersistenceProvider;
import javax.persistence.spi.PersistenceUnitInfo;
import java.util.Collection;
import java.util.List;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class PersistenceUnitLoaderImpl {

    private static Logger logger = LogDomains.getLogger(LogDomains.ACC_LOGGER);

    private String applicationLocation;

    private InstrumentableClassLoader classLoader;

    private ApplicationClientDescriptor appClient;

    public PersistenceUnitLoaderImpl(
            String applicationLocation, InstrumentableClassLoader classLoader,
            ApplicationClientDescriptor appClient) {
        this.applicationLocation = applicationLocation;
        this.classLoader = classLoader;
        this.appClient = appClient;
    }

    /**
     * {@inheritDoc}
     */
    public void load() {
        logger.info("Loading persistence units for application: " +
                applicationLocation);
        for (PersistenceUnitDescriptor pu : findReferencedPUs()) {
            load(pu);
        }
    }

    public void unload() {
        logger.info("Unloading persistence units for application: " +
                applicationLocation);
        // step #1: unload ear level PUs
        if (appClient.getApplication() != null) {
            closeEMFs(appClient.getApplication().getEntityManagerFactories());
        }

        // step #2: unload appclient-jar level PUs
        logger.info("Unloading persistence units for appclient module called " +
                appClient.getModuleDescriptor().getArchiveUri() +
                " in application " + applicationLocation);
        closeEMFs(appClient.getEntityManagerFactories());
        logger.info("Finished unloading persistence units for application: " +
                applicationLocation);
    }

    /**
     * This method is used to find out the precise list of PUs that are
     * referenced by the appclient.
     *
     * @return list of PU that are actually referenced by the appclient.
     */
    private List<PersistenceUnitDescriptor> findReferencedPUs() {
        List<PersistenceUnitDescriptor> result =
                new ArrayList<PersistenceUnitDescriptor>();

        for(EntityManagerFactoryReferenceDescriptor emfRef :
                appClient.getEntityManagerFactoryReferenceDescriptors()) {
            final String unitName = emfRef.getUnitName();
            PersistenceUnitDescriptor pu = appClient.findReferencedPU(unitName);
            if(pu == null) {
                throw new RuntimeException("No suitable persistence unit " +
                        "called " + unitName + " found in the scope of this " +
                        "application client.");
            }
            if(pu.getTransactionType() == "JTA") {
                throw new RuntimeException("Application client is referencing " +
                        "a persistence unit called " + unitName + ", but that" +
                        "is of transaction-type JTA. Application client " +
                        "can only use RESOURCE_LOCAL persistence unit.");

            }
            if (!result.contains(pu)) {
                result.add(pu);
            }
        }
        return result;
    }

    /**
     * Loads an individual PersistenceUnitDescriptor and registers the
     * EntityManagerFactory in appropriate DOL structure.
     *
     * @param pud PersistenceUnitDescriptor to be loaded.
     */
    private void load(PersistenceUnitDescriptor pud) {
        logger.info("loading pud " + pud.getAbsolutePuRoot());
        PersistenceUnitInfo pInfo = new PersistenceUnitInfoImpl(pud,
                applicationLocation,
                classLoader);
        logger.info("PersistenceInfo for this pud is :\n" + pInfo);
        PersistenceProvider provider;
        try {
            provider =
                    PersistenceProvider.class.cast(ClassLoader.class.cast(
                            classLoader)
                    .loadClass(pInfo.getPersistenceProviderClassName())
                    .newInstance());
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        } catch (InstantiationException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
        EntityManagerFactory emf = provider.createContainerEntityManagerFactory(
                pInfo, null);
        logger.logp(Level.INFO, "PersistenceUnitLoaderImpl", "load",
                "emf = {0}", emf);
        RootDeploymentDescriptor rootDD = pud.getParent().getParent();
        if (rootDD.isApplication()) {
            Application.class.cast(rootDD).addEntityManagerFactory(
                    pInfo.getPersistenceUnitName(), pud.getPuRoot(), emf);
        } else {
            BundleDescriptor.class.cast(rootDD).addEntityManagerFactory(
                    pInfo.getPersistenceUnitName(), emf);
        }
    }

    private void closeEMFs(
            Collection<EntityManagerFactory> entityManagerFactories) {
        for (EntityManagerFactory emf : entityManagerFactories) {
            try {
                emf.close();
            } catch (Exception e) {
                logger.log(Level.WARNING, e.getMessage(), e);
            }
        }
    }
}
