/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.appclient.jws;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.ApplicationClientDescriptor;
import com.sun.enterprise.deployment.interfaces.DeploymentImplConstants;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.instance.BaseManager;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

/**
 *Records mappings of paths to content that stem from a single app client.
 *
 * @author tjquinn
 */
    
public class AppclientContentOrigin extends UserContentOrigin {

    /** records the content origin's context root */
    private String contextRoot;

    /** module descriptor for the content origin */
    protected ModuleDescriptor moduleDescriptor;
    
    /**
     *Creates a new instance of the appclient content origin.
     *@param Application object for the appclient (either an artificial wrapper
     *around a stand-alone app client or the parent ear's Application if this is 
     *an embedded app client)
     *@param ModuleDescriptor for the specific app client
     *@param the context root to be used for this app client
     */
    public AppclientContentOrigin(Application application, ModuleDescriptor moduleDescriptor, String contextRoot) {
        super(application);
        this.moduleDescriptor = moduleDescriptor;

        /*
         *Make sure the context root is not empty and has a 
         *slash at the beginning.  The default value is legal but
         *a user-supplied one might not be.
         */
        if (contextRoot.length() < 2 || ( ! contextRoot.substring(0,1).equals("/") )) {
            String regName = application.getRegistrationName();
            throw new IllegalArgumentException("Java Web Start-related context root of '" + contextRoot + "' specified for app client " + regName + " must begin with a slash and contain at least one other character");
        }
        this.contextRoot = contextRoot;
    }
    
    /**
     *Returns the origin's context root.
     *@return the string value for the context root
     */
    public String getContextRoot() {
        return contextRoot;
    }

    /**
     *Returns the display name for the app client.
     *@return the display name as a String
     */
    public String getDisplayName() {
        return moduleDescriptor.getDescriptor().getDisplayName();        
    }
    
    /**
     *Returns the descriptor for the app client.
     *@return the descriptor
     */
    public String getDescription() {
        return getApplication().getDescription();
    }
    
    /**
     *Returns the path, within the virtual namespace provided by the JWS system
     *servlet, where the app client jar file for this app client resides.
     *@return the path to the client jar file
     */
    public String getAppclientJarPath() {
        return NamingConventions.TopLevelAppclient.appclientJarPath(this);
    }
    
    
    protected String getContentKeyPrefix() {
        return NamingConventions.TopLevelAppclient.contentKeyPrefix(this);
    }
    
    /**
     *Returns the path to which requests for the virtual path for this origin
     *should be dispatched so they can be served by the system servlet.
     *@return the path to which the virtual path should be mapped
     */
    protected String getTargetPath() {
        return NamingConventions.TopLevelAppclient.actualContextRoot(application);
    }
    
    /**
     *Returns the virtual path users can use to refer to the app client from
     *this origin.  If the developer did not specify one, the default path
     *is returned.
     *@return the path by which users can access the app client
     */
    public String getVirtualPath() {
        return NamingConventions.TopLevelAppclient.virtualContextRoot(application, moduleDescriptor);
    }
    
    public String toString() {
        return super.toString() + ", context root=" + getVirtualPath() + ", module name=" + moduleDescriptor.getName();
    }
    
    public String getVendor() {
        return ((ApplicationClientDescriptor) moduleDescriptor.getDescriptor()).getJavaWebStartAccessDescriptor().getVendor();
    }

    public String getName() {
        return application.getRegistrationName();
    }
}
