/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.deployment.backend;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.enterprise.deploy.shared.ModuleType;

import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.archivist.ApplicationArchivist;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.interfaces.ClientJarMaker;
import com.sun.enterprise.deployment.RootDeploymentDescriptor;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.util.zip.ZipItem;

/**
 * This class is responsible for creating an appclient jar file that
 * will be used by the appclient container to run the appclients for
 * the deployed application.
 *
 * @author deployment dev team
 */
class ModuleClientJarMaker implements ClientJarMaker {

    protected Properties props;

    /**
     * Default constructor for this stateless object
     * @param props are the implementation properties (if any)
     */
    public ModuleClientJarMaker(Properties props) {
        this.props = props;
    }
    
    /**
     * creates the appclient container jar file
     * @param descriptor is the loaded module's deployment descriptor
     * @param source is the abstract archive for the source module deployed
     * @param target is the abstract archive for the desired appclient container jar file
     * @param stubs are the stubs generated by the deployment codegen
     * @param props is a properties collection to pass implementation parameters
     *
     * @throws IOException when the jar file creation fail
     */
    public void create(RootDeploymentDescriptor descriptor, AbstractArchive source,
        AbstractArchive target,ZipItem[] stubs, Properties props)
        throws IOException {
        create(descriptor, source, null, target, stubs, props);
    }

    /**
     * creates the appclient container jar file
     * @param descriptor is the loaded module's deployment descriptor
     * @param source is the abstract archive for the source module deployed
     * @param source2 is the abstract archive for the generated xml directory
     * @param target is the abstract archive for the desired appclient container jar file
     * @param stubs are the stubs generated by the deployment codegen
     * @param props is a properties collection to pass implementation parameters
     *
     * @throws IOException when the jar file creation fail
     */
    public void create(RootDeploymentDescriptor descriptor, AbstractArchive source,
        AbstractArchive source2, AbstractArchive target,ZipItem[] stubs, 
        Properties props) throws IOException {
        
        // in all cases we copy the stubs file in the target archive
        ClientJarMakerUtils.populateStubs(target, stubs);

        AbstractArchive appclientSource = null;
        AbstractArchive appclientSource2 = null;

        // abstract out the one and only appclient content from ear file
        if (descriptor.isApplication()) {

            // copy and expand library files here
            // @@@ need to clean up to handle manifest override problem
            // @@@ for multiple library jar files
            List<String> libraries = ClientJarMakerUtils.getLibraryEntries(
                    Application.class.cast(descriptor), source);

            for (String entryName : libraries) {
                AbstractArchive subSource = null;
                try {
                    subSource = source.getEmbeddedArchive(entryName);
                    for (Enumeration subEntries = subSource.entries();
                                subEntries.hasMoreElements();) {
                        String subEntryName = 
                                String.class.cast(subEntries.nextElement());
                        ClientJarMakerUtils.copy(
                                subSource, target, subEntryName);
                    }
                } finally {
                    if (subSource != null) {
                        source.closeEntry(subSource);
                    }
                }
            }

            //there should only be one appclient in this ear file
            for (Iterator modules = Application.class.cast(descriptor).getModules();
                    modules.hasNext();) {
                ModuleDescriptor md = ModuleDescriptor.class.cast(modules.next());
                if (md.getModuleType().equals(ModuleType.CAR)) {
                    appclientSource = source.getEmbeddedArchive(md.getArchiveUri());
                    if (source2 != null) {
                        appclientSource2 = 
                            source2.getEmbeddedArchive(md.getArchiveUri());
                    }
                    break;
                }
            }
        } else {
            appclientSource = source;
            appclientSource2 = source2;
        }

        //copy over all content of the appclient
        if (appclientSource != null) {
            ClientJarMakerUtils.populateModuleJar(
                appclientSource, appclientSource2, target);
        } else {
            //this is a workaround because otherwise the appclient jar could
            //be empty which causes problems when closing the archive (there is
            //a requirement on having at least one entry in the archive before
            //closing)
            ClientJarMakerUtils.copyDeploymentDescriptors(
                new ApplicationArchivist(), source, source2, target);
        }

        // for backward compatibility, we need to include the content
        // of the ejb module as well, since many clients currently are
        // packaged without their ejb dependencies.  We will copy the
        // .class entries only.  We copy them here _after_ the appclient
        // classes have been copied so that if any duplicates exist, 
        // the class in the original appclient jar prevails.
        if (descriptor.isApplication()) {
            for (Iterator modules = Application.class.cast(descriptor).getModules();
                    modules.hasNext();) {
                ModuleDescriptor md = ModuleDescriptor.class.cast(modules.next());
                if (md.getModuleType().equals(ModuleType.EJB)) {
                    AbstractArchive subSource = 
                        source.getEmbeddedArchive(md.getArchiveUri());
                    for (Enumeration e = subSource.entries();e.hasMoreElements();) {
                        String entryName = String.class.cast(e.nextElement());
                        if (!entryName.endsWith(".class")) {
                            continue;
                        }
                        try {
                            ClientJarMakerUtils.copy(subSource, target, entryName);
                        } catch(IOException ioe) {
                            // duplicate, we ignore
                        } finally {
                            if (subSource != null) {
                                source.closeEntry(subSource);
                            }
                        }
                    }
                }
            }
        } 
    }
}
