/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

/*
 * ModuleDeployer.java
 *
 * Created on January 3, 2002, 12:45 AM
 */

package com.sun.enterprise.deployment.backend;

import java.io.*;
import java.util.Set;
import java.util.List;
import java.util.logging.*;
import java.util.Properties;
import javax.enterprise.deploy.shared.ModuleType;

import com.sun.ejb.codegen.IASEJBCTimes;
import com.sun.enterprise.instance.ModuleEnvironment;
import com.sun.enterprise.instance.WebModulesManager;
import com.sun.enterprise.instance.InstanceEnvironment;
import com.sun.enterprise.instance.BaseManager;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.ServerTags;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.io.FileSource;
import com.sun.enterprise.util.zip.ZipItem;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.enterprise.server.Constants;
import com.sun.enterprise.loader.EJBClassPathUtils;
import com.sun.enterprise.deployment.Application;
import com.sun.enterprise.deployment.BundleDescriptor;
import com.sun.enterprise.deployment.WebServiceEndpoint;
import com.sun.enterprise.deployment.WebServicesDescriptor;
import com.sun.enterprise.deployment.WebService;
import com.sun.enterprise.deployment.deploy.shared.FileArchive;
import com.sun.enterprise.deployment.deploy.shared.FileArchiveFactory;
import com.sun.enterprise.deployment.deploy.shared.AbstractArchive;
import com.sun.enterprise.deployment.archivist.ApplicationArchivist;
import com.sun.enterprise.deployment.archivist.Archivist;
import com.sun.enterprise.deployment.phasing.DeploymentServiceUtils;

// verifier imports
import com.sun.enterprise.tools.verifier.AppVerifier;

/** ModuleDeployer is an abstract class with common methods required for deploying Web Modules
 * and EJB Modules.
 *
 * @author  bnevins
 * @version
 */

public abstract class ModuleDeployer extends Deployer
{
	abstract protected BaseManager	createConfigManager(InstanceEnvironment ienv, ModuleEnvironment menv) throws IASDeploymentException, ConfigException;
	abstract protected void			preDeploy()			throws IASDeploymentException;
	abstract protected void			deploy()			throws IASDeploymentException, ConfigException;
	abstract protected void			preRedeploy()		throws IASDeploymentException, ConfigException;
	
	///////////////////////////////////////////////////////////////////////////
	
	protected ModuleDeployer(DeploymentRequest r)  throws IASDeploymentException
	{
		super(r);
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected boolean needsStubs()
	{
		// override this for any module that needs stubs created
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected boolean needsJSPs()
	{
		// override this for any module that works with generated JSPs
		return false;
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected final File getModuleDir()
	{
		return moduleDir;
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	public void doRequest() throws IASDeploymentException
	{
		doRequestPrepare();
		doRequestFinish();
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	public void doRequestPrepare() throws IASDeploymentException
	{
		try
		{
			begin();
		}
		catch(Exception e)
		{
			if(shouldRollback)
				rollback();
			
			String msg = localStrings.getString(
			"enterprise.deployment.backend.dorequest_exception");
			logger.log(Level.WARNING, msg, e);
			throw new IASDeploymentException(msg, e);
		}
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	public void doRequestFinish() throws IASDeploymentException
	{
		try
		{
			if(request.isDeploy())
			{
                                // retrieve the J2EECPhase deployment status
                                DeploymentStatus J2EECPhaseStatus =
                                    request.getCurrentDeploymentStatus();

				beginFinish();
				preDeploy();

                                // create a DeploymentStatus for runEJBC stage
                                // it is a substage of J2EECPhase status
                                DeploymentStatus runEJBCStatus =
                                    new DeploymentStatus(J2EECPhaseStatus);
                                request.setCurrentDeploymentStatus(
                                    runEJBCStatus);
				deploy();
                                register();

                                // create a DeploymentStatus for postDeploy
                                // stage, it is a substage of J2EECPhase status
                                DeploymentStatus postDeployStatus =
                                    new DeploymentStatus(J2EECPhaseStatus);
                                request.setCurrentDeploymentStatus(
                                    postDeployStatus);
				postDeploy();

				generatePolicy();
			}
			else if(request.isReDeploy())
			{
                                // retrieve the J2EECPhase deployment status
                                DeploymentStatus J2EECPhaseStatus =
                                    request.getCurrentDeploymentStatus();

				beginFinish();
				preRedeploy();

                                // create a DeploymentStatus for runEJBC stage
                                // it is a substage of J2EECPhase status
                                DeploymentStatus runEJBCStatus =
                                    new DeploymentStatus(J2EECPhaseStatus);
                                request.setCurrentDeploymentStatus(
                                    runEJBCStatus);
				redeploy();
                                register();

                                // create a DeploymentStatus for postDeploy
                                // stage, it is a substage of J2EECPhase status
                                DeploymentStatus postDeployStatus =
                                    new DeploymentStatus(J2EECPhaseStatus);
                                request.setCurrentDeploymentStatus(
                                    postDeployStatus);
				postRedeploy();

				generatePolicy();
			}
			else if(request.isUnDeploy())
			{
				beginFinish();
				preundeploy();
				undeploy();
				removePolicy();
			}
			else
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.unknown_deployment_command" );
				throw new IASDeploymentException( msg );
			}
		}
		catch(Exception e)
		{
			if(shouldRollback)
				rollback();
			
			String msg = localStrings.getString(
			"enterprise.deployment.backend.dorequest_exception");
			logger.log(Level.FINE, msg, e);
                        if (e instanceof IASDeploymentException) {
                            throw (IASDeploymentException) e;
                        } else {
                            throw new IASDeploymentException(msg, e);
                        }
		}
		finally
		{
			finish();
		}
	}
	
	// this method is overridden from Deployer.  It is final so you know FOR SURE
	// that it isn't overridden by any subclasses.
	protected final void begin() throws IASDeploymentException
	{
		super.begin();
		
		// get environment object refs
		
		InstanceEnvironment instanceEnv = getInstanceEnv();
		moduleEnv						= request.getModuleEnv();
		moduleName						= request.getName();
		
		// check them
		
		if(moduleEnv == null)
		{
			String msg = localStrings.getString("enterprise.deployment.backend.null_moduleenvironment");
			throw new IASDeploymentException(msg);
		}
		
		try
		{
			moduleEnv.verify();
			modulesMgr = createConfigManager(instanceEnv, moduleEnv);
			setDeployCommand();
			
			if(request.isReDeploy())
			{
                                originalModuleDir = new File(DeploymentServiceUtils.getLocation(moduleName, request.getType()));
				unregister();
                                removePolicy();
			}
		}
		catch(Exception e)
		{
			throw new IASDeploymentException(e);
		}
		
		shouldRollback = true;
		
		// for redeploy -- when we get to doRequestFinish() -- the module will NOT be registered
		// any longer.
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	// bnevins -- added August 2003 to allow for the MBean to unload the module between
	// begin() and now.
	
	private void beginFinish() throws IASDeploymentException
	{
		setDirs();
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private void setDirs() throws IASDeploymentException
	{
		assert modulesMgr != null;
		assert moduleName != null;
		assert moduleEnv != null;
		
		if(request.isDeploy())
			setDirsDeploy();
		else if(request.isReDeploy())
			setDirsReDeploy();
		else if(request.isUnDeploy())
			setDirsUnDeploy();
		else
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.deployment_type_error" );
			throw new IASDeploymentException( msg );
		}
		
		request.setDeployedDirectory(moduleDir);
		request.setJSPDirectory(jspDir);
		request.setStubsDirectory(stubsDir);
		request.setGeneratedXMLDirectory(xmlDir);
	}
	
    	/**
	 * @return the module classpath
	 */
	protected List getModuleClasspath(Archivist archivist,
                AbstractArchive archive) throws IASDeploymentException {
	    try {
		String location = request.getDeployedDirectory().getAbsolutePath();
		return EJBClassPathUtils.getModuleClasspath(request.getName(), location, this.getManager());
	    } catch(Exception e) {
		throw new IASDeploymentException(e);
	    }
	}
        
	///////////////////////////////////////////////////////////////////////////
	
	private void setDirsDeploy() throws IASDeploymentException
	{
		if(isRegistered())
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.deploy_error_module_exists" );
			throw new IASDeploymentException( msg );
		}
                xmlDir = new File(moduleEnv.getModuleGeneratedXMLPath());
                jwsDir = new File(moduleEnv.getJavaWebStartPath());
		
		if(needsStubs())
		{
			stubsDir = new File(moduleEnv.getModuleStubPath());
		}
		else
			stubsDir = null;
		
		if(needsJSPs())
		{
			assert (modulesMgr instanceof WebModulesManager);
			jspDir = new File(moduleEnv.getModuleJSPPath());
		}
		if(isArchive())
		{
			File parent = new File(getInstanceEnv().getModuleRepositoryPath());
                        moduleDir = new File(parent, moduleName);
                        moduleDir.mkdirs();
		}
		else if(isDirectory())
		{
			FileSource fileSource = request.getFileSource();
			
			if(fileSource == null || !fileSource.exists())
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.file_source_does_not_exist",
				fileSource );
				throw new IASDeploymentException( msg );
			}
			
			moduleDir = fileSource.getFile();
			
			if(!FileUtils.safeIsDirectory(moduleDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.deployment_directory_does_not_exist",
				moduleDir.getPath() );
				throw new IASDeploymentException( msg );
			}
		}
		else
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.deployment_not_dir_or_archive" );
			throw new IASDeploymentException( msg );
		}
	}
        
	/**
	 * @return a fully initialized and validated deployment descriptors for this 
	 * deployment request.
	 */
	protected Application loadDescriptors() throws IASDeploymentException {
            Application app = super.loadDescriptors();
            (new com.sun.enterprise.webservice.WsUtil()).genWSInfo(app, request);
            return app;
        }
                
	///////////////////////////////////////////////////////////////////////////
	
	private void setDirsUnDeploy() throws IASDeploymentException
	{
		// Use the already registered location
		try
		{
			if(!isRegistered())
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.undeploy_error_module_not_registered" );
				throw new IASDeploymentException( msg );
			}
			
			moduleDir	= new File(DeploymentServiceUtils.getLocation(moduleName, request.getType()));
                        xmlDir = new File(modulesMgr.getGeneratedXMLLocation(moduleName));
                        jwsDir = new File(moduleEnv.getJavaWebStartPath());
			stubsDir = null;
			jspDir   = null;
			
			if(needsStubs())
				stubsDir	= new File(modulesMgr.getStubLocation(moduleName));
			if(needsJSPs())
			{
				assert (modulesMgr instanceof WebModulesManager);
				WebModulesManager mgr = (WebModulesManager)modulesMgr;
				jspDir = new File(mgr.getJSPLocation(moduleName));
			}
		}
		catch(Exception e)
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.error_getting_module_directory",
			e );
			throw new IASDeploymentException( msg );
		}
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private void setDirsReDeploy() throws IASDeploymentException
	{
		if(!wasReg)
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.redeploy_error_module_not_registered" );
			throw new IASDeploymentException( msg );
		}
		
                xmlDir = new File(modulesMgr.getGeneratedXMLLocation(moduleName));
                jwsDir = new File(moduleEnv.getJavaWebStartPath());

		// let's get the easy stuff out of the way first -- stubs & jsps
		stubsDir = null;
		jspDir   = null;
		
		if(needsStubs()) {
			stubsDir	= new File(modulesMgr.getStubLocation(moduleName));
                }
		if(needsJSPs())
		{
			assert (modulesMgr instanceof WebModulesManager);
			WebModulesManager mgr = (WebModulesManager)modulesMgr;
			jspDir = new File(mgr.getJSPLocation(moduleName));
		}
		
		if(isArchive())
		{
			// be sure we have the original deployed directory
			if(!FileUtils.safeIsDirectory(originalModuleDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.modulesmanager_error_getting_module_location",
				moduleName );
				throw new IASDeploymentException(msg);
			}
			
			moduleDir		= originalModuleDir;
			oldModuleDir	= new File(moduleDir.getAbsolutePath() + "_old");
			
			
			
			// wipe it out if it happens to already exist
			FileUtils.whack(oldModuleDir);
			
			
			// note -- java does NOT change the path of moduleDir, it just renames the file on disk!
			// i.e. "moduleDir" will still be pointing at the same file after the call as before the call.
			moduleDirWasRenamed = FileUtils.renameFile(moduleDir,oldModuleDir);
			
			
			if(!moduleDirWasRenamed)
				
				
			{
				String msg = localStrings.getString("enterprise.deployment.backend.directory_rename_error",
				moduleDir.getAbsolutePath(), oldModuleDir.getAbsolutePath());
				throw new IASDeploymentException(msg);
			}
		}
		else if(isDirectory())
		{
			FileSource fileSource = request.getFileSource();
			
			if(!fileSource.exists())
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.file_source_does_not_exist",
				fileSource );
				throw new IASDeploymentException( msg );
			}
			
			assert fileSource.isDirectory();
			moduleDir = fileSource.getFile();
			
			oldModuleDir = new File(moduleEnv.getModuleBackupPath());
			
			if (!FileUtils.safeIsDirectory(oldModuleDir))
			{
				oldModuleDir = null;
			}
		}
		else
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.redeployment_not_dir_or_archive" );
			throw new IASDeploymentException( msg );
		}
		
		moduleDir.mkdirs();
	}

    /**
    * Attempt to delete the deployed directory-tree.
    * <p> This method call is <b>guaranteed</b> to never throw any kind of Exception
    */
    public void cleanup_internal()
    {
                try
                {
                        if(request.isUnDeploy())
                        {
                                if(isMaybeCMPDropTables)
                                        dropTables();

                                liquidate();
                        }

                        // note: oldModulDir will be pointing at the just-copied backup dir
                        // if this is a directory-redeploy!!
                        else if(request.isReDeploy() && isArchive() && oldModuleDir != null)
                        {
                                FileUtils.whack(oldModuleDir);
                        }

                        // nothing to do for Deploy
                }
                catch(Exception e)
                {
                        logger.warning("Exception caught and ignored in cleanup_internal()");
                }
    }

	
	///////////////////////////////////////////////////////////////////////////
	
	protected void liquidate() throws IASDeploymentException, ConfigException
	{
		assert !request.isReDeploy();	// only can be called for DEPLOY and UNDEPLOY
		assert moduleDir != null;
		
		boolean shouldDeleteModuleFiles = false;
		
		if(request.isDeploy() && isArchive())
		{
			shouldDeleteModuleFiles = true;
		}
		else if(request.isUnDeploy() && !(DeploymentServiceUtils.isDirectoryDeployed(request.getName(), request.getType()) || request.isReload()) )
		{
			shouldDeleteModuleFiles = true;
		}
		
		if(shouldDeleteModuleFiles)
		{
			if(FileUtils.safeIsDirectory(moduleDir))
			{
				FileUtils.whack(moduleDir);
				logger.fine("Deleted module Directory: " + moduleDir.getPath());
			}
			else
			{
				logger.warning("Can't delete module Directory -- it isn't a directory: "
				+ FileUtils.safeGetCanonicalPath(moduleDir));
			}
		}
		else
		{
			logger.fine("Did NOT delete module Directory (Directory-Deployment): " + moduleDir.getPath());
		}
		
		DeleteOrKeepFailedStubs(stubsDir);
		
		if(FileUtils.safeIsDirectory(jspDir))
		{
			FileUtils.whack(jspDir);
		}

		if(FileUtils.safeIsDirectory(xmlDir))
		{
			FileUtils.whack(xmlDir);
		}
                if(FileUtils.safeIsDirectory(jwsDir))
                {
                        FileUtils.whack(jwsDir);
                }
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	
	protected void preundeploy() throws IASDeploymentException, ConfigException
	{
                // setup the cmp stuff for undeployment...

                if(getRequest().isMaybeCMPDropTables())
                {
                        isMaybeCMPDropTables = true;
                }
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private void undeploy() throws IASDeploymentException, ConfigException
	{
		if(!isRegistered())
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.undeploy_error_module_not_registered" );
			throw new IASDeploymentException( msg );
		}
		
		try
		{
			unregister();
		}
		catch(ConfigException e)
		{
			String msg = localStrings.getString(
			"enterprise.deployment.backend.config_exception_on_remove",
			moduleName, e );
			throw new IASDeploymentException( msg, e);
		}
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private void rollbackUndeploy() throws ConfigException
	{
		// nothing to do -- the only thing that can get us here is an error
		// trying to unregister()...
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected void  postDeploy() throws IASDeploymentException, ConfigException
	{
                // save the object type in optional attributes 
                Properties optionalAttributes = request.getOptionalAttributes();
                if (optionalAttributes == null) {
                    optionalAttributes = new Properties();
                }
                String resourceType = getResourceType(moduleDir);
                if(resourceType != null) {
                    optionalAttributes.setProperty(ServerTags.OBJECT_TYPE,
                        resourceType);
                }
    
                handlePostDeployEvent();
          }

    /**
    * Called from postDeploy and postRedeploy to process extra events.
    * Currently only cmp is registered for these events.
    */
    protected void handlePostDeployEvent() throws IASDeploymentException
    {
                DeploymentEventInfo info = getEventInfo();
                DeploymentEvent ev	 = new DeploymentEvent(
                DeploymentEventType.POST_DEPLOY, info);
                DeploymentEventManager.notifyDeploymentEvent(ev);

     }
	
   /** 
    * Create DeploymentEvent info instance.
    * @return DeploymentEventInfo 
    */
    protected DeploymentEventInfo getEventInfo() throws IASDeploymentException
    {	
		return new DeploymentEventInfo(moduleDir, stubsDir, oldStubsDir,
			request.getDescriptor(), getRequest());
    }

	///////////////////////////////////////////////////////////////////////////
	
	protected final void  rollbackDeploy() throws IASDeploymentException
	{
		// pretty simple -- remove the module if it actually got registered (unlikely)
		// and hose-down the deployed files
		try
		{
			if(wasRegisteredThisSession)
			{
				// i.e. the error came AFTER registering.
				// this will be false if, say, we are trying to deploy an already deployed module
				unregister();
			}
			
			liquidate();
		}
		catch(ConfigException e)
		{
			// can't do anything -- we are already in an error situation!
		}
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected final void redeploy() throws IASDeploymentException, ConfigException
	{
		deploy();	// same thing!
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	private void rollback()
	{
		// IMPORTANT: WE ARE BEING CALLED FROM INSIDE A CATCH BLOCK --
		// THUS YOU BETTER NOT LET ANY EXCEPTIONS OUT OF HERE!!!
		
		try
		{
			if(request.isDeploy())
				rollbackDeploy();
			else if(request.isReDeploy())
			{
				// bnevins 9/8/2003
				// no rollbacks for dir-redeploy!
				if(request.isArchive())
					rollbackRedeploy();
				else
					request.setReRegisterOnFailure(false);
			}
			else if(request.isUnDeploy())
				rollbackUndeploy();
		}
		catch(Throwable t)
		{
			// can't do anything!
		}
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected void register() throws IASDeploymentException, ConfigException
	{
		modulesMgr.registerDescriptor(moduleName, request.getDescriptor());
		wasRegisteredThisSession = true;
	}
		
	///////////////////////////////////////////////////////////////////////////
	
	protected void liquidateModuleDirAndStubsDirIfTheyHappenToExist() throws IASDeploymentException
	{
		if(request.isArchive() && FileUtils.safeIsDirectory(moduleDir))
		{
			FileUtils.whack(moduleDir);
			
			if(FileUtils.safeIsDirectory(moduleDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.deploy_error_dir_is_locked",
				"Module", moduleDir.getAbsolutePath());
				throw new IASDeploymentException( msg );
			}
		}
		
		
		if(FileUtils.safeIsDirectory(stubsDir))
		{
			FileUtils.whack(stubsDir);
			
			if(FileUtils.safeIsDirectory(stubsDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.deploy_error_dir_is_locked",
				"Stubs", stubsDir.getAbsolutePath());
				throw new IASDeploymentException( msg );
			}
		}
		
		if(FileUtils.safeIsDirectory(jspDir))
		{
			FileUtils.whack(jspDir);
			
			if(FileUtils.safeIsDirectory(jspDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.deploy_error_dir_is_locked",
				"JSP", jspDir.getAbsolutePath());
				throw new IASDeploymentException( msg );
			}
		}

                if(FileUtils.safeIsDirectory(xmlDir))
                {               
                        FileUtils.whack(xmlDir);
                                
                        if(FileUtils.safeIsDirectory(xmlDir))
                        {       
                                String msg = localStrings.getString(
                                "enterprise.deployment.backend.deploy_error_dir_is_locked",
                                "XML", xmlDir.getAbsolutePath());
                                throw new IASDeploymentException( msg );
                        }
                }
                if(FileUtils.safeIsDirectory(jwsDir))
                {
                        FileUtils.whack(jwsDir);

                        if(FileUtils.safeIsDirectory(jwsDir))
                        {
                                String msg = localStrings.getString(
                                "enterprise.deployment.backend.deploy_error_dir_is_locked",
                                "JWS", jwsDir.getAbsolutePath());
                                throw new IASDeploymentException( msg );
                        }
                }
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected void setOldDirs() throws IASDeploymentException
	{
		assert moduleDir != null;
		assert moduleEnv != null;
		assert request.isReDeploy();	// this must only be called for Redeploy!!
		
		// note: oldModuleDir is already set (or null) from setDirsRedeploy()
		
		if(needsStubs() && stubsDir != null && stubsDir.exists() && stubsDir.isDirectory())
		{
			oldStubsDir = new File(stubsDir.getPath() + "_old");
			
			if(oldStubsDir.exists())
			{
				// what if there is a regular file with the name we need?  Let's check...
				
				FileUtils.whack(oldStubsDir);
			}
			
			if(!FileUtils.renameFile(stubsDir,oldStubsDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.directory_rename_error",
				stubsDir.getPath(), oldStubsDir.getPath() );
				throw new IASDeploymentException( msg );
			}
		}
		
		if(needsJSPs() && FileUtils.safeIsDirectory(jspDir))
		{
			oldJSPDir = new File(jspDir.getPath() + "_old");
			
			if(oldJSPDir.exists())
			{
				// what if there is a regular file with the name we need?  Let's check...
				
				FileUtils.whack(oldJSPDir);
                        }
			
			if(!FileUtils.renameFile(jspDir,oldJSPDir))
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.directory_rename_error",
				jspDir.getPath(), oldJSPDir.getPath() );
				throw new IASDeploymentException( msg );
			}
		}

                if(FileUtils.safeIsDirectory(xmlDir))
                {
                        oldXMLDir = new File(xmlDir.getPath() + "_old");

                        if(oldXMLDir.exists())
                        {
                                // what if there is a regular file with the name we need?  Let's check...

                                FileUtils.whack(oldXMLDir);
                        }

                        if(!FileUtils.renameFile(xmlDir,oldXMLDir))
                        {
                                String msg = localStrings.getString(
                                "enterprise.deployment.backend.directory_rename_error",
                                xmlDir.getPath(), oldXMLDir.getPath() );
                                throw new IASDeploymentException( msg );
                        }
                }
                if (FileUtils.safeIsDirectory(jwsDir)) {
                    // warning if the directory exist and has some contents
                    if (jwsDir.list().length > 0) {
                        DeploymentStatus jwsStatus = new DeploymentStatus(
                            request.getCurrentDeploymentStatus());
                        jwsStatus.setStageStatus(DeploymentStatus.WARNING);
                        jwsStatus.setStageStatusMessage(localStrings.getString( 
                            "enterprise.deployment.backend.jws_redeploy", 
                            jwsDir.getPath()));                               
                    }

                    oldJWSDir = new File(jwsDir.getPath() + "_old");
                                                                                   
                    if(oldJWSDir.exists())
                    {
                        // what if there is a regular file with the name we need?  
                        // Let's check...
                        FileUtils.whack(oldJWSDir);
                    }
                                                                                   
                    if(!FileUtils.renameFile(jwsDir, oldJWSDir))
                    {
                        String msg = localStrings.getString(
                            "enterprise.deployment.backend.directory_rename_error",
                            jwsDir.getPath(), oldJWSDir.getPath() );
                            throw new IASDeploymentException( msg );
                    }

                }
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected boolean isRegistered() throws IASDeploymentException
	{
		return DeploymentServiceUtils.isRegistered(moduleName, request.getType());
	}
	
	///////////////////////////////////////////////////////////////////////////
	protected boolean isSystem() throws IASDeploymentException
	{
		try
		{
                        boolean isSystem;
			isSystem = DeploymentServiceUtils.isSystem(moduleName, request.getType());
			boolean guardSystemApp = !(Boolean.valueOf(System.getProperty(Constants.ALLOW_SYSAPP_DEPLOYMENT, "false")).booleanValue());
			if(isSystem && guardSystemApp)
				return true;
			else
				return false;
		}catch(Exception e)
		{
			throw new IASDeploymentException(e);
		}
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	protected void setShared(boolean what) throws ConfigException
	{
	/* WBN 510-02
	 * This feature wasn't implemented by core server.
	 * I'm commenting-out the call but leaving everything else
	 * in place -- since it will probably be added in a later
	 * version
	 */
		//modulesMgr.setShared(moduleName, what);
	}
	
	///////////////////////////////////////////////////////////////////////////
	
	
	protected void unregister() throws ConfigException
	{
		modulesMgr.unregisterDescriptor(moduleName);
	}

        ///////////////////////////////////////////////////////////////////////////

        protected void postRedeploy() throws IASDeploymentException, ConfigException
        {
                handlePostDeployEvent();

                if(FileUtils.safeIsDirectory(oldStubsDir))
                {               
                        FileUtils.whack(oldStubsDir);
                }       

                if(FileUtils.safeIsDirectory(oldJSPDir))
                {
                        FileUtils.whack(oldJSPDir);
                }

                if(FileUtils.safeIsDirectory(oldXMLDir))
                {
                        FileUtils.whack(oldXMLDir);
                }

                if(FileUtils.safeIsDirectory(oldJWSDir))
                {
                        FileUtils.whack(oldJWSDir);
                }
        }

	
        ///////////////////////////////////////////////////////////////////////////
        
        protected void rollbackRedeploy() throws IASDeploymentException
        {       
                if(isArchive() && wasModuleDirRenamed())
                {       
                        if(FileUtils.safeIsDirectory(moduleDir))
                                FileUtils.whack(moduleDir);
                        
                        if(FileUtils.safeIsDirectory(oldModuleDir))
                                oldModuleDir.renameTo(moduleDir);
                }
                
                if(FileUtils.safeIsDirectory(oldStubsDir))
                {
                        DeleteOrKeepFailedStubs(stubsDir);
                        oldStubsDir.renameTo(stubsDir);
                }     

                if(FileUtils.safeIsDirectory(jspDir))
                        FileUtils.whack(jspDir);
                
                if(FileUtils.safeIsDirectory(oldJSPDir))
                        oldJSPDir.renameTo(jspDir);
                
                if(FileUtils.safeIsDirectory(xmlDir))
                        FileUtils.whack(xmlDir);
                
                if(FileUtils.safeIsDirectory(oldXMLDir))
                       oldJSPDir.renameTo(xmlDir);

                if(FileUtils.safeIsDirectory(oldJWSDir))
                       oldJWSDir.renameTo(jwsDir);

                try
                {   
                    register();
                 
                }
                catch(Exception e)
                {       
                        throw new IASDeploymentException(e);
                }
        }


	
	///////////////////////////////////////////////////////////////////////////
	
	protected final boolean wasModuleDirRenamed()
	{
		// this is of interest for redeploy rollback.  It needs to know whether
		// or not to rename xxx_old back to xxx.  If it never happened the rollback
		// would not work.
		// bnevins 8/03
		
		return moduleDirWasRenamed;
	}
	
	/**
	 * Runs the verifier on this module if verification is ON in
	 * the deployment request.
	 *
	 * @throws  IASDeploymentException  if an error found in this module
	 *                                  after verification
	 */
    protected void runVerifier() throws IASDeploymentException
    {
        if (request.isVerifying()) {
            try {
                String archive = request.getDeployedDirectory().getCanonicalPath();
                File jspOutDir = (request.getPrecompileJSP())? jspDir:null;
                new AppVerifier().verify(request.getDescriptor(),
                        (new FileArchiveFactory()).openArchive(archive),
                        request.getCompleteClasspath(),
                        jspOutDir);
            } catch (Exception e) {
                String msg = localStrings.getString(
                        "enterprise.deployment.backend.verifier_error");
                throw new IASDeploymentException(msg);
            }
        }
    }
	
	///////////////////////////////////////////////////////////////////////////
	
	private void setDeployCommand() throws IASDeploymentException
	{
		boolean isReg = isRegistered();
		boolean isSystem = false;
		wasReg = isReg;
		
		if(isReg)
		{
			isSystem = isSystem();
		}
		
		if(request.isUnDeploy())
		{
			if(!isReg)
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.undeploy_error_module_not_registered" );
				throw new IASDeploymentException( msg );
			}
			else if(isSystem)
			{
				String msg = localStrings.getString(
				"enterprise.deployment.backend.undeploy_error_module_is_a_system_resource" );
				throw new IASDeploymentException( msg );
			}
		}
		
		else if(request.isDeploy())
		{
			if(isReg)
			{
				if(isSystem)
				{
					String msg = localStrings.getString(
					"enterprise.deployment.backend.redeploy_error_module_is_a_system_resource" );
					throw new IASDeploymentException( msg );
				}
				
				//WBN 2/27/02 - By definition -- a REDEPLOY is now a forced DEPLOY if
				// the Module already exists
				if(request.isForced())
				{
					request.setCommand(DeploymentCommand.REDEPLOY);
				}
				else
				{
					String msg = localStrings.getString(
					"enterprise.deployment.backend.deploy_error_module_exists" );
					throw new IASDeploymentException( msg );
				}
			}
			else
			{
				// isReg is false.  This means that it isn't registered as the current type
				// of module.  But we might be clashing with a registered module of a different flavor.
				// E.g. this may be an ejb deployment and there is a web module already deployed
				// with the same name.
				// this will throw an IASDeploymentException if it is registered to another type...
				checkRegisteredAnywhereElse(moduleName);
			}
		}
	}
	
	/**
	 * runs the ejbc compiler for the deployable module
	 */
	protected ZipItem[] runEJBC() throws IASDeploymentException
	{
		ZipItem[] clientStubs = null;
		
		try
		{
			
			// ejbc timing info
			IASEJBCTimes timing	 = new IASEJBCTimes();
			
			EJBCompiler compiler = new EJBCompiler
			(
			moduleName,
			moduleDir,
			oldModuleDir,
			stubsDir,
			oldStubsDir,
			getManager(),
			request,
			timing
			);
			
			// runs ejbc
			clientStubs = compiler.compile();
			
			// add the ejbc timing info to deployment time
			addEJBCTime(timing);
			
		} catch (Exception e)
		{
			logger.log( Level.WARNING,
			"enterprise.deployment_ejbc_error", e );
			String msg = localStrings.getString(
			"enterprise.deployment.backend.ejbc_error" );
			throw new IASDeploymentException(msg, e);
		}
		
		// returns the client stubs or an empty array if no stubs
		return clientStubs;
	}
	
	/**
	 * @return the BaseManager implementation for this deployer
	 */
	protected  BaseManager getManager()
	{
		return modulesMgr;
	}
    
   ///////////////////////////////////////////////////////////////////////////

    /**
    * Call into CMP to drop tables.  This is called just before files are deleted as part
    * of the cleanup() call.  We do it very late in the process since dropping tables
    * can't be rolled-back.  Thus all of the other steps that can have errors that require a
    * rollback have already been done successfully - or we wouldn't be here.
    * bnevins April 2003
    * <p> This method call is <b>guaranteed</b> to never throw any kind of Exception
    */
    protected void dropTables()
    {
                assert isMaybeCMPDropTables; // programmer error if this is false!
                                                                                
                try
                {
                    Application moduleDD;
                    if (FileUtils.safeIsDirectory(xmlDir)) {
                        moduleDD =
                            DeploymentUtils.getModuleDescriptor(xmlDir.getAbsolutePath());
                            FileArchive archive = new FileArchive();
                            archive.open(moduleDir.getAbsolutePath());
                            try {
                                ApplicationArchivist.
                                    readPersistenceDeploymentDescriptorsRecursively(
                                        archive, moduleDD);
                            } finally {
                                archive.close();
                            }
                    } else {
                        moduleDD =
                            DeploymentUtils.getModuleDescriptor(moduleDir.getAbsolutePath());
                    }
                    moduleDD.setRegistrationName(moduleName);
                                                                                
                    DeploymentEventInfo info = new DeploymentEventInfo(moduleDir, stubsDir, oldStubsDir, moduleDD, request);
                                                                                
                        DeploymentEvent ev       = new DeploymentEvent(DeploymentEventType.PRE_UNDEPLOY, info);                                                 
                        DeploymentEventManager.notifyDeploymentEvent(ev);
                }
                catch(Throwable t)
                {
                        // yes we are swallowing all possible errors from outside this package!   
                        logger.log( Level.WARNING,

                        "enterprise.deployment_pre_undeploy_event_error", t);
                }
    }                                                                                 

    public void removePolicy() throws IASDeploymentException {
        //no op
    }
	
	///////////////////////////////////////////////////////////////////////////
	
	protected		BaseManager			modulesMgr						= null;
	protected		String				moduleName						= null;
	protected		File				moduleDir						= null;
	protected		File				stubsDir						= null;
	protected		File				jspDir							= null;
	protected		File				xmlDir							= null;
	protected		File				jwsDir							= null;
	protected		File				originalModuleDir				= null;
	protected		String				originalContextRoot				= null;
	protected		File				oldModuleDir					= null;
	protected		File				oldStubsDir						= null;
	protected		File				oldJSPDir						= null;
	protected		File				oldXMLDir						= null;
	protected		File				oldJWSDir						= null;
	protected		ModuleEnvironment	moduleEnv						= null;
	protected		String				oldRegisteredLocation			= null;
        protected boolean                                     isMaybeCMPDropTables    = false;
	private			boolean				wasReg							= false;
        private                 boolean                         wasRegisteredThisSession                = false;
	private			boolean				shouldRollback					= false;
	private			boolean				moduleDirWasRenamed				= false;
	private static	StringManager		localStrings					= StringManager.getManager( ModuleDeployer.class );
}
