/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.resource;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.HashMap;
import javax.naming.Context;
import javax.naming.Name;
import javax.naming.NamingException;
import javax.naming.NameNotFoundException;
import javax.naming.RefAddr;
import javax.naming.Reference;
import javax.naming.spi.ObjectFactory;
import javax.naming.spi.InitialContextFactory;


/**
 * A proxy object factory for an external JNDI factory
 */
public class JndiProxyObjectFactory implements ObjectFactory {

    // for every external-jndi-resource there is an InitialContext
    // created from the factory and environment properties
    private static Hashtable contextMap = new Hashtable();

    protected static Context removeInitialContext(String jndiName) {
        return (Context) contextMap.remove(jndiName);
    }

    /**
     * load the context factory
     */
    private Context loadInitialContext(String factoryClass, Hashtable env) {
	Object factory = ResourceInstaller.loadObject(factoryClass);
        if (factory == null) {
        	System.err.println("Cannot load external-jndi-resource " +
                                   "factory-class '" + factoryClass + "'");
                return null;
        } else if (! (factory instanceof
                            javax.naming.spi.InitialContextFactory)) {

                System.err.println("external-jndi-resource factory-class '"
                                  + factoryClass + "' must be of type "
                                  + "javax.naming.spi.InitialContextFactory");
                return null;
        }

        Context context = null;
        try {
        	context = ((InitialContextFactory)factory).getInitialContext(env);
        } catch (NamingException ne) {
          	System.err.println("Exception thrown creating initial context " +
                                   "for external JNDI factory '" +
                                   factoryClass + "' " + ne.getMessage());
        }

	return context;
    }

    /**
    * create the object instance from the factory
    */
    public Object getObjectInstance(Object obj,
                    Name name, Context nameCtx, Hashtable environment)
                    throws NamingException {

        // name to lookup in the external factory
        String jndiLookupName = "";
        String jndiFactoryClass = null;
 	    String bindName = null;

        // get the target initial naming context and the lookup name
        Reference ref = (Reference) obj;
        Enumeration addrs = ref.getAll();
        while (addrs.hasMoreElements()) {
            RefAddr addr = (RefAddr) addrs.nextElement();

            String prop = addr.getType();
            if (prop.equals("jndiName")) {
                bindName = (String)addr.getContent();
            }
            else if (prop.equals("jndiLookupName")) {
                jndiLookupName = (String) addr.getContent();
            }
            else if (prop.equals("jndiFactoryClass")) {
                jndiFactoryClass = (String) addr.getContent();
            }
        }

        if (bindName == null) {
		    throw new NamingException("JndiProxyObjectFactory: no bindName context info");
	    }

	    ProxyRefAddr contextAddr = (ProxyRefAddr)ref.get(bindName);
	    Hashtable env = null;
	    if (contextAddr == null || 
            jndiFactoryClass == null ||
	        (env = (Hashtable)(contextAddr.getContent())) == null) {
		    throw new NamingException("JndiProxyObjectFactory: no info in the reference about the target context; contextAddr = " + contextAddr + " env = " + env + " factoryClass = " + jndiFactoryClass);
	}

        // Context of the external naming factory
        Context context = (Context)contextMap.get(bindName);
        if (context == null) {
            synchronized (contextMap) {
                context = (Context)contextMap.get(bindName);
                if (context == null) {
                    context = loadInitialContext(jndiFactoryClass, env);
                    contextMap.put(bindName, context);
                }
            }
        }

        if (context == null)
            throw new NamingException("JndiProxyObjectFactory no InitialContext" + jndiFactoryClass);

        // use the name to lookup in the external JNDI naming context
        try {
            return context.lookup(jndiLookupName);
        } catch (NameNotFoundException e) {
            throw new ExternalNameNotFoundException(e);
        }
    }
}
