/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.resource;

import java.io.Serializable;
import com.sun.enterprise.connectors.PoolMetaData;
import com.sun.enterprise.connectors.ConnectorConstants;

/**
 * ResourceSpec is used as a key to locate the correct resource pool
 */

public class ResourceSpec implements Serializable {
    private String resourceId;
    private int resourceIdType;
    private boolean pmResource;
    private boolean nonTxResource;
    private boolean lazyEnlistable_;
    private boolean lazyAssociatable_;
    private Object connectionToAssoc_;
    private boolean isXA_;
    
    private String connectionPoolName;
    
    static public final int JDBC_URL = 0;
    static public final int JNDI_NAME = 1;
    static public final int JMS = 2;

    public ResourceSpec(String resourceId,
        int resourceIdType) {
        if (resourceId == null) throw new NullPointerException();
        this.resourceId = resourceId;
        this.resourceIdType = resourceIdType;
        if (resourceId.endsWith(ConnectorConstants.PM_JNDI_SUFFIX) ) {
            pmResource = true;
        }
	if (resourceId.endsWith(ConnectorConstants.NON_TX_JNDI_SUFFIX) ) {
	    nonTxResource = true;
	}
    }

    public ResourceSpec( String resourceId, int resourceIdType, 
        PoolMetaData pmd ) {
        this( resourceId, resourceIdType );
        
        if (pmd != null ) {
            if ( pmd.isPM() ) {
                pmResource = true;
            }
            if( pmd.isNonTx() ) {
                nonTxResource = true;
            }

            if( pmd.isLazyEnlistable() && !nonTxResource && !pmResource ) {
                lazyEnlistable_ = true;
            }

            if ( pmd.isLazyAssociatable() && !nonTxResource && !pmResource) {
                lazyAssociatable_ = true;    
                //The rationale behind doing this is that in the PoolManagerImpl
                //when we return from getResource called by associateConnections, 
                //enlistment should happen immediately since we are associating on
                //first use anyway, 
                lazyEnlistable_ = false;
            }
        
        }

    }
    
    public String getConnectionPoolName() {
        return connectionPoolName;
    }
    
    
    public void setConnectionPoolName(String name) {
        connectionPoolName = name;
    }
    
    
    /*  The  logic is                                              *
     *  If the connectionpool exist then equality check is against *
     *  connectionPoolName                                         *
     *                                                             *
     *  If connection is null then equality check is made against  *
     *  resourceId and resourceType                                *
     */

    public boolean equals(Object other) {
        if (other == null) return false;
        if (other instanceof ResourceSpec) {
            ResourceSpec obj = (ResourceSpec) other;
            if(connectionPoolName == null) {
                return (resourceId.equals(obj.resourceId) &&
                resourceIdType == obj.resourceIdType);
            } else {
                return (connectionPoolName.equals(obj.connectionPoolName));

            }
        } else {
            return false;
        }
    }

    /*  If the connectionpool exist then hashcode of connectionPoolName    *
     *  is returned.                                                       *
     *                                                                     *
     *  If connectionpool is null return the hashcode of                   *
     *  resourceId + resourceIdType                                        *
     */
    
    public int hashCode() {
        if(connectionPoolName == null) {
            return resourceId.hashCode() + resourceIdType;
        } else {
            return connectionPoolName.hashCode();
        }
    }
    
    public String getResourceId() {
        return resourceId;
    }
    
    public boolean isPM() {
        return pmResource;
    }

    /**
     * Returns the status of the noTxResource flag
     * 
     * returns true if this resource is a noTx resource
     */
   
    public boolean isNonTx() {
        return nonTxResource;
    }

    public boolean isXA() {
        return isXA_;
    }

    public void markAsXA() {
        isXA_ = true;
    }
    
    public boolean isLazyEnlistable() {
        return lazyEnlistable_;
    }


    public void setLazyEnlistable( boolean lazyEnlist ) {
        lazyEnlistable_ = lazyEnlist;
    }

    public boolean isLazyAssociatable() {
        return lazyAssociatable_;
    }


    public void setLazyAssociatable( boolean lazyAssoc ) {
        lazyAssociatable_ = lazyAssoc;
    }
    
    public void setConnectionToAssociate( Object conn ) {
        connectionToAssoc_ = conn;
    }
    
    public Object getConnectionToAssociate() {
        return connectionToAssoc_;
    }
    
    public String toString() {
        StringBuffer sb = new StringBuffer("ResourceSpec :- ");
        sb.append("\nconnectionPoolName : " + connectionPoolName );
        sb.append("\nisXA_ : " + isXA_ );
        sb.append("\nresoureId : " + resourceId );
        sb.append("\nresoureIdType : " + resourceIdType );
        sb.append("\npmResource : " + pmResource );
        sb.append("\nnonTxResource : " + nonTxResource );
        sb.append("\nlazyEnlistable : " + lazyEnlistable_ );
        sb.append("\nlazyAssociatable : " + lazyAssociatable_ );

        return sb.toString();
    }

}
