/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.security;

import java.util.*;
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
import javax.security.auth.login.*;
import com.sun.enterprise.config.serverbeans.*;
import com.sun.enterprise.config.*;
import com.sun.enterprise.server.*;
import com.sun.enterprise.security.auth.realm.Realm;
import com.sun.enterprise.security.auth.realm.BadRealmException;
import com.sun.enterprise.util.LocalStringManagerImpl;


/**
 * Process realm initial configuration.
 *
 * <P>This class contains S1AS-specific initialization code for Realms, used
 * by RealmManager.
 *
 *
 */

public class RealmConfig
{
    private static Logger logger =
        LogDomains.getLogger(LogDomains.SECURITY_LOGGER);

    
    public static void createRealms(String defaultRealm, AuthRealm[] realms) 
    {
        assert(realms != null);

        String goodRealm = null; // need at least one good realm

        for (int i=0; i < realms.length; i++) {

            AuthRealm aRealm = realms[i];
            String realmName = aRealm.getName();
            String realmClass = aRealm.getClassname();
            assert (realmName != null);
            assert (realmClass != null);

            try {
                ElementProperty[] realmProps =
                    aRealm.getElementProperty();

                Properties props = new Properties();

                for (int j=0; j < realmProps.length; j++) {
                    ElementProperty p = realmProps[j];
                    String name = p.getName();
                    String value = p.getValue();
                    props.setProperty(name, value);
                }
                Realm.instantiate(realmName, realmClass, props);

                logger.fine("Configured realm: " + realmName);

                if (goodRealm == null) {
                    goodRealm = realmName;
                }
            } catch (Exception e) {
                logger.log(Level.WARNING,
                           "realmconfig.disable", realmName);
                logger.log(Level.WARNING, "security.exception", e);
            }
        }

        // done loading all realms, check that there is at least one
        // in place and that default is installed, or change default
        // to the first one loaded (arbitrarily).

        if (goodRealm == null) {
            logger.severe("realmconfig.nogood");

        } else {
            try {
                Realm def = Realm.getInstance(defaultRealm);
                if (def == null) {
                    defaultRealm = goodRealm;
                }
            } catch (Exception e) {
                defaultRealm = goodRealm;
            }
            Realm.setDefaultRealm(defaultRealm);
            logger.fine("Default realm is set to: "+ defaultRealm);
        }
    }
    
    /**
     * Load all configured realms from server.xml and initialize each
     * one.  Initialization is done by calling Realm.initialize() with
     * its name, class and properties.  The name of the default realm
     * is also saved in the Realm class for reference during server
     * operation.
     *
     * <P>This method superceeds the RI RealmManager.createRealms() method.
     *
     * */
    public static void createRealms()
    {
        
        try {
            logger.fine("Initializing configured realms.");
            
            ConfigContext configContext =
                ApplicationServer.getServerContext().getConfigContext();
            assert(configContext != null);

            Server configBean =
                ServerBeansFactory.getServerBean(configContext);
            assert(configBean != null);

            SecurityService securityBean =
                ServerBeansFactory.getSecurityServiceBean(configContext);
            assert(securityBean != null);

                                // grab default realm name
            String defaultRealm = securityBean.getDefaultRealm();

                                // get set of auth-realms and process each
            AuthRealm[] realms = securityBean.getAuthRealm();
            assert(realms != null);

            createRealms(defaultRealm, realms);
            
        } catch (Exception e) {
            logger.log(Level.SEVERE, "realmconfig.nogood", e);

        }
    }



    
}
