/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.security.ssl;

import java.net.Socket;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import javax.net.ssl.X509KeyManager;

/**
 * This class combines an array of X509KeyManagers into one.
 * @author Shing Wai Chan
 **/
public class UnifiedX509KeyManager implements X509KeyManager {
    private X509KeyManager[] mgrs = null;
    private String[] tokenNames = null;

    /**
     * @param mgrs  
     * @param tokenNames Array of tokenNames with order corresponding to mgrs
     */
    public UnifiedX509KeyManager(X509KeyManager[] mgrs, String[] tokenNames) {
        if (mgrs == null || tokenNames == null) {
            throw new IllegalArgumentException("Null array of X509KeyManagers or tokenNames");
        }
        if (mgrs.length != tokenNames.length) {
            throw new IllegalArgumentException("Size of X509KeyManagers array and tokenNames array do not match.");
        }
        this.mgrs = mgrs;
        this.tokenNames = tokenNames;
    }

    // ---------- implements X509KeyManager ----------
    public String chooseClientAlias(String[] keyType, Principal[] issuers,
            Socket socket) {
        String alias = null;
        for (int i = 0; i < mgrs.length; i++) {
            alias = mgrs[i].chooseClientAlias(keyType, issuers, socket);
            if (alias != null) {
                break;
            }
        }
        return alias;
    }

    public String chooseServerAlias(String keyType, Principal[] issuers,
            Socket socket) {
        String alias = null;
        for (int i = 0; i < mgrs.length; i++) {
            alias = mgrs[i].chooseServerAlias(keyType, issuers, socket);
            if (alias != null) {
                break;
            }
        }
        return alias;
    }

    public X509Certificate[] getCertificateChain(String alias) {
        X509Certificate[] chain = null;
        for (int i = 0; i < mgrs.length; i++) {
            chain = mgrs[i].getCertificateChain(alias);
            if (chain != null) {
                break;
            }
        }
        return chain;
    }

    public String[] getClientAliases(String keyType, Principal[] issuers) {
        ArrayList clientAliases = new ArrayList();
        for (int i = 0; i < mgrs.length; i++) {
            String[] clAliases = mgrs[i].getClientAliases(keyType, issuers);
            if (clAliases != null && clAliases.length > 0) {
                for (int j = 0; j < clAliases.length; j++) {
                    clientAliases.add(clAliases[j]);
                }
            }
        }

        return (clientAliases.size() == 0) ? null :
            (String[])clientAliases.toArray(new String[clientAliases.size()]);
    }

    public PrivateKey getPrivateKey(String alias) {
        PrivateKey privKey  = null;
        for (int i = 0; i < mgrs.length; i++) {
            privKey = mgrs[i].getPrivateKey(alias);
            if (privKey != null) {
                break;
            }
        }
        return privKey;
    }

    public String[] getServerAliases(String keyType, Principal[] issuers) {
        ArrayList serverAliases = new ArrayList();
        for (int i = 0; i < mgrs.length; i++) {
            String[] serAliases = mgrs[i].getClientAliases(keyType, issuers);
            if (serAliases != null && serAliases.length > 0) {
                for (int j = 0; j < serAliases.length; j++) {
                    serverAliases.add(serAliases[j]);
                }
            }
        }

        return (serverAliases.size() == 0) ? null :
            (String[])serverAliases.toArray(new String[serverAliases.size()]);
    }

    // ---------- end of implements X509KeyManager ----------

    X509KeyManager[] getX509KeyManagers() {
        X509KeyManager[] kmgrs = new X509KeyManager[mgrs.length];
        System.arraycopy(mgrs, 0, kmgrs, 0, mgrs.length);
        return kmgrs;
    }

    String[] getTokenNames() {
        String[] tokens = new String[tokenNames.length];
        System.arraycopy(tokenNames, 0, tokens, 0, tokenNames.length);
        return tokens;
    }
}
