/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.server;

import com.sun.enterprise.admin.event.AdminEventListenerException;
import com.sun.enterprise.admin.event.AdminEventListenerRegistry;
import com.sun.enterprise.admin.event.BaseDeployEvent;
import com.sun.enterprise.admin.event.DeployEventListenerHelper;
import com.sun.enterprise.admin.event.ModuleDeployEvent;
import com.sun.enterprise.admin.event.ModuleDeployEventListener;
import com.sun.enterprise.appclient.jws.AppclientJWSSupportManager;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.deployment.backend.DeployableObjectType;
import com.sun.enterprise.deployment.node.J2EEDocumentBuilder;
import com.sun.enterprise.instance.AppclientModulesManager;
import com.sun.enterprise.management.StateManageable;
import com.sun.enterprise.util.i18n.StringManager;
import com.sun.logging.LogDomains;
import java.util.Hashtable;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.management.MBeanException;


/**
 * AppClientModules Manager  acts as a listener for the deployment events.
 *
 * @author  Sreenivas Munnangi
 * @since   JDK1.4
 */

class StandAloneAppClientModulesManager extends AbstractManager
    implements ModuleDeployEventListener {

    static Logger _logger = LogDomains.getLogger(LogDomains.CORE_LOGGER);
    private static StringManager localStrings = StringManager.getManager(
            "com.sun.enterprise.server");

    /**
     * Application Client Modules Manager
     */
    StandAloneAppClientModulesManager(
	AppclientModulesManager acModuleManager, ClassLoader sharedCL) {

	super(sharedCL, acModuleManager);

        AdminEventListenerRegistry.addModuleDeployEventListener(this);

        /* Make sure the manager is alive to receive all start-up load events. */
        AppclientJWSSupportManager.getInstance();

    }

    /**
     * Invoked when a standalone application client module is deployed.
     */
    public synchronized void moduleDeployed(ModuleDeployEvent event)
        throws AdminEventListenerException {


        boolean jsr77 = false;

        if (_logger.isLoggable(Level.FINEST)) {
            _logger.log(Level.FINEST,
                "In StandAloneAppClientModulesManager moduleDeployed");
            _logger.log(Level.FINEST, "ModuleType=" + event.getModuleType());
        }

        if (event.getModuleType().equals(event.TYPE_APPCLIENT)) {

            DeployEventListenerHelper.getDeployEventListenerHelper().synchronize(event);

            String modID = event.getModuleName();

            if (_logger.isLoggable(Level.FINEST)) {
                _logger.log(Level.FINEST, "modID=" + modID);
            }

            try {
                // refreshes the config context with the context from this event
                this.configManager.refreshConfigContext(event.getConfigContext());

                // set jsr77 flag
                // which is used to signify if the event is deploy or undeploy
                // to create or delete jsr77 mBeans
                String action = event.getAction();
                if ((action.equals(BaseDeployEvent.DEPLOY)) ||
                        (action.equals(BaseDeployEvent.REDEPLOY))) {
                    jsr77 = true;
                }

                if (!moduleDeployed(jsr77, modID)) {

                    // throw an exception if load fails
                    String msg = localStrings.getString("appClientModule deploy failed",
                            modID);
                    throw new AdminEventListenerException(msg);
                }
            } catch (ConfigException ce) {
                throw new AdminEventListenerException(ce.getMessage());
            }
        }
    }


    /**
     * Invoked when a standalone application client module is undeployed.
     */

    public synchronized void moduleUndeployed(ModuleDeployEvent event)
        throws AdminEventListenerException {

        boolean jsr77 = false;

        if (_logger.isLoggable(Level.FINEST)) {
            _logger.log(Level.FINEST,
                "In StandAloneAppClientModulesManager moduleUndeployed");
        }

        String action = event.getAction();

        if ((action.equals(BaseDeployEvent.UNDEPLOY)) ||
                (action.equals(BaseDeployEvent.REDEPLOY))) {
            jsr77 = true;
        }

        try {
            if (event.getModuleType().equals(event.TYPE_APPCLIENT)) {
                String modID = event.getModuleName();

                if (_logger.isLoggable(Level.FINEST)) {
                    _logger.log(Level.FINEST, "UnDeploying module: " + modID);
                }

                // unload and throw exception if it fails
                if (!moduleUndeployed(jsr77, modID)) {
                    String msg = localStrings.getString("appclient.appclient_undeployed_failed",
                            modID);
                    throw new AdminEventListenerException(msg);
                }

            }
        } catch (Exception e) {
            throw new AdminEventListenerException(e.getMessage());
        }

    }


    /**
     * Invoked when a standalone application client module is redeployed.
     */

    public synchronized void moduleRedeployed(ModuleDeployEvent event)
        throws AdminEventListenerException {

        if (_logger.isLoggable(Level.FINEST)) {
            _logger.log(Level.FINEST,
                "In StandAloneAppClientModulesManager moduleRedeployed");
        }

        if (event.getModuleType().equals(event.TYPE_APPCLIENT)) {

            String modID = event.getModuleName();

            if (_logger.isLoggable(Level.FINEST)) {
                _logger.log(Level.FINEST, "ReDeploying module: " + modID);
            }

            moduleUndeployed(event);
            moduleDeployed(event);
        }
    }

    /**
     * Invoked when a standalone application client module is enabled.
     */
    public synchronized void moduleEnabled(ModuleDeployEvent event)
        throws AdminEventListenerException {

        if (_logger.isLoggable(Level.FINEST)) {
            _logger.log(Level.FINEST,
                "In StandAloneAppClientModulesManager moduleEnabled");
        }

	// for an application client module
	// the operations enable/disable or start/stop do not make sense

	return;
    }


    /**
     * Invoked when a standalone application client module is disabled.
     */

    public synchronized void moduleDisabled(ModuleDeployEvent event)
        throws AdminEventListenerException {

        if (_logger.isLoggable(Level.FINEST)) {
            _logger.log(Level.FINEST,
                "In StandAloneAppClientModulesManager moduleDisabled");
        }

	// for an application client module
	// the operations enable/disable or start/stop do not make sense

	return;
    }


    /**
     * Deployed event handling
     */

    private boolean moduleDeployed(boolean jsr77, String modID) {

	boolean result = false;
        boolean loadJSR77 = jsr77 || loadJSR77(modID, DeployableObjectType.CAR);
	try {
            AbstractLoader modLoader = getLoader(modID);

            // create root mBean for this module
            if (loadJSR77) {
                try {
                    modLoader.createRootMBean();
                } catch (MBeanException mbe) {
                    _logger.log(Level.WARNING,
                        "core.error_while_creating_jsr77_root_mbean",mbe);
                }
            }

            result = modLoader.load(loadJSR77);
            if (result) {
                this.id2loader.put(modID, modLoader);
            }
	} catch (Exception ce) {
            _logger.log(Level.WARNING,
                "core.error_while_loading_application_client_module",ce);
            result = false;
	}

        return result;
    }


    /**
     * Undeployed event handling
     */

    private boolean moduleUndeployed(boolean jsr77, String modID) {

	boolean result = false;
        try {
            ApplicationClientModuleLoader modLoader =
                (ApplicationClientModuleLoader) this.id2loader.remove(modID);

            if (modLoader == null) {
                return true;
            }

            // delete root mBean for this module
            if (jsr77) {
                try {
                    modLoader.deleteRootMBean();
                } catch (MBeanException mbe) {
                    _logger.log(Level.WARNING,
                        "core.error_while_deleting_jsr77_root_mbean",mbe);
                    }
            }

            result = modLoader.unload(jsr77);

        } catch (Exception ce) {
            _logger.log(Level.WARNING,
                    "core.error_while_unloading_application_client_module",ce);
            result = false;
	}

        return result;
    }

    /**
     * Returns loader
     */

    protected AbstractLoader getLoader(String moduleId) {
	return new ApplicationClientModuleLoader(
			moduleId, 
			this.parentClassLoader, 
			(AppclientModulesManager) this.configManager);
    }

 
/**
     * Invoked when a  reference is created from a
     * server instance (or cluster) to a particular module.
     *
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void moduleReferenceAdded(ModuleDeployEvent event)
            throws AdminEventListenerException {
                
}

    /**
     * Invoked when a reference is removed from a
     * server instance (or cluster) to a particular module.
     *
     * @throws AdminEventListenerException when the listener is unable to
     *         process the event.
     */
    public void moduleReferenceRemoved(ModuleDeployEvent event)
            throws AdminEventListenerException {
                
    }

}
