/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.server.ondemand;

import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import com.sun.logging.LogDomains;

import com.sun.enterprise.server.ondemand.entry.*;

/**
 * Concrete implementation of ServerEntryListener. This directs the
 * entry to servicegroups.
 *
 * @author Binod PG
 * @see ServiceGroup
 */
public class ServerEntryListenerImpl implements ServerEntryListener {

    private OnDemandServer server = null;
    private ArrayList listeners = new ArrayList();

    static Logger _logger = LogDomains.getLogger(LogDomains.CORE_LOGGER);
    
    public ServerEntryListenerImpl(OnDemandServer server) {
        this.server  = server;
    }

    /**
     * If atleast one servicegroup that need to be started is interested 
     * in the entry context, return false. Otherwise return true.
     */
    public boolean isNotified(EntryContext context) {
        context.setServerContext(server.getServerContext());
        return this.server.getServiceGroup().isNotified(context);
    }

    /**
     * Notify the main servicegroup about the server entry. If ondemand 
     * initialization is switched off, no entry event (other than the 
     * one special STARTUP event) will be processed.
     */
    public void notifyEntry(EntryContext context) {
        if (server.isOnDemandOff() == true && 
            context.getEntryPointType() != EntryPoint.STARTUP) {
            return;
        }
        context.setServerContext(server.getServerContext());
        try {
            this.server.getServiceGroup().start(context);
        } catch (Exception e) {
            _logger.log(Level.SEVERE, e.getMessage(), e);
        }

        Iterator it = listeners.iterator();
        while (it.hasNext()) {
            ServerEntryListener listener = (ServerEntryListener) it.next();
            listener.notifyEntry(context);
        }
    }

    /**
     * If some other part of the code want to listen to the events this is
     * a helper methods that can be used to add a listener.
     */
    public void addServerEntryListener(ServerEntryListener listener) {
        listeners.add(listener);
    }

    /**
     * Remove a listener from the list.
     */
    public void removeServerEntryListener(ServerEntryListener listener) {
        listeners.remove(listener);
    }
}
