/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.ara;

import com.sun.enterprise.web.connector.grizzly.HtmlHelper;
import com.sun.enterprise.web.connector.grizzly.ReadTask;
import com.sun.enterprise.web.connector.grizzly.Rule;
import com.sun.enterprise.web.connector.grizzly.SelectorThread;

import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.logging.Level;

import com.sun.enterprise.web.ara.rules.ThreadRatioRule;

/**
 * This class execute sequentialy <code>Rule</code>s on a <code>Task</code> 
 *
 * @author Jeanfrancois Arcand
 */
public class IsolationRulesExecutor implements RulesExecutor<IsolatedTask> {
    
    public final static int RULE_OK = 0;
    public final static int RULE_DELAY = 1;
    public final static int RULE_BLOCKED = 2;   
    public final static int RULE_OK_NOCACHE = 3;
    public final static int RULE_CONTINUE = 4;
    
            
    private final static String RULE_CLASS = 
            "com.sun.enterprise.web.ara.rules"; 
    
    
    private final static String DELAY_VALUE = 
            "com.sun.enterprise.web.ara.delay"; 
    
    
    /**
     * Initial number of <code>Rule</code> to cache.
     */
    private final static int INITIAL_RULE_COUNT = 5;


    /**
     * The list of <code>Rule</code> to apply to a <code>Task</code>
     */   
    protected ArrayList<Rule> rules = new ArrayList<Rule>();
    
    
    /**
     * The current number of thread used.
     */
    private static int currentThreadCount;
    
    
    /**
     * The time this thread will sleep when a rule is delayed.
     */
    private static long delayValue = 5 * 1000;
    
    
    /**
     * Is caching allowed
     */
    private boolean isCachingAllowed = true;
    
    // ---------------------------------------------------------------------//
    
     
    public IsolationRulesExecutor() {
        loadRules();
        
        if ( System.getProperty(DELAY_VALUE) != null){
            delayValue = Long.valueOf(System.getProperty(DELAY_VALUE));         
        }
    }

    
    /**
     * Load the list of <code>Rules</code> to apply to a <code>Task</code>
     */
    protected void loadRules(){      
        if ( System.getProperty(RULE_CLASS) != null){
            StringTokenizer st = new StringTokenizer(
                    System.getProperty(RULE_CLASS),",");
            while (st.hasMoreTokens()){
                rules.add(loadInstance(st.nextToken()));                
            } 
        }   
        
        if ( rules.size() == 0){
            rules.add(new ThreadRatioRule());
        }
    }
    
    
    /**
     * Execute the <code>Rule</code>s on the <code>IsolatedTask</code>
     * @param isolatedTask the task used.
     * @return true if the request processing can continue.
     */
    public boolean execute(IsolatedTask isolatedTask) {
        ReadTask task = (ReadTask)isolatedTask.getWrappedTask();
                       
        Integer status = 0;
        int i = 0;
        isCachingAllowed = true;
        while(true) {
            rules.get(i).attach(task);
            
            try{
                status = (Integer)rules.get(i).call();  
            } catch(Exception ex) { 
                SelectorThread.logger().log(Level.SEVERE,"Rule exception",ex);
                return true;
            }
            
            isCachingAllowed = (status == RULE_OK_NOCACHE ? false:true);
            
            if (status == RULE_DELAY){   
                
                // Wait until the delay expire. The current thread will 
                // wait and then re-execute the rules.
                try{
                    Thread.sleep(delayValue);
                } catch (InterruptedException ex) {
                    SelectorThread.logger().
                            log(Level.SEVERE,"Rule delay exception",ex);
                }
                i = 0;
                continue;
            }  else if ( status == RULE_BLOCKED ){
                task.cancelTask("No resources available.", HtmlHelper.OK);
                return true;
            } 
            
            i++;
            if (i == rules.size()){
                break;
            }
        }
        
        return (status == RULE_OK || status == RULE_OK_NOCACHE);
 
    }
    
    
    /**
     * Is caching of <code>Rule</code> results allowed
     */
    public boolean isCachingAllowed(){
        return isCachingAllowed;
    }

    
    // -------------------------------------------------------- Util --------//
    
     /**
     * Instanciate a class based on a property.
     */
    private Rule loadInstance(String property){        
        Class className = null;                               
        try{                              
            className = Class.forName(property);
            return (Rule)className.newInstance();
        } catch (ClassNotFoundException ex){
        } catch (InstantiationException ex){
        } catch (IllegalAccessException ex){
        }
        return new ThreadRatioRule();
    }   
}
