/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * A wrapper around an <code>ThreadPoolExecutor</code>.
 *
 * @author Jean-Francois Arcand
 */
public class ExecutorServicePipeline implements Pipeline {
    
    /**
     * The number of thread waiting for a <code>Task</code>
     */
    private int waitingThreads = 0;
    
    
    /**
     * The maximum number of Thread
     */
    private int maxThreads = 50;
    

    /**
     * The minimum numbers of <code>WorkerThread</code>
     */
    private int minThreads = 10;
    
    /**
     * The minimum numbers of spare <code>WorkerThread</code>
     */
    private int minSpareThreads = 5;

    
    /**
     * The port used.
     */
    private int port = 8080;
    

    /**
     * The number of <code>WorkerThread</code>
     */
    private int threadCount =0;
    

    /**
     * The name of this Pipeline
     */
    private String name;
    
    
    /**
     * The Thread Priority
     */
    private int priority = Thread.NORM_PRIORITY;
    
    
    /**
     * Has the pipeline already started
     */
    private boolean isStarted = false; 
    
    
    /**
     * <code>ExecutorService</code> wrapped by this pipeline.
     */
    private ThreadPoolExecutor workerThreads;
    
     
    /**
     * The <code>PipelineStatistic</code> objects used when gathering statistics.
     */
    protected PipelineStatistic pipelineStat; 
    
    
    /**
     * Maximum pending connection before refusing requests.
     */
    private int maxQueueSizeInBytes = -1;
    // ------------------------------------------------ Lifecycle ------------/
    
    /**
     * Init the <code>Pipeline</code> by initializing the required
     * <code>ThreadPoolExecutor</code>. 
     */
    public void initPipeline(){
        if (isStarted){
            return;
        }
        isStarted = true;
        
        if (maxThreads < minSpareThreads){
            maxThreads = minSpareThreads;
        }
        workerThreads = new ThreadPoolExecutor(
                               maxThreads,
                               maxThreads,
                               0L,
                               TimeUnit.MILLISECONDS,
                               new LinkedBlockingQueue<Runnable>(),
                               new GrizzlyThreadFactory(name,port,priority)); 
    }

    /**
     * Start the <code>Pipeline</code> 
     */
    public void startPipeline(){
        if (isStarted){
            return;
        }
        ; // Do nothing
    }
    

    /**
     * Stop the <code>Pipeline</code>
     */
    public void stopPipeline(){
        if (!isStarted){
            return;
        }
        isStarted = false;
        workerThreads.shutdown();
    }

    
    /**
     * Interrupt the <code>Thread</code> using it thread id
     */
    public boolean interruptThread(long threadID){
        return ((GrizzlyThreadFactory)workerThreads.getThreadFactory())
            .interruptThread(threadID);
    }    
    // ---------------------------------------------------- Queue ------------//
  
    
    /**
     * Add an object to this pipeline
     */
    public void addTask(Task task){   
        if (workerThreads.getQueue().size() > maxQueueSizeInBytes ){
            task.cancelTask("Maximum Connections Reached: " 
                            + pipelineStat.getQueueSizeInBytes()
                            + " -- Retry later", HtmlHelper.OK);
            task.getSelectorThread().returnTask(task);
            return;                                               
        }       
        workerThreads.execute((Runnable)task);
              
        if ( pipelineStat != null) {
            // Add one to include the current requests
            pipelineStat.gather(size());
        }  
    }


    /**
     * Return a <code>Task</code> object available in the pipeline.
     * 
     */
    public Task getTask() {
        return null;
    }

    
    /**
     * Returns the number of tasks in this <code>Pipeline</code>.
     *
     * @return Number of tasks in this <code>Pipeline</code>.
     */
    public int size() {
        return workerThreads.getQueue().size();
    }


    // --------------------------------------------------Properties ----------//

    /**
     * Return the number of waiting threads.
     */
    public int getWaitingThread(){
        return workerThreads.getPoolSize() - workerThreads.getActiveCount();
    }
    
    
    /** 
     * Set the number of threads used by this pipeline.
     */
    public void setMaxThreads(int maxThreads){
        this.maxThreads = maxThreads;
    }    
    
    
    /** 
     * Return the number of threads used by this pipeline.
     */
    public int getMaxThreads(){
        return maxThreads;
    }
    
    
    /**
     * Return the current number of threads used.
     */
    public int getCurrentThreadCount() {
        return workerThreads.getPoolSize() ;
    }
      
      
    /**
     * Return the curent number of threads that are currently processing 
     * a task.
     */
    public int getCurrentThreadsBusy(){
        return workerThreads.getActiveCount();
    }
    
    
    /**
     * Return the maximum spare thread.
     */
    public int getMaxSpareThreads() {
        return getWaitingThread();
    }
    
    
    /**
     * Return the minimum spare thread.
     */
    public int getMinSpareThreads() {
        return 0;
    }   
    
    
    /**
     * Set the minimum space thread this <code>Pipeline</code> can handle.
     */
    public void setMinSpareThreads(int minSpareThreads) {
        this.minSpareThreads = minSpareThreads;
    }       

    
    /**
     * Set the thread priority of the <code>Pipeline</code>
     */
    public void setPriority(int priority){
        this.priority = priority;
    }
    
    
    /**
     * Set the name of this <code>Pipeline</code>
     */
    public void setName(String name){
        this.name = name;
    }
    
    
    /**
     * Return the name of this <code>Pipeline</code>
     * @return the name of this <code>Pipeline</code>
     */
    public String getName(){
        return name+port;
    }    

    
    /**
     * Set the port used by this <code>Pipeline</code>
     * @param port the port used by this <code>Pipeline</code>
     */
    public void setPort(int port){
        this.port = port;
    }
    
    
    /**
     * Set the minimum thread this <code>Pipeline</code> will creates
     * when initializing.
     * @param minThreads the minimum number of threads.
     */
    public void setMinThreads(int minThreads){
        this.minThreads = minThreads;
    }
    
    
    public String toString(){
       return "name: " + name + " maxThreads: " + maxThreads 
                + " minThreads:" + minThreads;        
    }
    

    /**
     * Set the maximum pending connection this <code>Pipeline</code>
     * can handle.
     */
    public void setQueueSizeInBytes(int maxQueueSizeInBytes){
        this.maxQueueSizeInBytes = maxQueueSizeInBytes;
        if ( pipelineStat != null )
            pipelineStat.setQueueSizeInBytes(maxQueueSizeInBytes);
    }
    
    
    /**
     * Get the maximum pending connection this <code>Pipeline</code>
     * can handle.
     */
    public int getQueueSizeInBytes(){
        return maxQueueSizeInBytes;
    }  
    
    
    public void setThreadsIncrement(int threadsIncrement){
        ; // Not Supported
    }
    
    
    public void setThreadsTimeout(int threadsTimeout){
        ; // Not Supported
    }

    
    /**
     * Set the <code>PipelineStatistic</code> object used
     * to gather statistic;
     */
    public void setPipelineStatistic(PipelineStatistic pipelineStatistic){
        this.pipelineStat = pipelineStatistic;
    }
    
    
    /**
     * Return the <code>PipelineStatistic</code> object used
     * to gather statistic;
     */
    public PipelineStatistic getPipelineStatistic(){
        return pipelineStat;
    }
    
}
