/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.web.connector.grizzly.handlers;
import com.sun.enterprise.web.connector.grizzly.Constants;
import com.sun.enterprise.web.connector.grizzly.FileCacheFactory;
import com.sun.enterprise.web.connector.grizzly.Handler;
import com.sun.enterprise.web.connector.grizzly.FileCache;
import com.sun.enterprise.web.connector.grizzly.SelectorThread;
import java.io.IOException;
import java.nio.channels.SocketChannel;

import org.apache.coyote.Request;
import org.apache.coyote.Response;
import org.apache.coyote.tomcat5.CoyoteConnector;
import org.apache.coyote.tomcat5.CoyoteRequest;
import org.apache.coyote.tomcat5.CoyoteAdapter;
import org.apache.tomcat.util.buf.Ascii;
import org.apache.tomcat.util.buf.ByteChunk;
import org.apache.tomcat.util.buf.MessageBytes;
import org.apache.tomcat.util.http.MimeHeaders;
/**
 * This <code>Handler</code> is invoked after the request line has been parsed. 
 *
 * @author Jeanfrancois Arcand
 */
public class NoParsingHandler implements Handler<Request> {
      
    /**
     * The <code>SocketChannel</code> used to send a static resources.
     */
    private SocketChannel socketChannel;
 
    
    /**
     * The FileCache mechanism used to cache static resources.
     */
    protected FileCache fileCache;     
    
    
    // ----------------------------------------------------- Constructor ----//
    
    
    public NoParsingHandler(int port){
        fileCache =  
             FileCacheFactory.getFactory(port).getFileCache();
    }
    
      
    /**
     * Attach a <code>SocketChannel</code> to this object.
     */
    public void attachChannel(SocketChannel socketChannel){
        this.socketChannel = socketChannel;
    }    
    
    
    /**
     * Intercept the request and decide if we cache the static resource. If the
     * static resource is already cached, return it.
     */
    public int handle(Request request, int handlerCode) throws IOException{
        if ( socketChannel == null || !FileCacheFactory.isEnabled())
            return Handler.CONTINUE;
                
        // If not initialized, dont' continue
        if ( fileCache == null && handlerCode != Handler.RESPONSE_PROCEEDED){
             return Handler.CONTINUE;  
        }
        
        if ( handlerCode == Handler.RESPONSE_PROCEEDED ){
            CoyoteRequest cr = 
                (CoyoteRequest)request.getNote(CoyoteAdapter.ADAPTER_NOTES);
            // We can cache it only if no security constraint and no
            // filter have been defined.
            if ( cr != null && cr.getWrapper() != null){

                String mappedServlet = cr.getWrapper().getName();
                
                if ( !mappedServlet.equals(FileCache.DEFAULT_SERVLET_NAME) ) 
                    return Handler.CONTINUE;
                
                if ( cr.getContext().findConstraints().length == 0 
                    && cr.getContext().findFilterDefs().length == 0 ){

                    if (!fileCache.isEnabled()) return Handler.CONTINUE;  
                    
                    String docroot;
                    if ( cr.getContextPath().equals("") ){
                        docroot = cr.getContext().getDocBase();
                    } else {
                        docroot = SelectorThread.getWebAppRootPath();
                    }
                    String requestURI = cr.getRequestURI();
                    Response response = cr.getCoyoteRequest().getResponse();  
                    MimeHeaders headers = response.getMimeHeaders();
                    boolean xPoweredBy = (
                            (CoyoteConnector)cr.getConnector()).isXpoweredBy();

                    fileCache.add(mappedServlet,docroot,requestURI,headers, 
                            xPoweredBy);
                }
            }       
        } else if ( handlerCode == Handler.REQUEST_LINE_PARSED ) {
            ByteChunk bc = request.requestURI().getByteChunk();
            
            if ( fileCache.sendCache(bc.getBytes(),bc.getStart(), 
                    bc.getLength(),socketChannel,keepAlive(request)) ){
                return Handler.BREAK;
            }
        }     
        return Handler.CONTINUE;
    }
       
    
    /**
     * Get the keep-alive header.
     */
    private boolean keepAlive(Request request){
        MimeHeaders headers = request.getMimeHeaders();

        // Check connection header
        MessageBytes connectionValueMB = headers.getValue("connection");
        if (connectionValueMB != null) {
            ByteChunk connectionValueBC = connectionValueMB.getByteChunk();
            if (findBytes(connectionValueBC, Constants.CLOSE_BYTES) != -1) {
                return false;
            } else if (findBytes(connectionValueBC, 
                                 Constants.KEEPALIVE_BYTES) != -1) {
                return true;
            }
        }
        return true;
    }
    
    
    /**
     * Specialized utility method: find a sequence of lower case bytes inside
     * a ByteChunk.
     */
    protected int findBytes(ByteChunk bc, byte[] b) {

        byte first = b[0];
        byte[] buff = bc.getBuffer();
        int start = bc.getStart();
        int end = bc.getEnd();

        // Look for first char 
        int srcEnd = b.length;

        for (int i = start; i <= (end - srcEnd); i++) {
            if (Ascii.toLower(buff[i]) != first) continue;
            // found first char, now look for a match
            int myPos = i+1;
            for (int srcPos = 1; srcPos < srcEnd; ) {
                    if (Ascii.toLower(buff[myPos++]) != b[srcPos++])
                break;
                    if (srcPos == srcEnd) return i - start; // found it
            }
        }
        return -1;
    }
}
