/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.enterprise.tools.verifier.persistence;

import com.sun.enterprise.tools.verifier.*;
import com.sun.enterprise.tools.verifier.tests.ComponentNameConstructor;
import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.PersistenceUnitDescriptor;
import com.sun.enterprise.deployment.BundleDescriptor;
import com.sun.enterprise.deployment.RootDeploymentDescriptor;
import com.sun.enterprise.deployment.util.ModuleDescriptor;
import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.util.LocalStringManagerImpl;

import javax.enterprise.deploy.shared.ModuleType;
import java.io.File;

/**
 * This class is responsible for checking a PU represented by a {@link
 * PersistenceUnitDescriptor}
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class PersistenceUnitCheckMgrImpl extends CheckMgr {

    // module for which this check mgr is running the test.
    // This string is one of the types defined in Result class.
    private String moduleName;
    private LocalStringManagerImpl smh = StringManagerHelper.getLocalStringsManager();
    

    public PersistenceUnitCheckMgrImpl(
            FrameworkContext frameworkContext, Context context) {
        this.frameworkContext = frameworkContext;
        this.context = context;
    }

    @Override protected void check(Descriptor descriptor) throws Exception {
        PersistenceUnitDescriptor pu =
                PersistenceUnitDescriptor.class.cast(descriptor);
        RootDeploymentDescriptor rootDD = pu.getParent().getParent();
        if(rootDD.isApplication()) {
            moduleName = Result.APP;
        } else {
            ModuleDescriptor mdesc =
                    BundleDescriptor.class.cast(rootDD).getModuleDescriptor();
            final ModuleType moduleType = mdesc.getModuleType();
            if(moduleType == ModuleType.EJB) {
                moduleName = Result.EJB;
            } else if (moduleType == ModuleType.WAR) {
                moduleName = Result.WEB;
            } else if (moduleType == ModuleType.CAR) {
                moduleName = Result.APPCLIENT;
            } else {
                throw new RuntimeException(
                        smh.getLocalString(getClass().getName()+".exception", // NOI18N
                                "Unknown module type : {0}", // NOI18N
                                new Object[] {moduleType}));
            }
        }
        super.check(descriptor);
    }

    /**
     * We override here because there is nothing like sun-persistence.xml.
     * @param uri
     */
    @Override protected void setRuntimeDDPresent(String uri) {
        isDDPresent = false;
    }

    protected ComponentNameConstructor getComponentNameConstructor(
            Descriptor descriptor) {
        return new ComponentNameConstructor(
                PersistenceUnitDescriptor.class.cast(descriptor));
    }

    protected String getTestsListFileName() {
        return "TestNamesPersistence.xml"; // NOI18N
    }

    protected void setModuleName(Result r) {
        r.setModuleName(moduleName);
    }

    protected String getSchemaVersion(Descriptor descriptor) {
        // A PU inherits its schema version from its parent.
        return PersistenceUnitDescriptor.class.cast(descriptor).getParent().
                getSpecVersion();
    }

    protected String getSunONETestsListFileName() {
        return null;
    }

    /**
     * This method returns the path to the module.
     * @param descriptor is a PersistenceUnitDescriptor
     * @return the path to the module
     */
    protected String getAbstractArchiveUri(Descriptor descriptor) {
        String archBase = context.getAbstractArchive().getArchiveUri();
        RootDeploymentDescriptor rootDD =
                PersistenceUnitDescriptor.class.cast(descriptor).getParent().getParent();
        if(rootDD.isApplication()) {
            return archBase;
        } else {
            ModuleDescriptor mdesc =
                    BundleDescriptor.class.cast(rootDD).getModuleDescriptor();
            if(mdesc.isStandalone()) {
                return archBase;
            } else {
                return archBase + File.separator +
                        FileUtils.makeFriendlyFileName(mdesc.getArchiveUri());
            }
        }
    }
}
