/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */


package com.sun.jdo.api.persistence.enhancer.classfile;

import java.io.*;

/**
 * ClassField models the static and non-static fields of a class within
 * a class file.
 */

final public class ClassField extends ClassMember {
  /* access flag bit mask - see VMConstants */
  private int accessFlags;

  /* The name of the field */
  private ConstUtf8 fieldName;

  /* The type signature of the field */
  private ConstUtf8 fieldSignature;

  /* The attributes associated with the field */
  private AttributeVector fieldAttributes;
  

  /* public accessors */

  /**
   * Is the field transient?
   */
  public boolean isTransient() {
    return (accessFlags & ACCTransient) != 0;
  }

  /**
   * Return the access flags for the field - see VMConstants
   */
  public int access() {
    return accessFlags;
  }

  /**
   * Update the access flags for the field - see VMConstants
   */
  public void setAccess(int newFlags) {
    accessFlags = newFlags;
  }

  /**
   * Return the name of the field
   */
  public ConstUtf8 name() {
    return fieldName;
  }

  /**
   * Change the name of the field
   */
  public void changeName(ConstUtf8 name) {
    fieldName = name;
  }

  /**
   * Return the type signature of the field
   */
  public ConstUtf8 signature() {
    return fieldSignature;
  }

  /**
   * Change the type signature of the field
   */
  public void changeSignature(ConstUtf8 newSig) {
    fieldSignature = newSig;
  }

  /**
   * Return the attributes associated with the field
   */
  public AttributeVector attributes() {
    return fieldAttributes;
  }

  /**
   * Construct a class field object
   */
  public ClassField(int accFlags, ConstUtf8 name, ConstUtf8 sig,
                    AttributeVector field_attrs) {
    accessFlags = accFlags;
    fieldName = name;
    fieldSignature = sig;
    fieldAttributes = field_attrs;
  }

  /* package local methods */

  static ClassField read(DataInputStream data, ConstantPool pool) 
    throws IOException {
    ClassField f = null;
    int accessFlags = data.readUnsignedShort();
    int name_index = data.readUnsignedShort();
    int sig_index = data.readUnsignedShort();
    AttributeVector fieldAttribs = AttributeVector.readAttributes(data, pool);
    f = new ClassField(accessFlags, 
		       (ConstUtf8) pool.constantAt(name_index),
		       (ConstUtf8) pool.constantAt(sig_index),
		       fieldAttribs);
    return f;
  }

  void write (DataOutputStream data) throws IOException {
    data.writeShort(accessFlags);
    data.writeShort(fieldName.getIndex());
    data.writeShort(fieldSignature.getIndex());
    fieldAttributes.write(data);
  }

  void print(PrintStream out, int indent) {
    ClassPrint.spaces(out, indent);
    out.print("'" + fieldName.asString() + "'");//NOI18N
    out.print(" sig = " + fieldSignature.asString());//NOI18N
    out.print(" access_flags = " + Integer.toString(accessFlags));//NOI18N
    out.println(" attributes:");//NOI18N
    fieldAttributes.print(out, indent+2);
  }
}

