/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.ejb.cmp3.persistence;

import java.net.URL;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.jar.JarInputStream;
import java.util.zip.ZipEntry;
import java.util.Enumeration;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.List;
import java.util.Vector;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.persistence.spi.PersistenceUnitInfo;

import org.xml.sax.XMLReader;
import org.xml.sax.InputSource;

import oracle.toplink.essentials.exceptions.PersistenceUnitLoadingException;
import oracle.toplink.essentials.exceptions.XMLParseException;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.parser.PersistenceContentHandler;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.parser.XMLException;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.parser.XMLExceptionHandler;
import oracle.toplink.essentials.logging.AbstractSessionLog;

/**
 * INTERNAL:
 * Utility Class that deals with persistence archives for EJB 3.0
 * Provides functions like searching for persistence archives, processing persistence.xml
 * and searching for Entities in a Persistence archive
 */
public class PersistenceUnitProcessor  {
   
    public static final String JAR_OPENING_EXCEPTION = "attempted_to_open_url_as_jar";
    public static final String DIRECTORY_OPENING_EXCEPTION = "attempted_to_open_url_as_directory";
    public static final String JAR_ENTRY_OPENING_EXCEPTION = "attempted_to_open_entry_in_url_as_jar";
    public static final String DIRECTORY_ENTRY_OPENING_EXCEPTION = "attempted_to_open_file_url_as_directory";

    /**
     * Get a list of persitence units from the file or directory at the given url
     * PersistenceUnits are built based on the presence of persistence.xml in a META-INF directory
     * at the base of the URL
     * @param url The url of a jar file or directory to check
     * @return 
     */
    public static List<SEPersistenceUnitInfo> getPersistenceUnits(URL url, ClassLoader loader){
        return processPersistenceArchive(url, loader);
    }
    
    
  /**
   * Open a stream on a url representing a persistence.xml and process it.
   * The URL passed into this method should be a persistence.xml file.
   * @param file 
   */
    protected static List<SEPersistenceUnitInfo> processPersistenceArchveFromUnconvertableURL(URL baseURL, ClassLoader loader){
        InputStream stream = null;
        try{
            stream = baseURL.openStream();
            return processPersistenceXML(baseURL, stream, loader);

        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionLoadingFromUrl(baseURL.toString(), exc);
        } finally {
            try{
                if (stream != null){
                    stream.close();                    
                }
            } catch (IOException exc){};

        }
    }
    
  /**
   * Search a directory for persistence.xml. It should be found in a
   * meta-inf directory contained within the passed directory.
   * The file passed into this method should be a jar file or null will be returned
   * @param file a File representing the directory to search
   */
    private static List<SEPersistenceUnitInfo> processDirectory(URL baseURL, File file, ClassLoader loader){
        FileInputStream inputStream = null;
        try{           
            String filePath = file.getPath();
            File persistenceXMLFile = new File(filePath + File.separator + "META-INF" + File.separator + "persistence.xml"); 
            
            if (!persistenceXMLFile.exists()){
                persistenceXMLFile = new File(filePath + File.separator + "meta-inf" + File.separator + "persistence.xml");
            }
            if (persistenceXMLFile.exists()){
                inputStream = new FileInputStream(persistenceXMLFile);
                return processPersistenceXML(baseURL, inputStream, loader);
            }
            return null;
        } catch (FileNotFoundException exc){
            throw PersistenceUnitLoadingException.exceptionLoadingFromDirectory(file, exc);            
        } finally {
            try{
                if (inputStream != null){
                    inputStream.close();                    
                }
            } catch (IOException exc){};

        }
    }
    
  /**
   * Search a jar file for persistence.xml.  It should be found in the
   * the meta-inf subdirectory.  When persistence.xml is found, process it.
   * The file passed into this method should be a jar file.
   * @param file 
   */
    protected static List<SEPersistenceUnitInfo> processJarFile(URL baseURL, ClassLoader loader){
        JarInputStream jarFile = null;
        InputStream stream = null;
        try{
            stream = createInputStreamForURL(baseURL);
            jarFile = new JarInputStream(stream);
            ZipEntry entry = jarFile.getNextEntry();
            while (entry != null && !entry.getName().equals("META-INF/persistence.xml") && !entry.getName().equals("meta-inf/persistence.xml")){
                entry = jarFile.getNextEntry();
            }
            if (entry != null &&  (entry.getName().equals("META-INF/persistence.xml") || entry.getName().equals("meta-inf/persistence.xml"))){
                return processPersistenceXML(baseURL, jarFile, loader);
            }
            return null;
        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionLoadingFromJar(baseURL, exc);
        } finally {
            try{
                if (jarFile != null){
                    jarFile.close();                    
                }
                if (stream != null){
                    stream.close();                    
                }
            } catch (IOException exc){};

        }
    }
    
    /**
     * Go through the jar file for this PeristeneUnitProcessor and process any XML provided in it
     */
    private static List<SEPersistenceUnitInfo> processPersistenceArchive(URL url, ClassLoader loader){
        File file = null;
        List<SEPersistenceUnitInfo> persistenceUnitList = null;
        // Should be called from JavaSE, and so the url string should still contain 
        // the meta-inf/persistence.xml at the end that should be stripped off to be handled as a file
        URL tempURL = truncateURLAndRemoveFileReference(url);
        try{
            persistenceUnitList = processJarFile(tempURL, loader);
            if (persistenceUnitList != null){
                return persistenceUnitList;
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_OPENING_EXCEPTION, url, exception);
        };
        
        try{
            // Only URL.toURI() handles spaces and special characters correctly
            file = new File(convertURLToURI(tempURL));
            if (file.isDirectory()){
                persistenceUnitList = processDirectory(tempURL, file, loader);
                if (persistenceUnitList != null){
                    return persistenceUnitList;
                }
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, DIRECTORY_OPENING_EXCEPTION, url, exception);
        };
            
        persistenceUnitList = processPersistenceArchveFromUnconvertableURL(url, loader);

        return persistenceUnitList;
    }
    
    /**
     *  Create an input stream for a file contained in a persistence unit.
     *  This method will use the root url in the persistence unit as a base and try to discover the 
     *  given filename within that context
     *  
     *  This method will handle root urls that can be converted to JarFiles and to directories in a 
     *  special manner and if it fails, will guess which System resource to use.
     */
    public static InputStream createInputStreamForFileInPersistenceUnit(String fileName, PersistenceUnitInfo persistenceUnitInfo, ClassLoader classLoader) throws IOException{
        InputStream stream = null;
        try{
            stream = createInputStreamForJarFile(fileName, persistenceUnitInfo.getPersistenceUnitRootUrl());
            if (stream != null){
                return stream;
            }
        } catch (Exception e){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_ENTRY_OPENING_EXCEPTION, persistenceUnitInfo.getPersistenceUnitRootUrl(), fileName, e);
        };
           
        try {
            // Only URL.toURI() handles spaces and special characters correctly
            File file = new File(convertURLToURI(persistenceUnitInfo.getPersistenceUnitRootUrl()));
            if (file.isDirectory()){
                stream = createInputStreamForDirectory(fileName, file);
                if (stream != null){
                    return stream;
                }
            }
        } catch (Exception e){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, DIRECTORY_ENTRY_OPENING_EXCEPTION, persistenceUnitInfo.getPersistenceUnitRootUrl(), fileName, e);
        };

        return createInputStreamForUnconvertableURL(fileName, persistenceUnitInfo.getPersistenceUnitRootUrl(), classLoader);
   }
    
    /**
     * Create an input stream for a file in a jar file
     */
    public static InputStream createInputStreamForJarFile(String fileName, URL url) throws IOException{
        JarInputStream jarInputStream = new JarInputStream(createInputStreamForURL(url));
 
        ZipEntry entry = jarInputStream.getNextEntry();
        while (entry != null && !entry.getName().equals(fileName)){
            entry = jarInputStream.getNextEntry();
        }
        if (entry != null &&  (entry.getName().equals(fileName))){
            return jarInputStream;
        }
        return null;
    }

    /**
     * Create an input stream for a file in a directory
     */
    public static InputStream createInputStreamForDirectory(String fileName, File inputFile) throws IOException{
        String filePath = inputFile.getPath();
        String tempFileName = fileName;
        if (!filePath.endsWith(File.separator) && !fileName.startsWith(File.separator)){
            tempFileName =  File.separator + tempFileName;
        }
        
        File xmlFile = new File(filePath + tempFileName); 
        if (xmlFile.exists()){
            return new FileInputStream(xmlFile);
        }
        return null;
    }

    /**
     *  Create an InputStream for a fileName in a URL that could not be converted to a JarFile
     *  or a directory.
     *  This method looks up the fileName as a resources on the given ClassLoader and guesses
     *  whether it is the correct file by comparing the URL of the discovered resource to the 
     *  base URL of the file being searched for
     */
    public static InputStream createInputStreamForUnconvertableURL(String fileName, URL url, ClassLoader loader) throws IOException{
        String persistenceUnitRoolUrlString = url.toString();
        Enumeration<URL> resources = loader.getResources(fileName);                
        while (resources.hasMoreElements()){   
            URL mappingFileResource = resources.nextElement();
            String mappingFileResourceString = mappingFileResource.toString();
            if(mappingFileResourceString.contains(persistenceUnitRoolUrlString)) {
                return url.openStream();
            }
        }
        return null;
    }
    
    /**
     *  This method will attempt to open an InputStream on a URL by first calling the url.openStream() method
     *  If that succeeds, the stream will be returned.  If it fails, we will attempt to modify the URL to a form
     *  that we can open and returned a stream based on the modified stream.
     *  If that fails, we will throw in exception we initially got on openStream()
     */
    public static InputStream createInputStreamForURL(URL url) throws IOException{
        InputStream stream = null;
        IOException initialException = null;
        try{
            // attempt to open a stream on the given URL
            stream = url.openStream();
        } catch (IOException exc){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_OPENING_EXCEPTION, url, exc);
            initialException = exc;
            try{
                // guess at an alternate form of the URL to open
                stream = createTruncatedJarURLFromString(url.toString()).openStream();
            } catch (PersistenceUnitLoadingException exception){
                AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_OPENING_EXCEPTION, url, exception);
                throw initialException;                    
            } catch (IOException exception2){
                AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_OPENING_EXCEPTION, url, exception2);
                throw initialException;
            }
        }
        return stream;
    }
    
  /**
   * Build a persistence.xml file into a SEPersistenceUnitInfo object 
   * @param input 
   */
    public static List<SEPersistenceUnitInfo> processPersistenceXML(URL baseURL, InputStream input, ClassLoader loader){
        SAXParserFactory spf = SAXParserFactory.newInstance();
        spf.setNamespaceAware(true);
        spf.setValidating(true);
        
        XMLReader xmlReader = null;
        SAXParser sp = null;
        XMLExceptionHandler xmlErrorHandler = new XMLExceptionHandler();

        // create a SAX parser
        try {
            sp = spf.newSAXParser();
	        sp.setProperty(XMLConstants.SCHEMA_LANGUAGE, XMLConstants.XML_SCHEMA);
	    } catch (javax.xml.parsers.ParserConfigurationException exc){
	    	throw XMLParseException.exceptionCreatingSAXParser(baseURL, exc);
	    } catch (org.xml.sax.SAXException exc){
	    	throw XMLParseException.exceptionCreatingSAXParser(baseURL, exc);
	    }
	        
	    // create an XMLReader
	    try {
            xmlReader = sp.getXMLReader();
	        xmlReader.setErrorHandler(xmlErrorHandler);
        } catch (org.xml.sax.SAXException exc){
        	throw XMLParseException.exceptionCreatingXMLReader(baseURL, exc);
        }
       
        // attempt to load the schema from the classpath
        URL schemaURL = loader.getResource(XMLConstants.PERSISTENCE_SCHEMA_NAME);
        if (schemaURL != null) {
            try {
            	sp.setProperty(XMLConstants.JAXP_SCHEMA_SOURCE, schemaURL.toString());
            } catch (org.xml.sax.SAXException exc){
            	throw XMLParseException.exceptionSettingSchemaSource(baseURL, schemaURL, exc);
            }
        }
        
        PersistenceContentHandler myContentHandler = new PersistenceContentHandler();
        xmlReader.setContentHandler(myContentHandler);

        InputSource inputSource = new InputSource(input);
        try{
            xmlReader.parse(inputSource);
        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionProcessingPersistenceXML(baseURL, exc);
        } catch (org.xml.sax.SAXException exc){
        	// XMLErrorHandler will handle SAX exceptions
        }
        
        // handle any parse exceptions
        XMLException xmlError = xmlErrorHandler.getXMLException();
        if (xmlError != null) {
            throw PersistenceUnitLoadingException.exceptionProcessingPersistenceXML(baseURL, xmlError);
        }

        Iterator<SEPersistenceUnitInfo> persistenceInfos = myContentHandler.getPersistenceUnits().iterator();
        while (persistenceInfos.hasNext()){
            SEPersistenceUnitInfo info = persistenceInfos.next();
            info.setPersistenceUnitRootUrl(baseURL);           
        }
        return myContentHandler.getPersistenceUnits();
    }
    
  /**
   * Strips down a URL for a persistence.xml so that it's base can be used to create a file 
   */
    private static URL truncateURLAndRemoveFileReference(URL url) {
        String newURLString = url.toString().substring(0, url.toString().length() - 25);
        return createTruncatedJarURLFromString(newURLString);

    }

    /**
     * Clean up a string that represents a jar.  This will create a URL based on the url parameter, but remove
     * a leading 'jar:' and trailing characters starting with the first '!'.  The functionality guesses at a URL
     * that a stream can be opened on based on the url parameter.
     * This method is implemented to attempt to provide a URL that can be opened based on a URL retrieved from
     * a classLoader.getResource(String) call on a TomCat deployment.
     * @param url
     * @return
     */
    private static URL createTruncatedJarURLFromString(String url){
        try{
            String cleanUrl = url;
            int index = url.lastIndexOf('!');
            if (index > 0){
                cleanUrl = cleanUrl.substring(0, index);
            }
            if (cleanUrl.startsWith("jar:")){
                cleanUrl = cleanUrl.substring(4, cleanUrl.length());
            }
            return new URL(cleanUrl);
                
        } catch (java.net.MalformedURLException exc){
            throw PersistenceUnitLoadingException.exceptionLoadingFromUrl(url, exc);
        }
    }
    
   /**
   * Entries in a zip file are directory entries using slashes to separate them.
   * Build a class name using '.' instead of slash and removing the '.class' extension.
   * @param classEntryString 
   * @return 
   */
    public static String buildClassNameFromEntryString(String classEntryString){
        String classNameForLoader = classEntryString;
        if (classEntryString.endsWith(".class")){
            classNameForLoader = classNameForLoader.substring(0, classNameForLoader.length() - 6);;
            classNameForLoader = classNameForLoader.replace("/", ".");              
        }
        return classNameForLoader;
    }

   /**
   * Build a set that contains all the class names at a URL
   * @return a Set of class name strings
   */
    public static Set<String> buildClassSet(PersistenceUnitInfo persistenceUnitInfo){
        Set<String> set = new HashSet<String>();
        set.addAll(persistenceUnitInfo.getManagedClassNames());
        Iterator i = persistenceUnitInfo.getJarFileUrls().iterator();
        while (i.hasNext()) {
            set.addAll(PersistenceUnitProcessor.getClassNamesFromURL((URL)i.next()));
        }
        if (!persistenceUnitInfo.excludeUnlistedClasses()){
            set.addAll(PersistenceUnitProcessor.getClassNamesFromURL(persistenceUnitInfo.getPersistenceUnitRootUrl()));
        }
       return set;
    }

  /**
   * Return a set of class names that are annotated as either @Entity or @Embeddable.
   * from the base URL of this PersistenceUnitProcessor
   * @param loader the class loader to load the classes with
   * @return 
   */
    public static Set<String> buildPersistentClassSet(PersistenceUnitInfo persistenceUnitInfo, ClassLoader loader){
        Set<String> set = new HashSet();
        
        for (String className : persistenceUnitInfo.getManagedClassNames()) {
            if (isClassPersistent(className, loader, true)) {
                set.add(className);
            }
        }
        
        Iterator i = persistenceUnitInfo.getJarFileUrls().iterator();
        while (i.hasNext()) {
            set.addAll(PersistenceUnitProcessor.getPersistentClassNamesFromURL((URL)i.next(), loader));
        }
        if (!persistenceUnitInfo.excludeUnlistedClasses()){
            set.addAll(PersistenceUnitProcessor.getPersistentClassNamesFromURL(persistenceUnitInfo.getPersistenceUnitRootUrl(), loader));
        }
       return set;
    }

  /**
   * Recursive method to look through a directory and build class names for all files
   * ending in '.class' in that directory and any subdirectory.  Strips out any extraneous
   * characters and returns a className with package names separated by '.'
   * @param directory 
   * @param leadingCharactersToRemove 
   * @return Returns a list of class names in a directory
   */
    protected static List<String> findClassesInDirectory(File directory, int leadingCharactersToRemove){
        Vector classes = new Vector();
        File[] files = directory.listFiles();
        for (File file: files){
            if (file.isDirectory()){
                classes.addAll(findClassesInDirectory(file, leadingCharactersToRemove));
            }
            if (file.isFile() && file.getName().endsWith(".class")){
                String className = file.getPath().substring(leadingCharactersToRemove + 1, file.getPath().length() - 6);
                className = className.replace("/", ".");
                className = className.replace("\\", ".");
                classes.add(className);
            }
        }
        return classes;
    }

    /**
     *  Search the classpath for persistence archives.  A persistence archive is defined as any
     *  part of the class path that contains a META-INF directory with a persistence.xml file in it. 
     *  Return a list of the URLs of those files.
     *  Use the current thread's context classloader to get the classpath.  We assume it is a URL class loader
     */
    public static Set<URL> findPersistenceArchives(){
        ClassLoader threadLoader = Thread.currentThread().getContextClassLoader();
        return findPersistenceArchives(threadLoader);
    }


    /**
     *  Search the classpath for persistence archives. A persistence archive is defined as any
     *  part of the class path that contains a META-INF directory with a persistence.xml file in it.. 
     *  Return a list of the URLs of those files.
     *  @param loader the class loader to get the class path from
     */
    public static Set<URL> findPersistenceArchives(ClassLoader loader){
         Set<URL> parURLs = new HashSet<URL>(); 
        try {
            Enumeration<URL> resources = loader.getResources("META-INF/persistence.xml");
            while (resources.hasMoreElements()){
                URL url = resources.nextElement();
                parURLs.add(url);
            }
        } catch (java.io.IOException exc){
            throw PersistenceUnitLoadingException.exceptionSearchingForPersistenceResources(loader, exc);
        }
        return parURLs;
    }

  /**
   * Return a list of all the names of classes stored in the jar stored at this URL.
   * Classes are assumed to be located in the base directory of the jar.
   * @param file 
   * @return 
   */
    private static List<String> getClassNamesFromJar(URL url){
        InputStream stream = null;
        List persistentClasses = null;
        IOException initialException;
        try {
            stream = createInputStreamForURL(url);
            persistentClasses = findClassesInJar(stream);
        } catch (IOException exc){
            throw PersistenceUnitLoadingException.exceptionSearchingForEntities(url, exc);
        } finally {
            try{
                if (stream != null){
                    stream.close();                    
                }
            } catch (IOException exc){};
        }
        return persistentClasses;

    }
    
    /**
     * Convert the given input stream to a jar input stream and look through the entities for classes
     * Return a list of those classes
     * @param stream
     * @return
     * @throws IOException
     */
    private static List findClassesInJar(InputStream stream) throws IOException {
        JarInputStream jarFile = null;
        List persistentClasses = new Vector();
        try{
            jarFile = new JarInputStream(stream);
            ZipEntry entry = jarFile.getNextEntry();
            if (entry == null){
                return null;
            }
            while (entry != null){
                String classNameForLoader = buildClassNameFromEntryString(entry.getName());
                 if (entry.getName().endsWith(".class")){
                     persistentClasses.add(classNameForLoader);
                }
                entry = jarFile.getNextEntry();
            }
        } finally {
            try{
                if (jarFile != null){
                    jarFile.close();                    
                }
            } catch (IOException exc){};
        }
        return persistentClasses;
    }
    
  /**
   * Return a list of class names from this URL.  This will work either with directories
   * or jar files and assume the classes are based in the base directory of the URL
   * @param url 
   * @return 
   */
    private static List<String> getClassNamesFromURL(URL url){
        List classNames = null;
        try {
            classNames = getClassNamesFromJar(url);
            if (classNames != null){
                return classNames;
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_OPENING_EXCEPTION, url, exception);
        };
        try{
            // Only URL.toURI() handles spaces and special characters correctly
            File file = new File(convertURLToURI(url));
            if (file.isDirectory()){
                return getClassNamesFromDirectory(file);                  
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, DIRECTORY_OPENING_EXCEPTION, url, exception);
        }
        return null;
    }

  /**
   *  Return a list of the names of classes that are stored in this directory.  Package
   *  name for the class will assume classes are based in the directoryURL
   * @param file the File representation of this directory
   * @return 
   */
    private static List<String> getClassNamesFromDirectory(File file){
        List classList = null;        

        if (!file.isDirectory()){
            return null;
        }
        int initialDirectoryNameLength = file.getPath().length();
        classList = findClassesInDirectory(file, initialDirectoryNameLength);        
        return classList;
    }

  /**
   *  Create a list of the classes that from a directory referenced by the given URL that return
   *  true from the isPersistent() call.
   * @param file the File representation of this directory
   * @param loader the ClassLoader to use
   * @return 
   */
    public static List<String> getPersistentClassNamesFromDirectory(File file, ClassLoader loader){
        List<String> persistentClasses = new Vector<String>();
        List<String> classList = getClassNamesFromDirectory(file);
        for (String className: classList){
            if (isClassPersistent(className, loader, false)){
                persistentClasses.add(className);
            }
        }
        return persistentClasses;
    }
    
  /**
   *  Return a list of persistent classes names accessible through a given URL
   *  If the URL refers to a .jar or .par file this method will look through the entries in 
   *  the jar for the class files.  If it refers to a directory, this method will recursively look
   *  for the classes in the directory.
   * @param url 
   * @return 
   */
    public static List getPersistentClassNamesFromURL(URL url, ClassLoader loader){
        List classNames = null;
        try {
            classNames = getPersistentClassNamesFromJar(url, loader);
            if (classNames != null){
                return classNames;
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, JAR_OPENING_EXCEPTION, url, exception);
        };
        try{
            // Only URL.toURI() handles spaces and special characters correctly
            File file = new File(convertURLToURI(url));            
            if (file.isDirectory()){
                classNames = getPersistentClassNamesFromDirectory(file, loader);
                if (classNames != null){
                    return classNames;
                }
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.FINEST, DIRECTORY_OPENING_EXCEPTION, url, exception);
        };
        throw PersistenceUnitLoadingException.couldNotGetClassNamesFromUrl(url);
    }

  /**
   *  Create a list of the classes that from the jar with the given name.  That return
   *  true from the isPersistent() call.
   * @param file the File representation of this jar
   * @param loader the ClassLoader to use
   * @return 
   */
    public static List<String> getPersistentClassNamesFromJar(URL url, ClassLoader loader){
        List<String> classList = getClassNamesFromJar(url);
        if (classList == null){
            return null;
        }
        List<String> persistentClasses = new Vector();
        for (String className: classList){
            if (isClassPersistent(className, loader, false)){
                persistentClasses.add(className);
            }
        }
        return persistentClasses;
    }

  /**
   *  Return whether the class with the given name is persistent.
   *  A class is persistent if it is annotated with @Entity or @Embeddable.
   * @param className 
   * @return 
   */
    public static boolean isClassPersistent(String className, ClassLoader loader, boolean throwExceptionIfNotFound){
        Class candidateClass = null;
        try{
            candidateClass = loader.loadClass(className);
        } catch (ClassNotFoundException exc){
            if (throwExceptionIfNotFound){
                throw PersistenceUnitLoadingException.exceptionLoadingClassWhileLookingForAnnotations(className, exc);
            } else {
                AbstractSessionLog.getLog().log(AbstractSessionLog.WARNING, "persistence_unit_processor_error_loading_class", exc.getClass().getName(), exc.getLocalizedMessage() , className);
                return false;                
            }
        } catch (Exception exception){
            AbstractSessionLog.getLog().log(AbstractSessionLog.WARNING, "persistence_unit_processor_error_loading_class", exception.getClass().getName(), exception.getLocalizedMessage() , className);
            return false;
        }
        return isClassPersistent(candidateClass);
    }
    /**
    *  Return whether a given class is persistent
    *  A class is persistent if it is annotated with @Entity.
    *  We don't consider Embeddable and MappedSuperclass classes as entities,
    *  because they are lazily processed when referenced from some entity class.
    * @param candidateClass 
    * @return 
    */
    public static boolean isClassPersistent(Class candidateClass){
        if (candidateClass.isAnnotationPresent(javax.persistence.Entity.class)) {
            return true;
        }
        return false;
    }

    /**
    * Converts URL to a URI to handle spaces and special characters correctly.
    * Validates the URL to represent a valid file name.
    * @param url the URL to be converted
    * @return the corresponding URI
    */
    private static URI convertURLToURI(URL url) {
        String filePath = url.getFile();      
        if (filePath.equals("") || filePath == null) {
                throw PersistenceUnitLoadingException.filePathMissingException(filePath);
        }

        URI uri = null;
        try {
            // attempt to use url.toURI since it will deal with all urls without special characters
            // and URISyntaxException allows us to catch issues with special characters
            // this will handle URLs what already have special characters replaced such as URLS
            // derived from searches for persistence.xml on the Java System class loader
            uri = url.toURI();
        } catch (URISyntaxException e) {
            try{
                // use multi-argument constructor for URI since single-argument constructor and URL.toURI() do not deal with special characters in path
                uri = new URI(url.getProtocol(), url.getUserInfo(), url.getHost(), url.getPort(), url.getPath(), url.getQuery(), null);
            } catch (URISyntaxException exc){};
            if (uri == null){
                throw PersistenceUnitLoadingException.exceptionProcessingPersistenceUnit(url, e);
            }
        }

        return uri;
    }

}
