/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
package oracle.toplink.essentials.internal.ejb.cmp3.metadata;

import java.io.InputStream;
import java.io.IOException;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.persistence.spi.PersistenceUnitInfo;

import oracle.toplink.essentials.ejb.cmp3.persistence.PersistenceUnitProcessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.ClassAccessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataClass;

import oracle.toplink.essentials.internal.ejb.cmp3.xml.accessors.XMLClassAccessor;

import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLHelper;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLLogger;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLValidator;

import oracle.toplink.essentials.internal.sessions.AbstractSession;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The object/relational metadata processor for the EJB3.0 specification. 
 * 
 * @author Guy Pelletier
 * @since TopLink EJB 3.0 Reference Implementation
 */
public class MetadataProcessor {
    protected ClassLoader m_loader;
    protected MetadataLogger m_logger;
    protected MetadataProject m_project;
    protected MetadataValidator m_validator;
    
    /**
     * INTERNAL:
     * Called from EntityManagerSetupImpl. The 'real' EJB 3.0 processing
     * that includes XML and annotations.
     */
    public MetadataProcessor(AbstractSession session, ClassLoader loader, boolean enableLazyForOneToOne) {
        m_loader = loader;
        m_project = new MetadataProject(session, enableLazyForOneToOne);
    }
    
    /**
     * INTERNAL:
     * Called from RelationshipWeaverTestSuite. Use this constructor to avoid
     * XML processing.
     */
    public MetadataProcessor(AbstractSession session, ClassLoader loader, Collection<Class> entities, boolean enableLazyForOneToOne) {
        m_loader = loader;
        m_project = new MetadataProject(session, enableLazyForOneToOne);
        
        for (Class entity : entities) {
            m_project.addDescriptor(new MetadataDescriptor(entity));
        }
    }
    
    /**
     * INTERNAL: 
     * Method to place EntityListener's on the descriptors from the given 
     * session. This call is made from the EntityManagerSetup deploy call.
     */
    public void addEntityListeners() {
        for (MetadataDescriptor descriptor: m_project.getDescriptors()) {
            // Process all descriptors that are in our project.
            ClassAccessor accessor = descriptor.getClassAccessor();
            
            // The class loader has changed, update the class stored for
            // our class accessor and its list of mapped superclasses.
            accessor.setAnnotatedElement(descriptor.getJavaClass());
            accessor.clearMappedSuperclasses();            
            
            accessor.processListeners(m_loader);
        }
    }
    
    /**
     * INTERNAL:
     * Method to place NamedQueries and NamedNativeQueries on the given session. 
     * This call is made from the EntityManagerSetup deploy call.
     */
    public void addNamedQueries() {
        m_project.processNamedQueries(m_validator);
        m_project.processNamedNativeQueries(m_loader);
    }
    
    /**
     * INTERNAL:
     * Return a set of class names for each entity, embeddable found in the xml 
     * descriptor instance document.
     */
    public static Set<String> buildClassSet(InputStream xmlDocumentInputStream, String fileName, ClassLoader loader) {
        XMLHelper helper = new XMLHelper(xmlDocumentInputStream, fileName, loader);
        
        // Process the package node.
        String defaultPkg = helper.getNodeValue(new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.PACKAGE, XMLConstants.TEXT});

        // Handle entities only. Mapped superclasses and embeddables are
        // discovered and processed separately.
        HashSet<String> classSet = new HashSet<String>();
        classSet.addAll(buildClassSetForNodeList(helper, XMLConstants.ENTITY, defaultPkg));

        return classSet;
    }
    
    /**
     * INTERNAL:
     * The class name of each node in the node list will be added to the 
     * provided collection.
     */
    protected static Set<String> buildClassSetForNodeList(XMLHelper helper, String xPath, String defaultPkg) {
    	HashSet<String> classNames = new HashSet<String>();
        NodeList nodes = helper.getNodes(XMLConstants.ENTITY_MAPPINGS, xPath);
    	int nodeCount = nodes.getLength();
        
        for (int i = 0; i < nodeCount; i++) {
            // Process the required class attribute node.
            classNames.add(XMLHelper.getFullyQualifiedClassName(helper.getNode(nodes.item(i), XMLConstants.ATT_CLASS).getNodeValue(), defaultPkg));
        }
        
        return classNames;
    }
    
    /** 
     * INTERNAL:
	 * Return the logger used by the processor.
	 */
	public MetadataLogger getLogger() {
        return m_logger;
    }
	
    /**
     * INTERNAL:
     */
     public MetadataProject getProject() {
         return m_project;
     }
    
    /** 
     * INTERNAL:
	 * Return the validator used by the processor.
	 */
	public MetadataValidator getValidator() {
        return m_validator;
    }
	
    /**
     * INTERNAL:
     * Called from RelationshipWeaverTestSuite which uses only annotations
     * and no XML.
     */
    public AbstractSession processAnnotations() {
        // Set the correct contextual validator and logger.
        m_validator = new MetadataValidator();
        m_logger = new MetadataLogger(m_project.getSession());

        // take a copy of the collection to avoid concurrent modification exception
        // that would result when embeddables are added lazily.
        for (MetadataDescriptor descriptor:
                m_project.getDescriptors().toArray(new MetadataDescriptor[]{})) {
            // Process all descriptors that are in our project.
            ClassAccessor accessor = descriptor.getClassAccessor();
                
            // If there is no accessor on this descriptor then it has not been
            // processed yet. Create one and process it.
            if (accessor == null) {
                accessor = new ClassAccessor(new MetadataClass(descriptor.getJavaClass()), this, descriptor);
                descriptor.setClassAccessor(accessor);
                accessor.process();
            }
        } 
        
        // Process the project and anything that was deferred like
        // sequencing and relationship mappings. Return the session and we
        // are done.
        return m_project.process();
    }
    
    /**
     * INTERNAL:
     * Process persistence unit metadata and defaults, and apply them to each 
     * entity in the collection. Any conflicts in elements defined in multiple 
     * documents will cause an exception to be thrown.  The first instance 
     * encountered wins, i.e. any conflicts between PU metadata definitions in 
     * multiple instance documents will cause an exception to be thrown.  The 
     * one exception to this rule is default listeners: all default listeners 
     * found will be added to a list in the order that they are read from the 
     * instance document(s). 
     */
     public void processPersistenceUnitMetadata(AbstractSession session, PersistenceUnitInfo persistenceUnitInfo, Collection<String> mappingFileNames, Collection<Class> entities) {
    	if (! mappingFileNames.isEmpty()) {
            // For each orm xml instance document, process persistence unit 
            // metadata/defaults and mapped superclasses.
            Iterator<String> fileNames = mappingFileNames.iterator();

            Iterator<String> mappingFileNamesIt = mappingFileNames.iterator();
            while (mappingFileNamesIt.hasNext()){
                String fileName = mappingFileNamesIt.next();
                InputStream inputStream = null;
                m_logger = new XMLLogger(session);
                try {
                    inputStream = PersistenceUnitProcessor.createInputStreamForFileInPersistenceUnit(fileName, persistenceUnitInfo, m_loader);         
                    
                    if (inputStream == null){
                        getLogger().logWarningMessage(XMLLogger.COULD_NOT_FIND_ORM_XML_FILE, fileName);
                        continue;
                    }
                    // Initialize a helper for navigating the instance document.
                    XMLHelper helper = new XMLHelper(inputStream, fileNames.next(), m_loader);
                
                    // Store all mapped-superclasses.
                    NodeList nodes = helper.getNodes(XMLConstants.ENTITY_MAPPINGS, XMLConstants.MAPPED_SUPERCLASS);
        
                    for (int i = 0; i < nodes.getLength(); i++) {
                        Node node = nodes.item(i);
                        Class cls = helper.getNodeValue(nodes.item(i), XMLConstants.ATT_CLASS, void.class);
                        m_project.addMappedSuperclass(cls, node, helper);
                    }        
                    
                    // Store all embeddable classes.
                    nodes = helper.getNodes(XMLConstants.ENTITY_MAPPINGS, XMLConstants.EMBEDDABLE);

                    for (int i = 0; i < nodes.getLength(); i++) {
                        Node node = nodes.item(i);
                        Class cls = helper.getNodeValue(nodes.item(i), XMLConstants.ATT_CLASS, void.class);
                        m_project.addEmbeddable(cls, node, helper);
                    }

                    // Look for a persistence-unit-metadata node.
                    Node persistenceUnitMetadataNode = helper.getNode(new String[] {XMLConstants.ENTITY_MAPPINGS, XMLConstants.PU_METADATA});
                
                    if (persistenceUnitMetadataNode != null) {
                        MetadataPersistenceUnit persistenceUnit = new MetadataPersistenceUnit();
                        
                        // Process the xml-mapping-metadata-complete tag.                        
                        persistenceUnit.setIsMetadataComplete(helper.getNode(persistenceUnitMetadataNode, XMLConstants.METADATA_COMPLETE) != null);

                        // process persistence unit defaults
                        Node persistenceUnitDefaultsNode = helper.getNode(persistenceUnitMetadataNode, XMLConstants.PU_DEFAULTS);
                        
                        if (persistenceUnitDefaultsNode != null) {
                            // Process the persistence unit access.
                            persistenceUnit.setAccess(helper.getNodeTextValue(persistenceUnitDefaultsNode, XMLConstants.ACCESS));
                    
                            // Process the persitence unit schema.
                            persistenceUnit.setSchema(helper.getNodeTextValue(persistenceUnitDefaultsNode, XMLConstants.SCHEMA));
                    
                            // Process the persistence unit catalog.
                            persistenceUnit.setCatalog(helper.getNodeTextValue(persistenceUnitDefaultsNode, XMLConstants.CATALOG));
                    
                            // Process the persistence unit cascade-persist.
                            persistenceUnit.setIsCascadePersist(helper.getNode(persistenceUnitDefaultsNode, XMLConstants.CASCADE_PERSIST) != null);
                    
                            // Process the default entity-listeners. No conflict 
                            // checking will be done, that is, any and all 
                            // default listeners will be added to the project.
                            NodeList listenerNodes = helper.getNodes(persistenceUnitDefaultsNode, XMLConstants.ENTITY_LISTENERS, XMLConstants.ENTITY_LISTENER);
                            if (listenerNodes != null) {
                                m_project.addDefaultListeners(listenerNodes, helper);
                            }
                        }
                        
                        // Add the metadata persistence unit to the project if 
                        // there is no conflicting metadata (from other 
                        // persistence unit metadata)
                        MetadataPersistenceUnit existingPersistenceUnit = m_project.getPersistenceUnit();
                        if (existingPersistenceUnit != null) {
                            if (! existingPersistenceUnit.equals(persistenceUnit)) {
                                (new XMLValidator()).throwPersistenceUnitMetadataConflict(existingPersistenceUnit.getConflict());
                            }
                        } else {
                            m_project.setPersistenceUnit(persistenceUnit);
                        }
                    }
                } catch (IOException exception){
                    getLogger().logWarningMessage(XMLLogger.ERROR_LOADING_ORM_XML_FILE, fileName, exception);
                } catch (RuntimeException re) {
                    throw re;
                } finally {
                    if (inputStream != null){
                        try{
                            inputStream.close();
                        } catch (IOException exc){};
                    }
                }

            }
        }
    	
        // Add a metadata descriptor to the project for every entity class.
        // Any persistence unit metadata/defaults will be applied 
		if (! entities.isEmpty()) {
            for (Class entity : entities) {
                m_project.addDescriptor(new MetadataDescriptor(entity));
	    	}
		}
    }
    
    /**
     * INTERNAL:
     * Process the xml and fill in the project. Process the entity-mappings
     * information then process the entities.
     */
    public void processXML(InputStream xmlDocumentStream, String fileName) {
        if (m_project.hasDescriptors()) {
             // Initialize the correct contextual objects.
            m_validator = new XMLValidator();
            m_logger = new XMLLogger(m_project.getSession());
            XMLHelper helper = new XMLHelper(xmlDocumentStream, fileName, m_loader);
        
            // Process the entity mappings ... this is a crude way of doing
            // things ... but hey ... the clock is ticking ...
            MetadataDescriptor desc = m_project.getDescriptors().iterator().next();
            XMLClassAccessor dummyAccessor = new XMLClassAccessor(new MetadataClass(desc.getJavaClass()), null, helper, this, desc);
            dummyAccessor.processEntityMappings();
        
            // Process the entity nodes for this xml document.
            NodeList entityNodes = helper.getNodes(XMLConstants.ENTITY_MAPPINGS, XMLConstants.ENTITY);

            if (entityNodes != null) {
                for (int i = 0; i < entityNodes.getLength(); i++) {
                    Node entityNode = entityNodes.item(i);
                    Class entityClass = helper.getClassForNode(entityNode);
                    MetadataDescriptor descriptor = m_project.getDescriptor(entityClass);

                    // Process all descriptors that are in our project.
                    ClassAccessor accessor = descriptor.getClassAccessor();
                
                    // If there is no accessor on this descriptor then it has not 
                    // been processed yet. Create one and process it.
                    if (accessor == null) {
                        accessor = new XMLClassAccessor(new MetadataClass(descriptor.getJavaClass()), entityNode, helper, this, descriptor);
                        descriptor.setClassAccessor(accessor);
                        accessor.process();
                    }
                }
            }
        } else {
            // WIP - log a warning that we have no entities to process ...
        }
    }
    
    /** 
     * INTERNAL:
	 * Use this method to set the correct class loader that should be used
     * during processing.
	 */
	public void setClassLoader(ClassLoader loader) {
        m_loader = loader;
    }
}
